<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use Illuminate\Http\Request;
use App\Enums\ItemCodeStatus;
use App\Enums\CorporationList;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Hrjp\Models\ItemCode;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Corporation;
use Suiterus\Hrjp\Models\Position as P;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Position_has_salary as PHS;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\Promotion\PromotionApplication;
use Suiterus\Hrjp\Models\Application;

class PositionController extends Controller
{

    use HasCustomLogs;

    public function create_position(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'title' => 'required|string',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();
        try {
            $position = P::create([
                'title' => $req->title,
                'created_by' => $req->user()->id,
            ]);

            $position['created_at'] = Carbon::parse($req->selected_value_from_datepickers);
            $position->save();
            DB::commit();
            return response()->json([
                'text' => 'Position created successfully!.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your Position as of now. Contact the developer to fix it. Error Code : '],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //CREATE FUNCTION

    public function create_position_job_posting(Request $req)
    {
        DB::beginTransaction();
        $date = Carbon::now();
        $formattedDate = $date->format('Y/m/d');
        $valid = Validator::make($req->all(), [
            'item_code_id' => 'required|array',
            'item_code_id.*.id' => 'required|numeric|exists:hrjp_db.item_codes,id',
            'position_id' => 'required|integer|exists:hrjp_db.positions,id',
            'hiring_manager' => 'required|integer',
            'salary' => 'required',
            'slots' => 'required',
            'post_status' => 'required',
            'post_due' => 'required',
            'employee_type' => 'required',

        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        try {
            foreach ($req->item_code_id as $itemCode) {
                $position = PHS::create([
                    'item_code_id'  => $itemCode['id'],
                    'position_id' => $req->position_id,
                    'hiring_manager' => $req->hiring_manager,
                    'description' => $req->description,
                    'education' => $req->education,
                    'training' => $req->training,
                    'experience' => $req->experience,
                    'eligibility' => $req->eligibility,
                    'place_of_assignment' => $req->place_of_assignment,
                    'salary' => $req->salary,
                    'post_start' => $req->post_status == 1 ? $formattedDate : null,
                    'slots' => $req->slots,
                    'post_status' => $req->post_status,
                    'post_due' => $req->post_due,
                    'employee_type' => $req->employee_type,
                    'show_salary' => $req->show_salary,
                    'created_by' => $req->user()->id,
                ]);

                $this->logCustomMessage('create_job_post', $position, Auth::user()->name . ' created job posting for position : ' . $position->itemCodes()->first()->item_code . '-' . $position->itemCodes()->first()->position->title, $position, 'job posting created', new Activity());
            }

                $position->save();
                DB::commit();
                return response()->json([
                    'text' => 'job created successfully!.',
                ]);
            // }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your Position as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }catch(ME $e){
            return response()->json([
                'errors'    => ['There is a job posted already in this position.']
            ], 500);
        }
    }

    // UPDATE FUNCTION

    public function update_position(Request $req)
    {
        $date = Carbon::now();
        $formattedDate = $date->format('Y/m/d');

        $valid = Validator::make($req->all(), [
            'hiring_manager' => 'required|integer',
            'salary' => 'required',
            'post_status' => 'required',
            'post_due' => 'required',
            'employee_type' => 'required',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        try {
            DB::beginTransaction();
            $positions = PHS::findOrFail($req->id);
            $oldPosition = clone $positions;

            $positions->update([
                'hiring_manager' => $req->hiring_manager,
                'description' => $req->description,
                'education' => $req->education,
                'training' => $req->training,
                'experience' => $req->experience,
                'eligibility' => $req->eligibility,
                'place_of_assignment' => $req->place_of_assignment,
                'salary' => $req->salary,
                'post_start' => $req->post_status == 1 ? $formattedDate : null,
                'post_status' => $req->post_status,
                'post_due' => $req->post_due,
                'employee_type' => $req->employee_type,
                'show_salary' => $req->show_salary,
                'updated_by' => $req->user()->id,
            ]);

            $positions->save();

            $positions->attributes = collect($positions);
            $positions->old = collect($oldPosition);

            $this->logCustomMessage('update_job_post', $positions, Auth::user()->name . ' updated job posting for position : ' . $positions->itemCodes()->first()->item_code . '-' . $positions->itemCodes()->first()->position->title, $positions, 'job posting updated', new Activity());


            DB::commit();
            return response()->json([
                'message' => $positions->position->title . " has been updated successfuly",
            ]);

        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t update your job post as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }
     //slot update
     public function update_slot(Request $req){
        try {
            DB::beginTransaction();
            $today = Carbon::now();
            $positions = PHS::where('slots', '<=', 0)->where('post_status', 1)
            ->orWhere(function ($query) use($today){
                $query->whereDate('post_due', '<', $today)->where('post_status', 1);
            })
            ->update([
                'post_status' => 2,
                'updated_by' => $req->user()->id,
            ]);
            DB::commit();
            return response()->json([
                'message' => "Post status has been updated successfuly",
            ]);

        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t update your job post as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //FETCH JOB BY ID
    public function fetch_job_by_id(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|integer|exists:hrjp_db.position_has_salaries,id',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        try {
            $data = PHS::where('id', $req->id)->get();
            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage(),
            ]);
        }
    }

    //FETCH ALL FUNCTION
    public function fetch_all_position(Request $req)
    {
        $valid = Validator::make($req->all(), ['position_id' => 'required|numeric|exists:hrjp_db.positions', 'page_count' => 'nullable']);
        if (P::count() == 0) {
            return response()->json([
                'message' => "No record found.",
            ]);
        }

        $paginate = $req->page_count ? intval($req->page_count) : 10;
        try {
            $data = PHS::paginate($paginate);
            return response()->json(['message' => 'Fetch successful.', 'data' => $data], 200);
        } catch (Exception $e) {
            return response()->json(['errors' => ['Can`t fetch your request as of now. Contact the developer to fix it. Error Code : SCHEDULE-0x01'], 'msg' => $e->getMessage()], 500);
        }
    }

    public function init_position_has_salary(Request $request)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');
        $employer_name = Corporation::where('employer_id',CorporationList::NKTI)->pluck('name')->first();

        $data = PHS::when(isset($request->hiring_manager_id) && $request->hiring_manager_id != null, function ($query) use ($request) {
            $query->whereHas('hiringManager', function ($query) use ($request) {
                $query->where('id', $request->hiring_manager_id);
            });
        })->when(isset($request->status) && $request->status != null, function ($query) use ($request) {
            $query->where('post_status', $request->status);
        })->when(isset($request->salary_id) && $request->salary_id != null, function ($query) use ($request) {
            $query->whereHas('getSalaryAttribute', function ($query) use ($request) {
                $query->where('id', $request->salary_id);
            });
        })->when(count($request->item_code_id) > 0, function ($query) use ($request) {
            $query->whereIn('item_code_id', $request->item_code_id);
        })->when(count($request->position_id) > 0, function ($query) use ($request) {
            $query->whereIn('position_id', $request->position_id);
        })->orderBy('created_at', 'desc')->paginate($paginate);

        $data->getCollection()->transform(function ($item) {
            $applicationCount = Application::where("phs_id", $item->id)->count();
            $promotionApplicationCount = PromotionApplication::where('phs_id', $item->id)->count();
            $item->candidates = $applicationCount + $promotionApplicationCount;
            return $item;
        });

        return response()->json([
            'employer_name' => $employer_name,
            'data' => $data
        ]);
    }

    public function position_has_salary_no_paginate(Request $req)
    {
        return PHS::select(
            'salary',
            'item_code_id',
            'position_id')->orderBy('id', 'desc')->when(isset($req->status), function ($query) use ($req){
            $query->where('post_status', $req->status);
        })->get();

    }



    //SEARCH FUNCTION

    public function search_position(Request $req)
    {
        $search = $req->input('title');
        $valid = Validator::make($req->all(), ['title' => 'required|string|exists:hrjp_db.positions', 'page_count' => 'nullable']);

        if (P::count() == 0) {
            return response()->json([
                'msg' => "No record found.",
            ]);
        }

        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');

        try {
            $search = P::query()
                ->where('title', 'LIKE', "%{$search}%")
                ->paginate($paginate);

            return response()->json([
                'data ' => $search,
            ]);

            if ($search->isEmpty()) {
                DB::rollback();

                return response()->json(
                    [
                        'message' => 'Title does not exists.',
                    ],
                    400
                );
            } else {
                return response()->json([
                    'data' => $search,
                ]);
            }
        } catch (Exception $e) {
            DB::rollback();

            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : BLOG-0x03'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    // DELETE FUNCTION
    public function delete_position(Request $req)
    {
        DB::beginTransaction();
        try {
            try {
                $pos = P::findOrFail($req->id);
                $pos->delete();
                DB::commit();

                return response()->json([
                    'text' => 'Position has been deleted.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Position not found!.'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : '],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function init_list_deleted_positions(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $deleted_positions = P::onlyTrashed()
            ->orderBy('deleted_at', 'desc')
            ->paginate($paginate);
        return response()->json([
            'text' => 'fetch successful.',
            'data' => $deleted_positions
        ]);
    }

    public function restore_deleted_position(Request $req)
    {
        try {
            try {
                $pos = P::onlyTrashed()
                    ->findOrFail($req->id)
                    ->restore();
                DB::commit();
                return response()->json([
                    'text' => 'Position has been restored.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Position not found!.'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : '],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function delete_deleted_position(Request $req)
    {
        DB::beginTransaction();
        try {
            try {
                $pos = P::onlyTrashed()
                    ->findOrFail($req->id)
                    ->forceDelete();
                DB::commit();
                return response()->json([
                    'text' => 'Position has been deleted.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Position not found!.'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : '],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //For hiring_manager
    public function fetch_users(Request $req) {
        $data = User::where('status', Status::ACTIVE)->orderBy('id', 'desc')->whereHas('employeeMetaInfo')
        ->with(['employeeMetaInfo' => function($q){

            $q->select('id','user_id','employee_id')->without('employeeType','corporation','division','department','branch','position');

        }])->without(
                    'currentRole',
                    'roles',
                    'permissions',
                    'storage',
                    'supervisor',
                    'user_supervisor',
                    'exitInterview',
                    'userProfilePicture',
                    'profileBasicInfo'
        )->select('id','name')->get();

        return response()->json([
            'data' => $data
        ]);
    }

    public function decrement_slot(Request $req){
        PHS::where('position_id', $req->phs_id)->decrement('slots');
        return response()->json([
            'text' => "Slot updated."
        ]);
    }

    public function fetch_hiring_manager() {
        return response()->json([
            'data' => User::where('status', Status::ACTIVE)->orderBy('id', 'desc')->whereHas('jobPost')->whereHas('employeeMetaInfo')
            ->with(['employeeMetaInfo' => function($query){
                $query->select('id','user_id','employee_id')->without('employeeType','corporation','division','department','branch');
            }])->without('storage', 'roles', 'permissions', 'supervisor')->select('id','name')->get()
        ]);
    }

    public function fetch_vacant_positions() {
        return response()->json([
            'data' => P::whereHas('itemCodes', function($query) {
                $query->select('item_code')->where('status', ItemCodeStatus::AVAILABLE);
            })->with(['itemCodes' => function($query) {
                $query->where('status', ItemCodeStatus::AVAILABLE)->without('position');
            }])->select('id','title')->get()
        ]);
    }

    public function fetch_position_with_record()
    {
        try {
            $data = P::whereHas('positionHasSalary')->get();

            return response()->json([
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function fetch_item_code_with_record()
    {
        try {
            $data = ItemCode::whereHas('positionHasSalary')->get();

            return response()->json([
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function fetch_active_positions() {
        return response()->json([
            'data' => P::whereHas('itemCodes', function($query) {
                $query->select('item_code')->where('status', ItemCodeStatus::ACTIVE);
            })->with(['itemCodes' => function($query) {
                $query->where('status', ItemCodeStatus::ACTIVE)->doesntHave('employeeMetaInfo')->without('position');
            }])->select('id','title')->get()
        ]);
    }
}
