<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use mikehaertl\pdftk\Pdf;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Hrjp\Models\AppointmentForm;
use Suiterus\Adg\Models\Activity\Activity;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Hrjp\Request\DownloadAppointmentFormRequest;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use ZipArchive;

class DownloadUploadAppointmentForm extends Controller
{

    use HasCustomLogs;

    public function uploadAppointmentForm(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'applicant_id'  =>  'required',
            'upload_appointment_form.*' => 'required|max:50000|mimes:pdf',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'   => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $path = 'applicant/' . $req->applicant_id;

            $appointmentForm = AppointmentForm::updateOrCreate([
                'applicant_id' => $req->applicant_id,
                'created_by' => $req->user()->id,
            ]);

            foreach ($req->file('upload_appointment_form') as $file) {
                $fileName = $file->getClientOriginalname();
                $filePath = Storage::disk('appointment_form')->putFileAs($path, $file, $fileName);
                FileVault::disk('appointment_form')->encrypt($filePath);

                $appointmentForm->files()->updateOrCreate([
                    'appointment_form_id' => $appointmentForm->id,
                    'file_path' => $filePath,

                ], [
                    'created_by' => $req->user()->id
                ]);
            }

            $applicant = $appointmentForm->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $this->logCustomMessage('upload_appointment_form', $appointmentForm, Auth::user()->name . ' uploaded an appointment form for applicant ' . $fullName, $appointmentForm, 'Upload appointment form', new Activity());
            
            $jpNotifService = new JobPortalMailNotificationService();
            $message = 'We have uploaded the appointment form. Please click the link below and dowload your appointment form';
            $jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $jpNotifService->sendNotificationToAdmin('Upload Appointment Form JP', Auth::id(), Auth::id(), null, null, null, $fullName);


            DB::commit();
            return response()->json([
                'text' => 'Appointment form has been uploaded successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'msg'   =>  $e->getMessage(),
                'line'  =>  $e->getLine()
            ], 500);
        }
    }

    public function downloadAppointmentForm(DownloadAppointmentFormRequest $req)
    {
        try {

            $appointmentForm = AppointmentForm::where('id', $req->id)->with('files')->first();

            if (!Storage::disk('local')->exists('job-form-zip')) {
                Storage::disk('local')->makeDirectory('job-form-zip');
            }

            $uuid = uniqid();
            $filePathsToRemove = [];
            $zipFilePath = storage_path('app/job-form-zip/' . $uuid . '.zip');
            $zip = new ZipArchive();
            $zip->open($zipFilePath, ZipArchive::CREATE | ZipArchive::OVERWRITE);

            foreach ($appointmentForm->files as $file) {
                $filePath = $file->file_path . '.enc';
                FileVault::disk('appointment_form')->decryptCopy($filePath);
                $decryptedFilePath = Storage::disk('appointment_form')->path($file->file_path);
                $filePathsToRemove[] = $decryptedFilePath;

                $pdf = new Pdf($decryptedFilePath, config('pdftk.options'));
                $result = $pdf->flatten()->saveAs($decryptedFilePath);

                if (!$result) {
                    throw new FileException('There was a problem processing your appointment form');
                }

                $file = basename($decryptedFilePath);
                $zip->addFile($decryptedFilePath, $file);
            }

            $zip->close();

            foreach ($filePathsToRemove as $filePath) {
                unlink($filePath);
            }

            return response()->download(Storage::disk('local')->path('job-form-zip/' . $uuid . '.zip'))->deleteFileAfterSend(true);
        } catch (Exception $e) {
            Storage::disk('appointment_form')->delete($req->path);
            return response()->json([
                'errors'    => ['File not found'],
                'message'   => $e->getMessage(),
            ], 404);
        }
    }

    public function fetchAppointmentForm()
    {
        $appointment_form = AppointmentForm::where('applicant_id', Auth::id())->latest()->first();

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $appointment_form
        ]);
    }
}
