<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Illuminate\Http\Request;
use App\Enums\Jobportal\Step;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Suiterus\Hrjp\Models\Application;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use App\Enums\Jobportal\ApplicationStatus;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Hrjp\Models\Applicant_exam_schedule;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;

class ApplicationController extends Controller
{

    use HasCustomLogs;

    public function get_step_1(Request $request)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        $data = AC::whereHas('extraField')->whereHas('application', function ($q) {
            $q->where('status', ApplicationStatus::ACTIVE)->where('step', Step::STEP_ONE);
        })->with(['application' => function($q){
            $q->select('id','applicant_id', 'position_id','phs_id','step');
        },'application.position' => function($q){
            $q->select('id', 'title');
        }
        , 'personalInformation', 'pdsChoices', 'otherInformation', 'permanentAddress', 'residentialAddress','spouse', 'children', 'father', 'maidenName', 'elementary', 'secondary', 'vocational', 'college', 'graduate', 'civilService', 'workExperience', 'voluntaryWork', 'learningDevelopment', 'skillHobby', 'choice', 'reference', 'governmentIssuedId', 'fileAttachment'])
        ->select('id', 'fname', 'mname', 'lname')->paginate($paginate);

        return response()->json([
            'data' => $data
        ]);
    }

    public function fetch_applicants()
    {
        return AC::all();
    }

    public function fetch_list_all_applicants(Request $request) {
        return AC::whereHas('application', function ($query) use ($request) {
                $query->where('status', $request->status);
            })
            ->with(['preEmployment' => function ($query) {
                $query->where('table_name', 'contact_information')
                    ->with('preEmploymentColumn', function ($query) {
                        $query->where('field_name', 'date_of_application');
                    });
            }, 'application' => function ($query) {
                $query->with('phs');
            }])
            ->get();
        
    }

    public function list_of_all_applicants(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        $records = AC::whereHas('application', function ($query) use ($request){
            $query->where('status', $request->status);
        })->with(['preEmployment' => function ($query) {
            $query->where('table_name', 'contact_information')
                ->with('preEmploymentColumn', function ($query) {
                    $query->where('field_name', 'date_of_application');
                });
        }, 'application' => function ($query) {
            $query->with('phs', 'applicantRequirements');
        }])
        ->when($request->step != null, function($query) use ($request) {
            $query->whereHas('application', function($query) use ($request) {
                $query->where('step', $request->step);
            });
        })->when($request->item_code_id != null, function($query) use ($request) {
            $query->whereHas('application.phs', function($query) use ($request) {
                $query->where('item_code_id', $request->item_code_id);
            });
        })->when($request->position_id != null, function($query) use ($request) {
            $query->whereHas('application', function($query) use ($request) {
                $query->where('position_id', $request->position_id);
            });
        });

        if (is_array($request->applicant_id) && count($request->applicant_id) > 0) {
            $records->whereIn('id', $request->applicant_id);
        }

        return response()->json([
            'data'  => $records->paginate($paginate)
        ]);
    }

    public function file_attachment_download(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'path' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        return Storage::download($req->path);
    }

    //Update current step
    public function approve(Request $req)
    {
        $application = Application::where('applicant_id', $req->applicant_id)->where('status', ApplicationStatus::ACTIVE)->first();

        $oldApplication = clone $application;

        $application->increment('step');

        $application->attributes = collect($application);
        $application->old = collect($oldApplication);

        $account = $application->ApplicantAccount()->first();
        $fullName = $account->fname . ' ' . $account->mname . ' ' . $account->lname;

        $this->logCustomMessage('approve_applicant_information', $application, Auth::user()->name . ' has approved applicant information for applicant: ' . $fullName, $application, 'applicant information approved', new Activity());
        $jpNotifService = new JobPortalMailNotificationService();
        $message = 'Your personal data sheet has been approved for the job title of '. $application->position->title .'. Please click the link below to see the application process';
        $jpNotifService->sendMailNotification($account->email, $message, $fullName);
        $jpNotifService->sendNotificationToAdmin('Approve PDS JP', Auth::id(), Auth::id(), null, null, null, $fullName);

        return response()->json([
            'text' => 'step increment successful.'
        ]);
    }

    //STEP 6
    public function get_attachments(Request $request)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        $data = AC::whereHas('application', function ($q) {
            $q->where('step', Step::STEP_EIGHT)->whereIn('status', [ApplicationStatus::ACTIVE, ApplicationStatus::INACTIVE]);
        })
            ->with(['application' => function ($q) {
                $q->select('id', 'applicant_id', 'position_id', 'step', 'status', 'phs_id');
            }, 'application.position' => function ($q) {
                $q->select('id', 'title');
            }, 'personalInformation', 'pdsChoices', 'otherInformation', 'permanentAddress', 'residentialAddress', 'spouse', 'children', 'father', 'maidenName', 'elementary', 'secondary', 'vocational', 'college', 'graduate', 'civilService', 'workExperience', 'voluntaryWork', 'learningDevelopment', 'skillHobby', 'choice', 'reference', 'governmentIssuedId', 'fileAttachment'])
            ->select('id', 'fname', 'mname', 'lname')->paginate($paginate);

        return response()->json([
            'data' => $data
        ]);
    }
}
