<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Illuminate\Http\Request;
use App\Enums\Jobportal\Step;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Enums\Jobportal\ApplicationStatus;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Hrjp\Models\ApplicantRequirementsSchedule as ARS;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;

class ApplicantRequirementsScheduleController extends Controller
{
    use HasCustomLogs;
    
    protected $jpNotifService;

    public function __construct() {
        $this->jpNotifService = new JobPortalMailNotificationService();
    }

    // Create Requirement sched
    public function create_requirement_schedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'applicant_id' => 'required|integer|exists:hrjp_db.applicant_accounts,id',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();

        try {
            $ars = ARS::create([
                "applicant_id"      => $req->applicant_id,
                "personnel"         => $req->personnel,
                "office"            => $req->office,
                "schedule"          => $req->schedule,
                "status"            => 1,
                'created_by'        => Auth::id(),
                "created_at"        => now(),
            ]);

            $applicant = $ars->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $this->logCustomMessage('requirement_submission_schedule_created', $ars, Auth::user()->name . ' has created a requirement submission schedule for applicant ' . $fullName, $ars, 'Create a requirement submission schedule', new Activity());

            $message = 'We are informing you that we have successfully created a schedule for the submission of requirements. Please click the link below to acknowledge the schedule and continue the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Set Schedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Submission of Requirements');

            DB::commit();

            return response()->json([
                'text' => 'Submission of requirements schedule created successfully!.',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your request as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    // Fetching Requirement Sched

    public function fetch_requirements_schedule(Request $req)
    {

        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $data = AC::whereHas('application', function ($q) {
                $q->whereIn('step', [Step::STEP_NINE])->whereIn('status', [ApplicationStatus::ACTIVE, ApplicationStatus::INACTIVE]);
            })->with(['requirementsSchedule' => function ($q) {
                $q->whereNotIn('status', [4]);
            }, 'application' => function ($q) {
                $q->with(['position' => function ($q) {
                    $q->select('id', 'title');
                }])->select('id', 'applicant_id', 'position_id');
            }, 'appointmentSchedule' => function ($q) {
                $q->select('id', 'applicant_id', 'schedule', 'personnel', 'office', 'status');
            },])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    //Set complete status
    public function complete_initial_requirements(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_requirements_schedules,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $ars = ARS::where('id', $req->schedule_id)->first();

            $applicant = $ars->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $oldArs = clone $ars;

            $ars->update([
                'status'       => 3,
                'updated_at'   => now(),
            ]);

            $ars->attributes = collect($ars);
            $ars->old = collect($oldArs);

            $this->logCustomMessage('tag_requirements_complete', $ars, Auth::user()->name . ' tagged the requirements as complete for applicant ' . $fullName, $ars, 'Tag requirements complete', new Activity());

            $message = 'We are informing you that we have completed reviewing your requirements. Please click the link below to view your application status.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Complete Requirements JP', Auth::id(), Auth::id(), null, null, null, $fullName);
    
            DB::commit();
            return response()->json([
                'text'  => "Record has been set to Completed."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in completing the submmision of requirements.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}
