<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Illuminate\Http\Request;
use App\Enums\ItemCodeStatus;
use App\Enums\Jobportal\Step;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Hrjp\Models\Application;
use Illuminate\Support\Facades\Validator;
use App\Enums\Jobportal\ApplicationStatus;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\ApplicantRequirements;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Hrjp\Models\ApplicantRequireAppointment as ARA;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;


class ApplicantRequireAppointmentController extends Controller
{
    use HasCustomLogs;
    // Create Requirement sched
    public function create_appointment_schedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'applicant_id' => 'required|integer|exists:hrjp_db.applicant_accounts,id',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();

        try {
            $ara = ARA::create([
                "applicant_id"      => $req->applicant_id,
                "personnel"         => $req->personnel,
                "office"            => $req->office,
                "schedule"          => $req->schedule,
                "status"            => 1,
                'created_by'        => Auth::id(),
                "created_at"        => now(),
            ]);

            $applicant = $ara->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $this->logCustomMessage('appointment_schedule_created', $ara, Auth::user()->name . ' has created an appointment schedule for applicant ' . $fullName, $ara, 'Create an appointment schedule', new Activity());

            $jpNotifService = new JobPortalMailNotificationService();
            $message = 'We are informing you that we have successfully created a schedule for your appointment. Please click the link below to view your application status';
            $jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $jpNotifService->sendNotificationToAdmin('Set Schedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Appointment');

            Application::where('applicant_id', $req->applicant_id)->where('status', ApplicationStatus::ACTIVE)
                ->increment('step');

            DB::commit();

            return response()->json([
                'text' => 'Submission of appointment schedule created successfully!.',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your request as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function fetch_appointment_schedule(Request $req)
    {

        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $data = AC::whereHas('application', function ($q) {
                $q->where('step', Step::STEP_TEN)->whereIn('status', [ApplicationStatus::ACTIVE, ApplicationStatus::INACTIVE]);
            })->with(['appointmentSchedule' => function ($q) {
                $q->select('id', 'applicant_id', 'schedule', 'personnel', 'office', 'status',);
            }, 'application' => function ($q) {
                $q->with(['position' => function ($q) {
                    $q->select('id', 'title')->has('positionHasSalary')->with(['itemCodes' => function ($q) {
                        $q->where('status', ItemCodeStatus::AVAILABLE);
                    }]);
                }])->select('id', 'applicant_id', 'position_id', 'phs_id')
                    ->with(['phs' => function ($q) {
                        $q->with(['employeeType' => function ($q) {
                            $q->select('id', 'title');
                        }])->select('id', 'employee_type')->without('position');
                    }]);
            }, 'requirementsSchedule' => function ($q) {
                $q->select('id', 'applicant_id', 'schedule', 'personnel', 'office', 'status');
            }, 'appointmentForm' => function ($q) {
                $q->select('id', 'applicant_id');
            }])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }
}
