<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Enums\Jobportal\Step;
use App\Enums\RescheduleStatus;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Enums\Jobportal\ExamStatus;
use App\Http\Controllers\Controller;
use Suiterus\Hrjp\Models\Application;
use Illuminate\Support\Facades\Validator;
use App\Enums\Jobportal\ApplicationStatus;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Hrjp\Models\Position_has_salary as PHS;
use Suiterus\Hrjp\Models\ApplicantPositionOption as APO;
use Suiterus\Hrjp\Models\ApplicantInitialInterview as AII;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\ApplicantInitialInterviewReschedule as AIIR;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;

class ApplicantInitialInterviewController extends Controller
{

    use HasCustomLogs;
    protected $jpNotifService;
    public function __construct() {
        $this->jpNotifService = new JobPortalMailNotificationService();
    }

    public function create_initial_interview_schedule(Request $req){
        $valid = Validator::make($req->all(), [
            'applicant_id' => 'required|integer|exists:hrjp_db.applicant_accounts,id',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();

        try{
            $interview = AII::create([
                    "applicant_id"   => $req->applicant_id,
                    "interviewer"    => $req->interviewer,
                    "zoomlink"       => $req->zoomlink,
                    "description"     => $req->description,
                    "schedule"       => $req->schedule,
                    "status"         => 1,
                    "created_by"     => $req->user()->id,
                    "created_at"     => now(),
            ]);

            $applicant = $interview->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $message = 'We are informing you that we have successfully created a schedule for the initial interview. Please click the link below to acknowledge the schedule and continue the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Set Schedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Initial Interview');

            $this->logCustomMessage('initial_interview_schedule_created', $interview, Auth::user()->name . ' has created an Initial Interview schedule for applicant ' . $fullName, $interview, 'Create an initial interview schedule', new Activity());

            DB::commit();

            return response()->json([
                'text' => 'Initial Interview schedule created successfully!.',
                'data' => $interview
            ]);
        }catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your request as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }


    //Process schedule
    public function process_schedule(Request $req){
        $valid = Validator::make($req->all(), [
            'applicant_id'  =>  'required|integer|exists:hrjp_db.applications,applicant_id',
        ]);

        // TO CHECK IF THE ARGUMENT FAILS
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        DB::beginTransaction();

        try{
            //check if a record exists in schedules
            if(isset($req->schedule_id) && !is_Null($req->schedule_id)){
                //Set status to accepted the reschedule table
                if($req->schedule_status == 4){
                    $interview = AIIR::where('initial_interview_schedule', $req->schedule_id)->first();

                    $oldInterview = clone $interview;

                    $interview->update([
                        'status'       => RescheduleStatus::HR_APPROVED,
                        'updated_at'   => now(),
                    ]);

                    $interview->attributes = collect($interview);
                    $interview->old = collect($oldInterview);

                    $applicant = $interview->applicant()->first();
                    $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

                    $this->logCustomMessage('initial_interview_reschedule_approved', $interview, Auth::user()->name . ' approved an initial interview reschedule for applicant ' . $fullName, $interview, 'Approved an initial interview reschedule', new Activity());

                    //create new schedule
                    $this->create_initial_interview_schedule($req);
                    //soft delete old schedule
                    $record = AII::findOrFail($req->schedule_id);
                    $record->delete();

                    $message ='We are informing you that your request for reschedule of your initial interview has been approved. Please click the link below and acknowledge the schedule to continue the process';
                    $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
                    $this->jpNotifService->sendNotificationToAdmin('Approve Reschedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Initial Interview');

                }else{
                    //For schedule update only of an existing record
                    $interview = AII::where('id', $req->schedule_id)->first();

                    $oldInterview = clone $interview;

                    $interview->update([
                        "schedule"      =>  $req->schedule,
                        "proctor"    => $req->proctor,
                        "zoomlink"       => $req->zoomlink,
                        "description"     => $req->description,
                        "updated_at"    => now()
                    ]);

                    $interview->attributes = $interview;
                    $interview->old = $oldInterview;

                    $applicant = $interview->applicant()->first();
                    $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

                    $this->logCustomMessage('initial_interview_schedule_update', $interview, Auth::user()->name . ' updated an initial interview reschedule for applicant ' . $fullName, $interview, 'Updated an initial interview schedule', new Activity());
                }
                DB::commit();
                return response()->json([
                    'message' => "Successfully rescheduled a record."
                ]);
            }else{
                $this->create_initial_interview_schedule($req);

                DB::commit();
                return response()->json([
                    'message' => "Successfully added a schedule."
                ]);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'error' => $e->getMessage(),
                'line' => $e->getLine()
            ],500);
        }
    }

    //Fetching of schedule
    public function fetch_initial_interview(Request $req){

        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $data = AC::whereHas('application', function($q){
                $q->where('step', Step::STEP_TWO)->whereIn('status', [ApplicationStatus::ACTIVE, ApplicationStatus::INACTIVE]);
            })->with(['initialInterview' => function($q){
                $q->whereNotIn('status', [ExamStatus::PASSED, ExamStatus::TAG_TO_ANOTHER_POSITION])->with(['InitialInterviewReschedule']);
            }, 'application' => function($q){
                $q->with(['position' => function($q){
                    $q->select('id','title');
                }])->select('id', 'applicant_id', 'position_id');
            }])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data
            ], 200);
        }
        catch(Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage()
            ], 500);
        }

    }

    //pass
    public function pass_initial_interview(Request $req){
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_initial_interviews,id',
            'applicant_id' => 'required|integer'
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $initialInterview = AII::where('id', $req->schedule_id)->first();

            $oldInterview = clone $initialInterview;

            $initialInterview->update([
                'status'       => ExamStatus::PASSED,
                'updated_at'   => now(),
            ]);

            $applicant = $initialInterview->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $initialInterview->attributes = collect($initialInterview);
            $initialInterview->old = collect($oldInterview);

            Application::where('applicant_id', $req->applicant_id)->update([
                'status'      => ApplicationStatus::ACTIVE,
                'updated_at'  => now(),
            ]);

            $message = 'Congratulations! You have passed your initial interview. Please click the link below and acknowledge the application to continue with the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Pass Applicant JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Initial Interview');

            $this->logCustomMessage('initial_interview_schedule_pass', $initialInterview, Auth::user()->name . " passed applicant {$fullName} in the initial interview", $initialInterview, 'Passed an initial interview', new Activity());

            DB::commit();
            return response()->json([
                'text'  => "Record has been set to passed."
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in passing the initail interview.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    //fail
    public function fail_initial_interview(Request $req){
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_initial_interviews,id',
            'applicant_id' => 'required|integer'
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $initialInterview = AII::where('id', $req->schedule_id)->first();

            $oldInterview = clone $initialInterview;

            $initialInterview->update([
                'status'      => ExamStatus::FAILED,
                'updated_at'  => now(),
            ]);

            $applicant = $initialInterview->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $initialInterview->attributes = collect($initialInterview);
            $initialInterview->old = collect($oldInterview);

            Application::where('applicant_id', $req->applicant_id)->update([
                'status'      => ApplicationStatus::FAILED,
                'updated_at'  => now(),
            ]);

            $message = 'We regret to inform you that you have not passed your initial interview. Please click the link below to view your application status.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Fail Applicant JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Initial Interview');

            $this->logCustomMessage('initial_interview_schedule_fail', $initialInterview, Auth::user()->name . " failed applicant {$fullName} in the initial interview", $initialInterview, 'Failed an initial interview', new Activity());

            DB::commit();
            return response()->json([
                'text'  => "Record has been set to failed."
            ]);

        }catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in failing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    //fetch initial interview status
    public function fetch_by_status_initial(Request $req){
        $valid = Validator::make($req->all(), [
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        try {
            $data = AC::whereHas('application', function($q){
                $q->where('step', Step::STEP_THREE);
            })->whereHas('initialInterview', function($q){
                $q->whereIn('status', [ExamStatus::PENDING, ExamStatus::FAILED, ExamStatus::TAG_TO_ANOTHER_POSITION]);
            })->with(['application' => function($q){
                $q->with(['position' => function($q){
                    $q->select('id','title');
                }])->select('id', 'applicant_id', 'position_id');
            }, 'initialInterview'])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data,
            ],200);

        }catch(Exception $e){
            return response()->json([
                'errors'    =>  [ 'Can`t fetch your request as of now. Contact the developer to fix it. Error Code : SCHEDULE-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }

    }


     //Decline
     public function decline(Request $req) {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_initial_interviews,id',
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $reschedule = AIIR::where('initial_interview_schedule', $req->schedule_id)->first();

            $applicant = $reschedule->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;


            $oldReschedule = clone $reschedule;

            $reschedule->update([
                'status'       => 3,
                'updated_at'   => now(),
            ]);

            $reschedule->attributes = collect($reschedule);
            $reschedule->old = collect($oldReschedule);

            AII::where('id', $req->schedule_id)->update([
                'status'      => 1,
                'updated_at'  => now(),
            ]);

            $this->logCustomMessage('initial_interview_reschedule_declined', $reschedule, Auth::user()->name . ' has declined an initial interview reschedule for applicant ' . $fullName, $reschedule, 'Declined initial interview reschedule', new Activity());


            $applicant = $reschedule->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $message = 'We regret to inform you that your request for reschedule of initial interview has been declined. Please click the link below and acknowledge the schedule to continue the process';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Decline Reschedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Initial Interview');

            DB::commit();
            return response()->json([
                'text'  => "Request has been successfully declined."
            ]);

        }catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function fetch_positions_status(Request $request){
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');
        $today = Carbon::now();
        $data = PHS::where('post_status', 1)->where('slots', '>', 0)
        ->whereDate('post_due', '>=', $today)
        ->orderBy('id','desc')->paginate($paginate);

        return response()->json([
            'data' => $data
        ]);
    }

    //update
    public function update_position(Request $req){

        $position = Application::findOrFail($req->applicant_id);

        $application = Application::where('applicant_id', $req->applicant_id)->first();

        $oldApplication = clone $application;

        $application->update([
            'old_position_id'   => $position->position_id,
            'position_id'       => $req->position_id,
            'updated_at'        => now(),
        ]);

        $application->attributes = collect($application);
        $application->old = collect($oldApplication);

        AII::where('applicant_id', $req->applicant_id)->update([
            'status'            => 3,
            'updated_at'        => now(),
        ]);

        $account = $application->ApplicantAccount()->first();
        $fullName = $account->fname . ' ' . $account->mname . ' ' . $account->lname;


        $message = 'We are informing you that you were tagged to another position. Please click the link below to view your application status.';
        $this->jpNotifService->sendMailNotification($account->email, $message, $fullName);
        $this->jpNotifService->sendNotificationToAdmin('Tag To Another Position JP', Auth::id(), Auth::id(), null, null, null, $fullName);

        $this->logCustomMessage('tag_to_another_position', $application, Auth::user()->name . " has tagged an applicant {$fullName} to another position", $application, 'Tagged applicant to another position', new Activity());

        return response()->json([
            'text' => 'Update successful.'
        ]);
    }

    //fetch choices

    public function fetch_choice(Request $req){

        try {
            $data = APO::where('applicant_id', $req->applicant_id)
            ->with(['applicant' => function ($query){
                $query->select('id', 'email', 'fname', 'mname', 'lname');
            }])
            ->with(['positionOption' => function($query){
                $query->select('id', 'title');
            }])->select('id', 'applicant_id', 'position_option')->orderBy('id', 'asc')->get();

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data
            ], 200);
        }
        catch(Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage()
            ], 500);
        }

    }






}
