<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Illuminate\Http\Request;
use App\Enums\Jobportal\Step;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Enums\Jobportal\ExamStatus;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Hrjp\Models\Application;
use Illuminate\Support\Facades\Validator;
use App\Enums\Jobportal\ApplicationStatus;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Hrjp\Models\Applicant_exam_schedule as AES;
use Suiterus\Hrjp\Models\ApplicantExamReschedule as AER;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;

class ApplicantExamScheduleController extends Controller
{

    use HasCustomLogs;

    protected $jpNotifService;

    public function __construct() {
        $this->jpNotifService = new JobPortalMailNotificationService();
    }

    // FETCH ALL FUNCTION
    public function fetch_all_exam_schedule(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $data = AC::whereHas('application', function ($q) {
                $q->where('step', Step::STEP_FOUR)->whereIn('status', [ApplicationStatus::ACTIVE, ApplicationStatus::INACTIVE]);
            })->with(['examSchedule' => function ($q) {
                $q->whereNotIn('status', [ExamStatus::PASSED, ExamStatus::TAG_TO_ANOTHER_POSITION]);
            }, 'application' => function ($q) {
                $q->with(['position' => function ($q) {
                    $q->select('id', 'title');
                }])->select('id', 'applicant_id', 'position_id');
            }, 'examReschedule'])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    //Process schedule
    public function process_schedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'applicant_id'  =>  'required|integer|exists:hrjp_db.applications,applicant_id',
        ]);

        // TO CHECK IF THE ARGUMENT FAILS
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {
            //check if a record exists in schedules
            if (isset($req->schedule_id) && !is_Null($req->schedule_id)) {
                //Set status to accepted the reschedule table
                if ($req->schedule_status == 4) {
                    $exampSchedule = AER::where('exam_schedule', $req->schedule_id)->first();

                    $oldSchedule = clone $exampSchedule;

                    $exampSchedule->update([
                        'status'       => 2,
                        'updated_at'   => now(),
                    ]);

                    $exampSchedule->attributes = collect($exampSchedule);
                    $exampSchedule->old = collect($oldSchedule);

                    $applicant = $exampSchedule->applicant()->first();
                    $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

                    $this->logCustomMessage('exam_reschedule_approved', $exampSchedule, Auth::user()->name . ' approved an exam reschedule for applicant ' . $fullName, $exampSchedule, 'Approved an exam reschedule', new Activity());

                    //create new schedule
                    $this->create_exam_schedule($req);
                    //soft delete old schedule
                    $record = AES::findOrFail($req->schedule_id);
                    $record->delete();

                    $message ='We are informing you that your request for reschedule of your exam has been approved. Please click the link below and acknowledge the schedule to continue the process';
                    $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
                    $this->jpNotifService->sendNotificationToAdmin('Approve Reschedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Exam');
        
                } else {
                    //For schedule update only of an existing record
                    $examSchedule = AES::where('id', $req->schedule_id)->first();

                    $oldSchedule = clone $examSchedule;

                    $examSchedule->update([
                        "schedule"      =>  $req->schedule,
                        "proctor"    => $req->proctor,
                        "zoomlink"       => $req->zoomlink,
                        "description"     => $req->description,
                        "updated_at"    => now()
                    ]);

                    $examSchedule->attributes = collect($examSchedule);
                    $examSchedule->old = collect($oldSchedule);

                    $applicant = $examSchedule->applicant()->first();
                    $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

                    $this->logCustomMessage('exam_schedule_update', $examSchedule, Auth::user()->name . ' updated an exam reschedule for applicant ' . $fullName, $examSchedule, 'Updated an exam schedule', new Activity());
                }
                DB::commit();
                return response()->json([
                    'message' => "Successfully rescheduled a record."
                ]);
            } else {
                $this->create_exam_schedule($req);
                DB::commit();
                return response()->json([
                    'message' => "Successfully added a schedule."
                ]);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    //Decline
    public function decline(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_exam_schedules,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $examSchedule = AER::where('exam_schedule', $req->schedule_id)->first();

            $applicant = $examSchedule->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $oldExamSchedule = clone $examSchedule;

            $examSchedule->update([
                'status'       => 3,
                'updated_at'   => now(),
            ]);

            $examSchedule->attributes = collect($examSchedule);
            $examSchedule->old = collect($oldExamSchedule);

            AES::where('id', $req->schedule_id)->update([
                'status'      => 1,
                'justification' => $req->justification,
                'updated_at'  => now(),
            ]);

             
            $applicant = $examSchedule->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $message = 'We regret to inform you that your request for reschedule of exam has been declined. Please click the link below and acknowledge the schedule to continue the process';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Decline Reschedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Exam');

            $this->logCustomMessage('exam_reschedule_declined', $examSchedule, Auth::user()->name . ' has declined an exam reschedule for applicant ' . $fullName, $examSchedule, 'Declined exam reschedule', new Activity());

            DB::commit();
            return response()->json([
                'text'  => "Request has been successfully declined."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // CREATE FUNCTION
    public function create_exam_schedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'applicant_id' => 'required|integer|exists:hrjp_db.applicant_accounts,id',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();

        try {
            $interview = AES::create([
                "applicant_id"   => $req->applicant_id,
                "schedule"       => $req->schedule,
                "proctor"    => $req->proctor,
                "zoomlink"       => $req->zoomlink,
                "description"     => $req->description,
                "status"         => 1,
                "created_by"     => $req->user()->id,
                "created_at"     => now(),
            ]);

            $applicant = $interview->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $this->logCustomMessage('exam_schedule_created', $interview, Auth::user()->name . ' has created an exam schedule for applicant ' . $fullName, $interview, 'Create an exam schedule', new Activity());

            $message = 'We are informing you that we have successfully created a schedule for the exam. Please click the link below to acknowledge the schedule and continue the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Set Schedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Exam');

            DB::commit();

            return response()->json([
                'text' => 'Exam schedule created successfully!.',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your request as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /*
        EXAM STATUS
    */

    // FETCH BY FUNCTION
    public function fetch_by_status_exam(Request $req)
    {
        $valid = Validator::make($req->all(), []);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        try {
            $data = AC::whereHas('application', function ($q) {
                $q->where('step', Step::STEP_FIVE);
            })->whereHas('examSchedule', function ($q) {
                $q->whereIn('status', [ExamStatus::PENDING, ExamStatus::FAILED, ExamStatus::TAG_TO_ANOTHER_POSITION]);
            })->with(['application' => function ($q) {
                $q->with(['position' => function ($q) {
                    $q->select('id', 'title');
                }])->select('id', 'applicant_id', 'position_id');
            }, 'examSchedule'])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data,
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['Can`t fetch your request as of now. Contact the developer to fix it. Error Code : SCHEDULE-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }
    public function pass_exam(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_exam_schedules,id',
            'applicant_id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $aes = AES::where('id', $req->schedule_id)->first();

            $applicant = $aes->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $oldAes = clone $aes;

            $aes->update([
                'status'       => ExamStatus::PASSED,
                'updated_at'   => now(),
            ]);

            $aes->attributes = collect($aes);
            $aes->old = collect($oldAes);

            $this->logCustomMessage('exam_schedule_passed', $aes, Auth::user()->name . " passed applicant {$fullName} in the exam", $aes, 'Pass an exam', new Activity());

            Application::where('applicant_id', $req->applicant_id)->update([
                'status'      => ApplicationStatus::ACTIVE,
                'updated_at'  => now(),
            ]);

            $message = 'Congratulations! You have passed your exam. Please click the link below and acknowledge the application to continue with the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Pass Applicant JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Exam');

            DB::commit();
            return response()->json([
                'text'  => "Record has been set to passed."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in passing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
    public function fail_exam(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_exam_schedules,id',
            'applicant_id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $aes = AES::where('id', $req->schedule_id)->first();

            $applicant = $aes->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $oldAes = clone $aes;

            $aes->update([
                'status'       => ExamStatus::FAILED,
                'updated_at'   => now(),
            ]);

            $aes->attributes = collect($aes);
            $aes->old = collect($oldAes);

            $this->logCustomMessage('exam_schedule_failed', $aes, Auth::user()->name . " failed applicant {$fullName} in the exam", $aes, 'Fail an exam', new Activity());

            Application::where('applicant_id', $req->applicant_id)->update([
                'status'      => ApplicationStatus::FAILED,
                'updated_at'  => now(),
            ]);

            $applicant = $aes->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $message = 'We regret to inform you that you have not passed your Exam. Please click the link below to view your application status.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Fail Applicant JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Exam');


            DB::commit();
            return response()->json([
                'text'  => "Record has been set to failed."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in failing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}
