<?php

namespace Suiterus\Hrjp\Controllers;

use Exception;
use Illuminate\Http\Request;
use App\Enums\Jobportal\Step;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Enums\Jobportal\ExamStatus;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Hrjp\Models\Application;
use Illuminate\Support\Facades\Validator;
use App\Enums\Jobportal\ApplicationStatus;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Hrjp\Models\Applicant_exam_interview as AEI;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Hrjp\Models\ApplicantExamInterviewReschedule as AIR;
use Suiterus\Hrjp\Services\JobPortalMailNotificationService;

class ApplicantExamInterviewController extends Controller
{

    use HasCustomLogs;

    protected $jpNotifService;

    public function __construct() {
        $this->jpNotifService = new JobPortalMailNotificationService();
    }


    // CREATE FUNCTION
    public function create_exam_schedule(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'applicant_id' => 'required|integer|exists:hrjp_db.applicant_accounts,id',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();

        try {

            $interview = AEI::create([
                "applicant_id"   => $req->applicant_id,
                "schedule"       => $req->schedule,
                "interviewer"    => $req->interviewer,
                "zoomlink"       => $req->zoomlink,
                "description"     => $req->description,
                "status"         => 1,
                "created_by"     => $req->user()->id,
                "created_at"     => now(),
            ]);

            $applicant = $interview->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $this->logCustomMessage('final_interview_schedule_created', $interview, Auth::user()->name . ' has created a final interview schedule for applicant ' . $fullName, $interview, 'Create a final schedule', new Activity());

            $message = 'We are informing you that we have successfully created a schedule for the final interview. Please click the link below to acknowledge the schedule and continue the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Set Schedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Final Interview');

            DB::commit();

            return response()->json([
                'text' => 'Interview schedule created successfully!.',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your request as of now.'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    // FETCH ALL FUNCTION
    public function fetch_all_exam_schedule(Request $req)
    {
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        try {
            $data = AC::whereHas('application', function ($q) {
                $q->where('step', Step::STEP_SIX)->whereIn('status', [ApplicationStatus::ACTIVE, ApplicationStatus::INACTIVE]);
            })->with(['interviewSchedule' => function ($q) {
                $q->whereNotIn('status', [ExamStatus::PASSED, ExamStatus::TAG_TO_ANOTHER_POSITION])->with(['interviewReschedule']);
            }, 'application' => function ($q) {
                $q->with(['position' => function ($q) {
                    $q->select('id', 'title');
                }])->select('id', 'applicant_id', 'position_id');
            }])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'errors' => ['Can`t fetch your request as of now.'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    //Process schedule
    public function process_schedule(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'applicant_id'  =>  'required|integer|exists:hrjp_db.applications,applicant_id',
        ]);

        // TO CHECK IF THE ARGUMENT FAILS
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {
            //check if a record exists in schedules
            if (isset($req->schedule_id) && !is_Null($req->schedule_id)) {
                //Set status to accepted the reschedule table
                if ($req->schedule_status == 4) {

                    $interview = AIR::where('interview_schedule', $req->schedule_id)->first();

                    $oldInterview = clone $interview;

                    $interview->update([
                        'status'       => 2,
                        'updated_at'   => now(),
                    ]);

                    $interview->attributes = collect($interview);
                    $interview->old = collect($oldInterview);

                    $applicant = $interview->applicant()->first();
                    $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

                    $this->logCustomMessage('final_interview_reschedule_approved', $interview, Auth::user()->name . ' approved a final interview reschedule for applicant ' . $fullName, $interview, 'Approved a final interview reschedule', new Activity());
                    //create new schedule
                    $this->create_exam_schedule($req);

                    $message ='We are informing you that your request for reschedule of your final interview has been approved. Please click the link below and acknowledge the schedule to continue the process';
                    $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
                    $this->jpNotifService->sendNotificationToAdmin('Approve Reschedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Final Interview');
                    //soft delete old schedule
                    $record = AEI::findOrFail($req->schedule_id);
                    $record->delete();
                } else {
                    //For schedule update only of an existing record
                    $interview = AEI::where('id', $req->schedule_id)->first();

                    $oldInterview = clone $interview;

                    $interview->update([
                        "schedule"      =>  $req->schedule,
                        "updated_at"    => now()
                    ]);

                    $interview->attributes = $interview;
                    $interview->old = $oldInterview;

                    $applicant = $interview->applicant()->first();
                    $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

                    $this->logCustomMessage('final_interview_schedule_update', $interview, Auth::user()->name . ' updated a final interview reschedule for applicant ' . $fullName, $interview, 'Updated a final interview schedule', new Activity());
                }
                DB::commit();
                return response()->json([
                    'message' => "Successfully rescheduled a record."
                ]);
            } else {
                $this->create_exam_schedule($req);
                DB::commit();
                return response()->json([
                    'message' => "Successfully added a schedule."
                ]);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    //Decline
    public function decline(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|exists:hrjp_db.applicant_exam_schedules,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            AIR::where('interview_schedule', $req->schedule_id)->update([
                'status'       => 3,
                'updated_at'   => now(),
            ]);

            AEI::where('id', $req->schedule_id)->update([
                'status'      => 1,
                'updated_at'  => now(),
            ]);


            $examSchedule = AEI::whereId($req->schedule_id)->first();
            $applicant = $examSchedule->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $message = 'We regret to inform you that your request for reschedule of final interview has been declined. Please click the link below and acknowledge the schedule to continue the process';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Decline Reschedule JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Final Interview');


            DB::commit();
            return response()->json([
                'text'  => "Request has been successfully declined."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in declining the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    //EXAM STATUS

    public function fetch_by_status_exam(Request $req)
    {
        $valid = Validator::make($req->all(), []);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
        try {
            $data = AC::whereHas('application', function ($q) {
                $q->where('step', Step::STEP_SEVEN);
            })->whereHas('interviewSchedule', function ($q) {
                $q->whereIn('status', [ExamStatus::PENDING, ExamStatus::FAILED, ExamStatus::TAG_TO_ANOTHER_POSITION]);
            })->with(['application' => function ($q) {
                $q->with(['position' => function ($q) {
                    $q->select('id', 'title');
                }])->select('id', 'applicant_id', 'position_id');
            }, 'interviewSchedule'])->select('id', 'fname', 'lname')->paginate($paginate);

            return response()->json([
                'message' => 'Fetch successful.',
                'data' => $data,
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['Can`t fetch your request as of now. Contact the developer to fix it. Error Code : SCHEDULE-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }
    public function pass_exam(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer|',
            'applicant_id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $aei = AEI::where('id', $req->schedule_id)->first();

            $oldExam = clone $aei;

            $aei->update([
                'status'       => ExamStatus::PASSED,
                'updated_at'   => now(),
            ]);

            $aei->update([
                'status'       => ExamStatus::PASSED,
                'updated_at'   => now(),
            ]);

            $applicant = $aei->applicant()->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $aei->attributes = collect($aei);
            $aei->old = collect($oldExam);

            Application::where('applicant_id', $req->applicant_id)->update([
                'status'      => ApplicationStatus::ACTIVE,
                'updated_at'  => now(),
            ]);

            $this->logCustomMessage('final_interview_schedule_pass', $aei, Auth::user()->name . " passed applicant {$fullName} in the final interview", $aei, 'Passed the final interview', new Activity());

            
            $message = 'Congratulations! You have passed your final interview. Please click the link below and acknowledge the application to continue with the process.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Pass Applicant JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Final Interview');

            DB::commit();
            return response()->json([
                'text'  => "Record has been set to passed."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in passing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
    public function fail_exam(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'schedule_id' => 'required|integer',
            'applicant_id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $aei = AEI::where('id', $req->schedule_id)->first();

            $oldExam = clone $aei;

            $aei->update([
                'status'      => ExamStatus::FAILED,
                'updated_at'  => now(),
            ]);

            $applicant = AEI::whereId($req->schedule_id)->first();
            $fullName = $applicant->fname . ' ' . $applicant->mname . ' ' . $applicant->lname;

            $aei->attributes = collect($aei);
            $aei->old = collect($oldExam);

            Application::where('applicant_id', $req->applicant_id)->update([
                'status'      => ApplicationStatus::FAILED,
                'updated_at'  => now(),
            ]);

            $this->logCustomMessage('final_interview_schedule_fail', $aei, Auth::user()->name . " failed applicant {$fullName} in the final interview", $aei, 'Failed the final interview', new Activity());

            $message = 'We regret to inform you that you have not passed your final interview. Please click the link below to view your application status.';
            $this->jpNotifService->sendMailNotification($applicant->email, $message, $fullName);
            $this->jpNotifService->sendNotificationToAdmin('Fail Applicant JP', Auth::id(), Auth::id(), null, null, null, $fullName, 'Final Interview');

            DB::commit();
            return response()->json([
                'text'  => "Record has been set to failed."
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in failing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}
