<?php

namespace Suiterus\Fdg\Controllers\AM;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Http\Controllers\Controller;
use Suiterus\Fdg\Models\AM\PaymentTerm as PT;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Exception;
use Validator;
use Auth;
use DB;

class PaymentTermController extends Controller
{

    public function view_payment_term(Request $req){

        $valid = Validator::make($req->all(),[
            'id'    =>  'required|numeric',
            'name'  =>  'required|string'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $payment_term = PT::where([ 'id' => $req->id, 'name' => $req->name ])->firstOrFail();
                DB::commit();
                return response()->json([
                    'data'  =>  $payment_term
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Payment Term does not exist' ]
                ],404);
            }
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }
    
    public function create_payment_term(Request $req){
        $valid = Validator::make($req->all(),[
            'payment'       =>  'required|min:2|unique:fdg_db.payment_terms,name',
            'credit'        =>  'required|numeric',
            'invoice'       =>  'required',
            'due'           =>  'required',
            'description'   =>  'required|min:2',
        ],[
            'payment.required'  =>  'Payment Term Name is a required field.',
            'payment.min'       =>  'Payment Term Name must be greater than 2 or more string.',
            'payment.unique'    =>  'Payment Term Name is already taken.',
            'credit.required'   =>  'Credit Days is a required field.',
            'credit.numeric'    =>  'Credit Days must be a numeric.',
            'invoice.required'  =>  'Invoice Proportion is a required field.',
            'due.required'      =>  'Due Date based On is a required field.',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            PT::create([
                'name'              =>  $req->payment,
                'portion'           =>  $req->invoice,
                'due_based'         =>  $req->due,
                'credit_days'       =>  $req->credit,
                'description'       =>  $req->description,
                'discount_type'     =>  $req->discount_type,
                'discount_amount'   =>  $req->discount_amount
            ]);
            DB::commit();
            $user = $req->user();
            return response()->json([
                'text'  =>  'Payment Term created successfully!.'
            ]);
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There`s an error creating new Payment Term at the moment. Please Contact us to fix it. Error Code : AM-pt-0x02' ]
            ],400);
        }
    }

    public function list_payment_term(Request $req){

        $paginate = $req->paginate ?  intval($req->paginate) : 10;
        $page = $req->page ? $req->page : 0;
        $sort = $req->sorting ? $req->sorting : 'id_desc';

        if($req->sorting == 'name_asc'){
            $data = PT::orderBy('name', 'asc')->paginate($paginate);
        }else if($req->sorting == 'name_desc'){
            $data = PT::orderBy('name', 'desc')->paginate($paginate);
        }else if($req->sorting == 'created_at_asc'){
            $data = PT::orderBy('created_at', 'asc')->paginate($paginate);
        }else if($req->sorting == 'created_at_desc'){
            $data = PT::orderBy('created_at', 'desc')->paginate($paginate);
        }else if($req->sorting == 'id_asc'){
            $data = PT::orderBy('id', 'asc')->paginate($paginate);
        }else{
            $data = PT::orderBy('id', 'desc')->paginate($paginate);
        }
        
        return response()->json([
            'message' => 'Fetch successful.',
            'sorting' => $sort,
            'data' => $data,
        ]);

    }

    public function search_payment_term(Request $req){
        $valid = Validator::make($req->all(),[
            'keyword'   =>  'required'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            $data = PT::where('name', 'LIKE', '%'.$req->keyword.'%')->orderBy('id', 'desc')->paginate(config('app.manuals.pagination'));
            DB::commit();
            return response()->json([
                'text'  =>  'search keyword found.',
                'data'  =>  $data
            ]);
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There`s an error searching a Payment Term at the moment. Please Contact us to fix it. Error Code : AM-pt-0x03' ]
            ],400);
        }
    }

    public function init_list_deleted_payment_term()
    {
        $deleted_payment_term = PT::onlyTrashed()->orderBy('deleted_at', 'desc')->paginate(config('app.manuals.pagination'));
        return response()->json([
            'text'  =>  'fetch successful.',
            'data'  =>  $deleted_payment_term
        ]);
    }

    public function restore_deleted_payment_term(Request $req)
    {

        DB::beginTransaction();
        try {
            try {
                $payment_term = PT::onlyTrashed()->findOrFail($req->id)->restore();
                DB::commit();
                return response()->json([
                    'text'  =>  'Payment Term has been restored.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json([
                    'errors' => ['Payment Term not found!.'],
                ], 404);
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong while processing your request. Error Code : AM-comp-0x03'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function edit_payment_term(Request $req){
        DB::beginTransaction();
        try{
            $payment_term = PT::findOrFail($req->id);

            if(strcmp($payment_term['name'], $req->name) == 0){
                $notif_message = "Payment Term ".$payment_term['name']." updated";
            }
            else{
                $notif_message = "Payment Term ".$payment_term['name']." updated to ".$req->name;
            }

            $payment_term->update([
                'name'              =>  $req->name,
                'portion'           =>  $req->portion,
                'due_based'         =>  $req->due_based,
                'credit_days'       =>  $req->credit_days,
                'description'       =>  $req->description,
                'discount_type'     =>  $req->discount_type,
                'discount_amount'   =>  $req->discount_amount
            ]);
            
            DB::commit(); 
            return response()->json([
                'text' => 'PaymentTerm Updated!',
            ]);
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There`s an error searching a Payment Term at the moment. Please Contact us to fix it. Error Code : AM-pt-0x04' ]
            ],400);
        }
    }

    public function delete_payment_term (Request $req) {
        DB::beginTransaction();
        try{
            try{
                $pt = PT::findOrFail($req->id);
                $prev_name = $pt['name'];
                $pt->delete();
                DB::commit();

                $user = $req->user();
                return response()->json([
                    'text'  =>  'Payment Term has been deleted.',
                    
                ]);
            }catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Payment Term not found!.'],
                    ],
                    404
                );
            }
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Something went wrong while processing your request. Error Code : AM-pt-0x05' ],
                'message'   =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_deleted_payment_term(Request $req)
    {
        DB::beginTransaction();
        try {
            try {
                $payment_term = PT::onlyTrashed()->findOrFail($req->id)->forceDelete();
                DB::commit();
                return response()->json([
                    'text'  =>  'Payment Term has been deleted.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json([
                    'errors' => ['Payment Term not found!.'],
                ], 404);
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong while processing your request. Error Code : AM-comp-0x03'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

}
