<?php

namespace Suiterus\Fdg\Controllers\AM;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Events\SystemNotification as SN;
use Illuminate\Pagination\CursorPaginator;
use App\Http\Controllers\Controller;
use Suiterus\Fdg\Models\AM\FiscalYear;
use Illuminate\Http\Request;
use Suiterus\Fdg\Models\AM\Company;
use Carbon\Carbon;
use Validator;
use Auth;
use DB;

class FiscalYearController extends Controller
{
    public function view_fiscal(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric',
            'name' => 'required|string',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();
        try {
            try {
                $fy = FiscalYear::where(['id' => $req->id, 'name' => $req->name])->firstOrFail();
                //$this->activityLog(4, 'Fiscal Year ['.$req->name.'] Visited', 'Fiscal Year ['.$req->name.'] has been visited', $req->ip());
                DB::commit();
                return response()->json([
                    'data' => $fy,
                ]);
            } catch (ME $ee) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Fiscal Year does not exist'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function create_fiscal(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'year' => 'required|max:255|unique:fdg_db.fiscal_years,name',
            'start' => 'required|date',
            'end' => 'required|date',
            'company' => 'required|array',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        $comp = $req->company;
        for ($i = 0; $i < count($comp); $i++) {
            $val[] = $comp[$i]["company"];
        }
        $comp = implode(", ", $val);

        //initiate creation
        $error = [];
        DB::beginTransaction();
        try {
            $fy = FiscalYear::create([
                'name' => $req->year,
                'status' => $req->is_disabled && $req->is_disabled === true ? 0 : 1,
                'is_short' => $req->is_short && $req->is_short === true ? 1 : 0,

                'year_start' => Carbon::parse($req->start),
                'year_end' => Carbon::parse($req->end),
                'companies' => $comp,
            ]);

            foreach ($req->company as $comp) {
                try {
                    $company = Company::where(['company' => $comp['company']])->firstOrFail();
                    $fy->fy_company()->create([
                        'company_id' => $company->id,
                    ]);
                } catch (ME $e) {
                    $error[] = ['Company [' . $comp['company'] . '] not found'];
                }
            }
            if (count($error) > 0) {
                DB::rollback();
                return response()->json([
                    'text' => 'Fiscal year values inserted!. But ' . count($error) . ' of companies not exists.<br> ' . implode(',', $error),
                ]);
            } else {
                DB::commit();
                $user = $req->user();
                //broadcast(new SN('created','Fiscal year '.$req->year.' created', $user));
                //$this->activityLog(1, 'Fiscal Year Creation', 'Fiscal Year ['.$req->year.'] has been created', $req->ip());
                return response()->json([
                    'text' => 'Fiscal year values inserted!.',
                ]);
            }
        } catch (\Exception $ex) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x01'],
                    'message' => $ex->getMessage(),
                    'main-error' => $error,
                ],
                400
            );
        }
    }

    public function edit_fiscal(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'name' => 'required|max:255|unique:fdg_db.fiscal_years,name,' . $req->id,
            'year_start' => 'required|date',
            'year_end' => 'required|date',
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        $comp = $req->fy_company;

        for ($i = 0; $i < count($comp); $i++) {
            if ($comp[$i]['company'] != 'Sample Company') {
                $val[] = $comp[$i]["company"]["company"];
            }
        }
        $comp = implode(", ", $val);

        // Fiscal Year Update
        DB::beginTransaction();

        try {
            try {
                $fiscal_year = FiscalYear::findOrFail($req->id);

                if (strcmp($fiscal_year['name'], $req->name) == 0) {
                    $notif_message = "Fiscal year " . $fiscal_year['name'] . " updated";
                } else {
                    $notif_message = "Fiscal year [" . $fiscal_year['name'] . "] updated to [" . $req->name . "]";
                }

                $fiscal_year->update([
                    'name' => $req->name,
                    'status' => $req->status && $req->status === true ? 0 : 1,
                    'is_short' => $req->is_short && $req->is_short === true ? 1 : 0,
                    'year_start' => Carbon::parse($req->year_start),
                    'year_end' => Carbon::parse($req->year_end),
                    'companies' => $comp,
                ]);

                $fiscal_year->fy_company()->forceDelete();
                $error = [];

                foreach ($req->fy_company as $comp) {
                    //Removing the deleted companies array with the requested companies array
                    if (count($req->deleted_company) > 0) {
                        for ($i = 0; $i < count($req->deleted_company); $i++) {
                            if (strcmp($comp['company']['company'], $req->deleted_company[$i]) == 0) {
                                $req->deleted_company = array_diff($req->deleted_company, [$comp['company']['company']]);
                            }
                        }
                    }

                    if ($comp['company'] != 'Sample Company') {
                        try {
                            $company = Company::where(['company' => $comp['company']['company']])->firstOrFail();
                            $fiscal_year->fy_company()->create([
                                'company_id' => $company->id,
                            ]);
                        } catch (ME $e) {
                            $error[] = ['Company [' . $comp['company']['company'] . '] not found'];
                        }
                    }
                }

                $final_deleted_companies = implode(", ", $req->deleted_company);

                if (count($req->deleted_company) > 0) {
                    $notif_message = $notif_message . "\nand " . $final_deleted_companies . " deleted";
                }

                DB::commit();
                $user = $req->user();
                //broadcast(new SN('edited',$notif_message, $user));
                //$this->activityLog(2, 'Fiscal Year Update', $notif_message, $req->ip());
                return response()->json([
                    'text' => 'Fiscal Year Updated!',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Fiscal Year not found!.'],
                    ],
                    404
                );
            }
        } catch (Exception $ex) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong'],
                ],
                500
            );
        }
    }

    public function init_list_fiscal(Request $req)
    {

        $paginate = $req->paginate ?  intval($req->paginate) : 10;
        $page = $req->page ? $req->page : 0;
        $sort = $req->sorting ? $req->sorting : 'id_desc';

        if ($req->sorting === 'name_asc') {
            $data = FiscalYear::orderBy('name', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'name_desc') {
            $data = FiscalYear::orderBy('name', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'created_at_asc') {
            $data = FiscalYear::orderBy('created_at', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'created_at_desc') {
            $data = FiscalYear::orderBy('created_at', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'id_asc') {
            $data = FiscalYear::orderBy('id', 'asc')->paginate($paginate);
        } else {
            $data = FiscalYear::orderBy('id', 'desc')->paginate($paginate);
        }

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data,
            'sorting' => $sort
        ]);
    }

    public function search_fiscal(Request $req)
    {
        $paginate = 10;
        $valid = Validator::make($req->all(), [
            "keyword" => "required|min:2",
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        DB::beginTransaction();
        try {
            DB::commit();
            //$this->activityLog(4, 'Fiscal Year Searched', 'Fiscal Year has been searched', $req->ip());
            return response()->json([
                'text' => 'search found!',
                'data' => FiscalYear::where('name', 'LIKE', '%' . $req->keyword . '%')->paginate($paginate),
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : AM-comp-0x02'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function search_company(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'keyword' => 'required|string|min:2',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        return response()->json([
            'data' => Company::where('company', 'LIKE', '%' . $req->keyword . '%')
                ->limit(5)
                ->cursor(),
        ]);
    }

    public function delete_fiscal(Request $req)
    {
        DB::beginTransaction();
        try {
            try {
                $fy = FiscalYear::findOrFail($req->id);
                $prev_name = $fy['name'];
                $fy->fy_company()->forceDelete();
                $fy->delete();
                DB::commit();

                $user = $req->user();
                // broadcast(new SN('deleted','Fiscal year '.$prev_name.' deleted', $user));
                // $this->activityLog(3, 'Fiscal Year Deleted', 'Fiscal Year  ['.$prev_name.'] has been deleted', $req->ip());
                return response()->json([
                    'text' => 'Fiscal Year has been deleted.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Fiscal Year not found!.'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : AM-comp-0x03'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function init_list_deleted_fiscal()
    {
        $paginate = 10;
        $deleted_fiscal = FiscalYear::onlyTrashed()
            ->orderBy('deleted_at', 'desc')
            ->paginate($paginate);
        return response()->json([
            'text' => 'fetch successful.',
            'data' => $deleted_fiscal,
        ]);
    }

    public function restore_deleted_fiscal(Request $req)
    {
        //Fetching it first via ID
        $restored_fiscal = FiscalYear::onlyTrashed()
            ->findOrFail($req->id)
            ->restore();
        return response()->json([
            'text' => 'recover successful.',
            'data' => $restored_fiscal,
        ]);
    }

    public function delete_deleted_fiscal(Request $req)
    {
        $forceDeleted_fiscal = FiscalYear::onlyTrashed()
            ->findOrFail($req->id)
            ->forceDelete();

        return response()->json([
            'text' => 'forcefully deleted.',
            'data' => $forceDeleted_fiscal,
        ]);
    }
}
