<?php

namespace Suiterus\Fdg\Controllers\AM;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Http\Controllers\Controller;
use Suiterus\Fdg\Models\AM\Company;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Exception;
use Validator;
use Auth;
use DB;

class CompanyController extends Controller
{

    public function view_company(Request $req){
        $valid = Validator::make($req->all(),[
            'id'    =>  'required|numeric',
            'name'  =>  'required|string'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $comp = Company::where([ 'id' => $req->id, 'company' => $req->name ])->firstOrFail();
                /* $this->activityLog(4, 'Company ['.$req->name.'] Visited', 'Company ['.$req->name.'] has been visited', $req->ip()); */
                DB::commit();
                return response()->json([
                    'data'  =>  $comp,
                    'text'  =>  'for visiting'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Company doesnt exists.' ]
                ],404);
            }
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }
    
    public function create_company(Request $req){
        
        // dd($req->all());
        //parent variable validation
        $valid = Validator::make($req->all(),[
            'company'       =>  'required|min:2|string|unique:fdg_db.companies',
            'domain'        =>  'required',
            'abbrevation'   =>  'required',
            'default_tax'   =>  'required',
            'type_of_entry' =>  'required'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        //default value variable validation
        $valid = Validator::make($req->dv,[
            'ccoabo'        =>  'required',
            'doe'           =>  'required',
            'holiday_list'  =>  'required',
            'tax_id'        =>  'required'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        //company information variable validation
        $valid = Validator::make($req->ci,[
            'date_of_incorporation'     =>  'required',
            'description'               =>  'required',
            'email'                     =>  'required',
            'phone_number'              =>  'required'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        //initiate creation
        DB::beginTransaction();
        try{
            Company::create([
                'company'                       =>  $req->company,
                'domain'                        =>  $req->domain,
                'abbrevation'                   =>  $req->abbrevation,
                'parent_company'                =>  $req->parent_company,
                'default_tax'                   =>  $req->default_tax,
                'type_of_entry'                 =>  $req->type_of_entry,
                'ccoabo'                        =>  $req->dv['ccoabo'],
                'tax_id'                        =>  $req->dv['tax_id'],
                'holiday_list'                  =>  implode(',', $req->dv['holiday_list']),
                'date_of_establishment'         =>  Carbon::parse($req->dv['doe']),
                'date_of_incorporation'         =>  Carbon::parse($req->ci['date_of_incorporation']),
                'fax'                           =>  $req->ci['fax'],
                'phone_number'                  =>  $req->ci['phone_number'],
                'email'                         =>  $req->ci['email'],
                'website'                       =>  $req->ci['website'],
                'description'                   =>  $req->ci['description']
            ]);
            $user = $req->user();
            /* broadcast(new SN('created','Company '.$req->company.' created', $user)); */
            /* $this->activityLog(1, 'Company Creation', 'Company has been created', $req->ip()); */
            DB::commit();
            return response()->json([
                'text'  =>  'New Company created!.'
            ]);
        }catch(\Exception $ex){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x01' ],
                'message'   =>  $ex->getMessage()
            ],500);
        }

    }

    public function edit_company(Request $req)
    {

        // dd($req->all());
        // Company Data Validation
        $valid = Validator::make($req->all(),[
            'company'       =>  'required|min:2|string|unique:fdg_db.companies,company,'.$req->id,
            'domain'        =>  'required',
            'abbrevation'   =>  'required',
            'default_tax'   =>  'required',
            'type_of_entry' =>  'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        // Company Update
        DB::beginTransaction();

        try {
            try {
                $company_model = Company::findOrFail($req->id);
                if(strcmp($company_model['company'], $req->company) == 0){
                    $notif_message = "Company ".$company_model['company']." updated";
                }
                else{
                    $notif_message = "Company ".$company_model['company']." updated to ".$req->company;
                }
                $company_model->update([
                    'company'                       =>  $req->company,
                    'domain'                        =>  $req->domain,
                    'abbrevation'                   =>  $req->abbrevation,
                    'parent_company'                =>  $req->parent_company,
                    'default_tax'                   =>  $req->default_tax,
                    'type_of_entry'                 =>  $req->type_of_entry,
                    'ccoabo'                        =>  $req->ccoabo,
                    'tax_id'                        =>  $req->tax_id,
                    'holiday_list'                  =>  implode(',', $req->holiday_list),
                    'date_of_establishment'         =>  Carbon::parse($req->doe),
                    'date_of_incorporation'         =>  Carbon::parse($req->date_of_incorporation),
                    'fax'                           =>  $req->fax,
                    'phone_number'                  =>  $req->phone_number,
                    'email'                         =>  $req->email,
                    'website'                       =>  $req->website,
                    'description'                   =>  $req->description
                ]);

                $user = $req->user();
                /* broadcast(new SN('edited',$notif_message, $user)); */
                /* $this->activityLog(2, 'Company Update', 'Company has been updated', $req->ip()); */
                DB::commit();
                return response()->json([
                    'text' => 'Company Updated!',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Company not found!.'],
                    ],
                    404
                );
            }
        } catch (Exception $ex) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong']
            ], 500);
        }
    }





    public function init_list_company(Request $req){
        $paginate = $req->paginate ?  intval($req->paginate) : 10;
        $page = $req->page ? $req->page : 0;
        $sort = $req->sorting ? $req->sorting : 'id_desc';

        if($sort === 'name_asc'){
            $smsg = 'Name Ascending';
            $d = Company::orderBy('company', 'asc')->paginate($paginate);
        }else if($sort=== 'name_desc'){
            $smsg = 'Name Descending';
            $d = Company::orderBy('company', 'desc')->paginate($paginate);
        }else if($sort === 'created_at_asc'){
            $smsg = 'Created Date Ascending';
            $d = Company::orderBy('created_at', 'asc')->paginate($paginate);
        }else if($sort === 'created_at_desc'){
            $smsg = 'Created Date Descending';
            $d = Company::orderBy('created_at', 'desc')->paginate($paginate);
        }else if($sort === 'id_asc'){
            $smsg = 'ID Ascending';
            $d = Company::orderBy('id', 'asc')->paginate($paginate);
        }else{
            $smsg = 'Default';
            $d = Company::orderBy('id', 'desc')->paginate($paginate);
        }
        //identify
        if($page != 0){
            $pmsg = 'Page '.$page;
        }else if($sort != 'none'){
            $pmsg = 'sorted by '.$smsg;
        }else{
            $pmsg = 'Page '.$page.' and sorted by'.$smsg;
        }
        /* $this->activityLog(4, 'Company listed', 'Company list has been visited on/with : '.$pmsg, $req->ip()); */
        return response()->json([
            'text'  =>  'fetch successful.',
            'data'  =>  $d,
            'sorting' => $sort
        ]);
    }

    public function search_company(Request $req){
        $paginate = 10;
        $valid = Validator::make($req->all(),[
            'keyword'   =>  'required|min:2'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            /* $this->activityLog(4, 'Company Searched', 'Company has been searched', $req->ip()); */
            DB::commit();
            return response()->json([
                'text'  =>  'search found!',
                'data'  =>  Company::where('company', 'LIKE', '%'.$req->keyword.'%')->paginate($paginate)
            ]);
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Something went wrong while processing your request. Error Code : AM-comp-0x02' ],
                'message'   =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_company(Request $req){
        DB::beginTransaction();
        try{
            try{
                $comp = Company::findOrFail($req->id);
                $prev_comp = $comp['company'];
                /* $accounting_period = AP::where('company', $prev_comp);
                $accounting_period->update([
                    'company' => $prev_comp."(deleted)"
                ]);
                $comp->fiscalYear()->delete();
                $comp->accountingDimension()->delete(); */
                $comp->delete();
                $user = $req->user();
                /* broadcast(new SN('deleted','Company '.$prev_comp.' deleted', $user)); */
                /* $this->activityLog(3, 'Company Deleted', 'Company has been deleted', $req->ip()); */
                DB::commit();
                return response()->json([
                    'text'  =>  'Company has been deleted.'
                ]);
            }catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Company not found!.'],
                    ],
                    404
                );
            }
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Something went wrong while processing your request. Error Code : AM-comp-0x03' ],
                'message'   =>  $e->getMessage()
            ],500);
        }
    }

    public function init_list_deleted_company(){
        $paginate = 10;
        $deleted_companies = Company::onlyTrashed()->orderBy('deleted_at', 'desc')->paginate($paginate);
        return response()->json([
            'text'  =>  'fetch successful.',
            'data'  =>  $deleted_companies
        ]);
    }

    public function restore_deleted_company(Request $req){

        //Fetching it first via ID
        $prev_comp = Company::onlyTrashed()->findOrFail($req->id);
        $restored_company = Company::onlyTrashed()->findOrFail($req->id)->restore();
        
        
        //Dependencies of other submodules to Company
        DB::beginTransaction();
        try{
            /* $accounting_period = AP::where('company', $prev_comp['company']."(deleted)");
            $accounting_period->update([
                'company' => $prev_comp['company']
            ]); */
            
            DB::commit();
            return response()->json([
                'text' => 'recover successful.',
                'data' => $restored_company
            ]);
    
            
        }catch(\Exception $e){
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong']
            ], 500);
        }
        
      
    }

    public function delete_deleted_company(Request $req){
        $forceDeleted_company = Company::onlyTrashed()->findOrFail($req->id)->forceDelete();
        return response()->json([
            'text' => 'forcefully deleted.',
            'data' => $forceDeleted_company
        ]);
    }

}
