<?php

namespace Suiterus\Fdg\Controllers\AM;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Events\SystemNotification as SN;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Suiterus\Fdg\Models\AM\Company;
use Suiterus\Fdg\Models\AM\AccountingPeriod as AP;
use Carbon\Carbon;
use Validator;
use Auth;
use DB;

class AccountingPeriodController extends Controller
{
    public function view_accounting_period(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric',
            'period' => 'required|string',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();
        try {
            try {
                $ap = AP::where(['id' => $req->id, 'period' => $req->period])->firstOrFail();
                //$this->activityLog(4, 'Accounting Period ['.$req->period.'] Visited', 'Accounting Period ['.$req->period.'] has been visited', $req->ip());
                DB::commit();
                return response()->json([
                    'data' => $ap,
                ]);
            } catch (ME $ee) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Accounting Period does not exist'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function create(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'period' => 'required|unique:fdg_db.accounting_periods,period|max:255',
            'company' => 'required|max:255',
            'start' => 'required|date',
            'end' => 'required|date',
            'documents' => 'required|array',
        ]);
        if ($validate->fails()) {
            return response()->json(
                [
                    "errors" => $validate->errors(),
                ],
                400
            );
        }

        $docu = $req->documents;
        for ($i = 0; $i < count($docu); $i++) {
            $val[] = $docu[$i]["docu"]["text"];
        }
        $documents = implode(", ", $val);

        DB::beginTransaction();
        try {
            $company = Company::where(['company' => $req->company])->firstOrFail();
            $ap = AP::create([
                'period' => $req->period,
                'company' => $company->company,
                'start' => Carbon::parse($req->start),
                'end' => Carbon::parse($req->end),
                'documents' => $documents,
            ]);
            //
            foreach ($req->documents as $key) {
                $ap->ap_documents()->create([
                    "document_type" => $key['docu']['value'],
                    "closed" => $key['close'],
                ]);
            }
            DB::commit();
            $user = $req->user();
            //broadcast(new SN('created','Accounting period '.$req->period.' created', $user));
            //$this->activityLog(1, 'Accounting Period Creation', 'Accounting Period ['.$req->period.'] has been created', $req->ip());
            return response()->json([
                "message" => "Accounting Period values inserted!",
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    "errors" => ["Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-accntp-0x01"],
                    "message" => $e->getMessage(),
                ],
                400
            );
        }
    }

    public function edit_accounting_period(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'period' => 'required|max:255',
            'company' => 'required|max:255',
            'start' => 'required|date',
            'end' => 'required|date',
            'ap_documents' => 'required|array',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        $docu = $req->ap_documents;
        for ($i = 0; $i < count($docu); $i++) {
            $val[] = $docu[$i]["document_type"];
        }
        $documents = implode(", ", $val);
        DB::beginTransaction();
        try {
            try {
                $accounting_period = AP::findOrFail($req->id);
                if (strcmp($accounting_period['period'], $req->period) == 0) {
                    $notif_message = "Accounting period [" . $accounting_period['period'] . "] updated";
                } else {
                    $notif_message = "Accounting period [" . $accounting_period['period'] . "] updated to [" . $req->period . "]";
                }

                $accounting_period->update([
                    'period' => $req->period,
                    'company' => $accounting_period->company,
                    'start' => Carbon::parse($req->start),
                    'end' => Carbon::parse($req->end),
                    'documents' => $documents,
                ]);
                $accounting_period->ap_documents()->forceDelete();
                $error = [];
                foreach ($req->ap_documents as $key) {
                    try {
                        $accounting_period->ap_documents()->create([
                            'document_type' => $key['document_type'],
                            'closed' => $key['closed'],
                        ]);
                    } catch (ME $e) {
                        $error[] = ['Accounting Period not found'];
                    }
                }
                DB::commit();
                if (count($error) > 0) {
                    DB::rollback();
                    return response()->json([
                        'text' => 'Accounting Period values updated!. But ' . count($error) . ' of companies not exists.<br> ' . implode(',', $error),
                    ]);
                } else {
                    DB::commit();
                    $user = $req->user();
                    //broadcast(new SN('edited',$notif_message, $user));
                    //$this->activityLog(2, 'Accounting Period Update', $notif_message, $req->ip());
                    return response()->json([
                        'text' => 'Accounting Period values updated!.',
                        'data' => $req->all(),
                    ]);
                }
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Accounting Dimension not found!.'],
                    ],
                    404
                );
            }
        } catch (Exception $ex) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong'],
                ],
                500
            );
        }
    }

    public function init_list_accounting_period(Request $req)
    {
        $paginate = $req->paginate ?  intval($req->paginate) : 10;
        $page = $req->page ? $req->page : 0;
        $sort = $req->sorting ? $req->sorting : 'id_desc';



        if ($req->sorting === 'period_asc') {
            $data = AP::orderBy('period', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'period_desc') {
            $data = AP::orderBy('period', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'company_asc') {
            $data = AP::orderBy('company', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'company_desc') {
            $data = AP::orderBy('company', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'yearstart_asc') {
            $data = AP::orderBy('start', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'yearstart_desc') {
            $data = AP::orderBy('start', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'yearend_asc') {
            $data = AP::orderBy('end', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'yearend_desc') {
            $data = AP::orderBy('end', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'created_at_asc') {
            $data = AP::orderBy('created_at', 'asc')->paginate($paginate);
        } elseif ($req->sorting === 'created_at_desc') {
            $data = AP::orderBy('created_at', 'desc')->paginate($paginate);
        } elseif ($req->sorting === 'id_asc') {
            $data = AP::orderBy('id', 'asc')->paginate($paginate);
        } else {
            $data = AP::orderBy('id', 'desc')->paginate($paginate);
        }

        return response()->json([
            'message' => 'Fetch successful.',
            'sorting' => $sort,
            'data' => $data,
        ]);
    }

    public function search_accounting_period(Request $req)
    {

        $paginate = 10;
        $valid = Validator::make($req->all(), [
            'keyword' => 'required|min:2',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        DB::beginTransaction();
        try {
            DB::commit();
            //$this->activityLog(4, 'Accounting Period Searched', 'Accounting Period has been searched', $req->ip());
            return response()->json([
                'text' => 'Search successful!',
                'data' => AP::where('period', 'LIKE', '%' . $req->keyword . '%')->paginate($paginate),
                'keyword' => $req->keyword,
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : AM-comp-0x02'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }
    public function search_company(Request $req)
    {
        $paginate = 10;
        $valid = Validator::make($req->all(), [
            'keyword' => 'required|string|min:2',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        return response()->json([
            'data' => Company::where('company', 'LIKE', '%' . $req->keyword . '%')
                ->limit($paginate)
                ->cursor(),
        ]);
    }

    public function delete_accounting_period(Request $req)
    {
        DB::beginTransaction();
        try {
            try {
                $ap = AP::findOrFail($req->id);
                $prev_period = $ap['period'];
                $ap->ap_documents()->forceDelete();
                $ap->delete();
                DB::commit();

                $user = $req->user();
                //broadcast(new SN('deleted','Accounting period '.$prev_period.' deleted', $user));
                //$this->activityLog(3, 'Accounting Period Deleted', 'Accounting Period  ['.$prev_period.'] has been deleted', $req->ip());
                return response()->json([
                    'text' => 'Accounting Period has been deleted.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Accounting Period not found!.'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : AM-comp-0x03'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function init_list_deleted_accounting_period()
    {
        $paginate = 10;
        $deleted_period = AP::onlyTrashed()
            ->orderBy('deleted_at', 'desc')
            ->paginate($paginate);
        return response()->json([
            'text' => 'fetch successful.',
            'data' => $deleted_period,
        ]);
    }

    public function restore_deleted_accounting_period(Request $req)
    {
        //Fetching it first via ID
        $restored_period = AP::onlyTrashed()
            ->findOrFail($req->id)
            ->restore();
        return response()->json([
            'text' => 'recover successful.',
            'data' => $restored_period,
        ]);
    }

    public function delete_deleted_accounting_period(Request $req)
    {
        $forceDeleted_period = AP::onlyTrashed()
            ->findOrFail($req->id)
            ->forceDelete();

        return response()->json([
            'text' => 'forcefully deleted.',
            'data' => $forceDeleted_period,
        ]);
    }
}
