<?php

namespace Suiterus\Dms\Services\Repositories;

use Activity;
use App\Enums\Dms\DriveType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Http\JsonResponse;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Contracts\Repositories\FolderListing;
use Suiterus\Dms\Contracts\Repositories\RootFolderListing;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFile;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFolder;
use Suiterus\Dms\Models\Files\File;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Traits\Repositories\HasChildren;
use Suiterus\Dms\Traits\Repositories\HasPath;

class PublicFolderService extends FolderService implements RootFolderListing, SearchAbleFolder, SearchAbleFile, FolderListing
{

    use HasPath, HasChildren, HasCustomLogs;

    /**
     * It returns a paginated list of sections where the parent_id is equal to the  parameter
     * and the type is equal to the ::PUBLIC constant
     * @param int parentId The id of the parent folder
     * @param DriveType driveType This is the type of drive you want to list.
     * @param int page The number of items to be returned per page
     * @return JsonResponse A JsonResponse object.
     */
    public function listRootFolder(int $parentId, DriveType $driveType, int $page): LengthAwarePaginator
    {
        return Section::where([['parent_id', $parentId], ['type', $driveType::PUBLIC]])->latest()->paginate($page);
    }

    /**
     * It searches for a folder in a specific parent folder and returns the result in a paginated json
     * response
     * @param int parentId The id of the parent folder
     * @param string folderName The name of the folder you want to search for.
     * @param DriveType driveType This is the type of drive you want to search.
     * @param int page The number of items to be displayed per page
     * @return JsonResponse A JsonResponse object.
     */
    public function searchFolder(int $parentId, string $folderName, DriveType $driveType, int $page): LengthAwarePaginator
    {
        $data = Section::where([['type', $driveType::PUBLIC], ['parent_id', $parentId], ['name', 'LIKE', '%' . $folderName . '%']])
            ->orderBy('id', 'desc')->paginate($page);
        if ($folderName) {
            $this->logCustomMessage(
                'search_folder_shared',
                null,
                Auth::user()->name . ' searched for "' . $folderName . '"',
                null,
                'Search',
                new Activity()
            );
        }
        return $data;
    }

    public function searchFile(int $sectionId, string $fileName, DriveType $driveType, int $page): LengthAwarePaginator
    {

        $files = File::where([['name', 'LIKE', '%' . $fileName . '%'], ['section_id', $sectionId], ['type', $driveType::PUBLIC]])
            ->orderBy('name', 'asc')->paginate($page);

        if ($fileName) {
            $this->logCustomMessage(
                'search_file_shared',
                null,
                Auth::user()->name . ' searched for "' . $fileName . '"',
                null,
                'Search',
                new Activity()
            );
        }
        return $files;
    }

    // Fetch list of sections to which folders/files can be moved/copied.
    public function movableFolderList(array $folders, DriveType $driveType, int $currentSectionId)
    {
        $all_children = new Collection();

        foreach ($folders as $folder) {
            $folder = Section::findOrFail($folder['id']);
            $all_children->push($this->getAllChildren($folder));
        }

        $not_in_folders = $all_children->flatten()->merge(collect($folders))->pluck('id');

        $data = Section::where([['type', $driveType::PUBLIC], ['id', '!=', $currentSectionId]])
            ->whereNotIn('id', $not_in_folders)
            ->with('parent')
            ->get();

        foreach ($data as $folder) {
            $folder->path = $this->getPath($folder);
        }

        return $data;
    }
}
