<?php

namespace Suiterus\Dms\Services\Repositories;

use Activity;
use App\Enums\Dms\DriveType;
use App\Models\AccessManagement\GroupManagement\ModelHasRole;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Contracts\Repositories\RootFolderListing;
use Suiterus\Dms\Models\Repositories\Section;
use App\Models\AccessManagement\GroupManagement\Role;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Suiterus\Dms\Contracts\Repositories\FolderListing;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFile;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFolder;
use Suiterus\Dms\Models\Files\File;
use Suiterus\Dms\Traits\Repositories\HasChildren;
use Suiterus\Dms\Traits\Repositories\HasPath;

class PrivateFolderService extends FolderService implements RootFolderListing, SearchAbleFolder, SearchAbleFile, FolderListing
{

    use HasPath, HasChildren, HasCustomLogs;

    /**
     * It fetches all the folders that are private and have access to the user.
     * @param int parentId The id of the parent folder
     * @param DriveType driveType This is a class that has a constant PRIVATE.
     * @param int page the number of items to be returned
     * @return JsonResponse
     */
    public function listRootFolder(int $parentId, DriveType $driveType, int $page): LengthAwarePaginator
    {

        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $sections = Section::where([['parent_id', $parentId], ['type', $driveType::PRIVATE]])
            ->whereHas('access', function (Builder $query) use ($user_groups) {
                $query->where('user_id', Auth::user()->id)
                    ->orWhereIn('group_id', $user_groups);
            })->latest()->paginate($page);

        $sections->transform(function ($item) {
            return $item = $this->fetch_access_group($item);
        });
        return $sections;
    }


    /**
     * It fetches the access groups for a section and returns the section with the access groups
     * attached
     * @param Section section The section object
     * @return Section An array of objects.
     */
    public function fetch_access_group(Section $section): Section
    {
        $groupIds = $section->access->pluck('group_id');
        $groups = Role::whereIn('id', $groupIds)->whereNotIn('name', ['Developer', 'Admin', 'Super Admin'])->get(['id', 'name']);
        $users = User::whereNotIn('id', [1, 2])->whereHas('roles', function (Builder $query) use ($groupIds) {
            $query->whereIn('id', $groupIds);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get(['id', 'name']);

        $currentUser = $users->find(Auth::user()->id);
        if ($currentUser) {
            $currentUser->name = 'Me';
        }

        $section->members = array_merge($groups->toArray(), $users->toArray());
        return $section;
    }

    public function fetchFileAccessGroup(File $file): File
    {
        $fileIds = $file->access->pluck('group_id');
        $groups = Role::whereIn('id', $fileIds)->whereNotIn('name', ['Developer', 'Admin', 'Super Admin'])->get(['id', 'name']);
        $users = User::whereNotIn('id', [1, 2])->whereHas('roles', function (Builder $query) use ($fileIds) {
            $query->whereIn('id', $fileIds);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get(['id', 'name']);

        $currentUser = $users->find(Auth::user()->id);
        if ($currentUser) {
            $currentUser->name = 'Me';
        }

        $file->members = array_merge($groups->toArray(), $users->toArray());
        return $file;
    }

    /**
     * It fetches all the folders that are private and have the same parent id and name as the one
     * passed in the function.
     * </code>
     * @param int parentId The id of the parent folder
     * @param string folderName The name of the folder you want to search for.
     * @param DriveType driveType This is the type of drive you want to search in.
     * @param int page the number of items to be displayed per page
     * @return JsonResponse A JsonResponse object.
     */
    public function searchFolder(int $parentId, string $folderName, DriveType $driveType, int $page): LengthAwarePaginator
    {
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data =  Section::where([['type', $driveType::PRIVATE], ['parent_id', $parentId], ['name', 'LIKE', '%' . $folderName . '%']])
            ->whereHas('access', function (Builder $query) use ($user_groups) {
                $query->where('user_id', Auth::user()->id)
                    ->orWhereIn('group_id', $user_groups);
            })->orderBy('id', 'desc')->paginate($page);
        $data->transform(function ($item) {
            return $item = $this->fetch_access_group($item);
        });
        if ($folderName) {
            $this->logCustomMessage(
                'search_folder_private',
                null,
                Auth::user()->name . ' searched for "' . $folderName . '"',
                null,
                'Search',
                new Activity()
            );
        }
        return $data;
    }

    public function searchFile(int $sectionId, string $fileName, DriveType $driveType, int $page): LengthAwarePaginator
    {
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = File::where([['name', 'LIKE', '%' . $fileName . '%'], ['section_id', $sectionId], ['type', $driveType::PRIVATE]])
            ->whereHas('access', function (Builder $query) use ($user_groups) {
                $query->where('user_id', Auth::user()->id)
                    ->orWhereIn('group_id', $user_groups);
            })->orderBy('name', 'asc')->paginate($page);
        $data->transform(function ($item) {
            return $item = $this->fetchFileAccessGroup($item);
        });

        if ($fileName) {
            $this->logCustomMessage(
                'search_file_private',
                null,
                Auth::user()->name . ' searched for "' . $fileName . '"',
                null,
                'Search',
                new Activity()
            );
        }

        return $data;
    }

    // Fetch list of sections to which folders/files can be moved/copied.
    public function movableFolderList(array $folders, DriveType $driveType, int $currentSectionId)
    {
        $all_children = new Collection();

        foreach ($folders as $folder) {
            $folder = Section::findOrFail($folder['id']);
            $all_children->push($this->getAllChildren($folder));
        }

        $not_in_folders = $all_children->flatten()->merge(collect($folders))->pluck('id');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = Section::where([['type', $driveType::PRIVATE], ['id', '!=', $currentSectionId]])
            ->whereHas('access', function ($query) use ($user_groups) {
                $query->where('user_id', Auth::user()->id)
                    ->orWhereIn('group_id', $user_groups);
            })
            ->whereNotIn('id', $not_in_folders)
            ->with('parent')
            ->get();

        foreach ($data as $folder) {
            $folder->path = $this->getPath($folder);
        }

        return $data;
    }
}
