<?php

namespace Suiterus\Dms\Services\Repositories;

use Activity;
use App\Enums\Dms\DriveType;
use Suiterus\Dms\Contracts\Repositories\RootFolderListing;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Contracts\Repositories\FolderListing;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFile;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFolder;
use Suiterus\Dms\Models\Files\File;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Traits\Repositories\HasChildren;
use Suiterus\Dms\Traits\Repositories\HasPath;

class PersonalFolderService extends FolderService implements RootFolderListing, SearchAbleFolder, SearchAbleFile, FolderListing
{

    use HasPath, HasChildren;

    /**
     * It returns a paginated list of folders and files that are in the root folder of a drive
     * @param int parentId The id of the parent folder.
     * @param DriveType driveType This is the type of drive you want to list.
     * @param int page the number of items to be returned
     * @return JsonResponse A JsonResponse object.
     */
    public function listRootFolder(int $parentId, DriveType $driveType, int $page): LengthAwarePaginator
    {
        return Section::where([['type', $driveType::PERSONAL], ['parent_id', $parentId]])
            ->whereHas('access', function (Builder $query) {
                $query->where('user_id', Auth::user()->id);
            })->latest()->paginate($page);
    }

    /**
     * It searches for a folder in a specific folder by name and returns the result in a paginated
     * format.
     * @param int parentId The id of the parent folder
     * @param string folderName The name of the folder you want to search for.
     * @param DriveType driveType This is the type of drive you want to search in.
     * @param int page the number of items to be displayed per page
     * @return JsonResponse A JsonResponse object.
     */
    public function searchFolder(int $parentId, string $folderName, DriveType $driveType, int $page): LengthAwarePaginator
    {
        $folders = Section::where([['type', $driveType::PERSONAL], ['parent_id', $parentId], ['name', 'LIKE', '%' . $folderName . '%']])
            ->whereHas('access', function (Builder $query) {
                $query->where('user_id', Auth::user()->id);
            })->orderBy('id', 'desc')->paginate($page);

        if ($folderName) {
            $this->logCustomMessage(
                'search_folder_personal',
                null,
                Auth::user()->name . ' searched for "' . $folderName . '"',
                null,
                'Search',
                new Activity()
            );
        }

        return $folders;
    }

    public function searchFile(int $sectionId, string $fileName, DriveType $driveType, int $page): LengthAwarePaginator
    {

        $files = File::where([['name', 'LIKE', '%' . $fileName . '%'], ['section_id', $sectionId], ['type', $driveType::PERSONAL]])
            ->whereHas('access', function (Builder $query) {
                $query->where('user_id', Auth::user()->id);
            })->orderBy('name', 'asc')->paginate($page);

        if ($fileName) {
            $this->logCustomMessage(
                'search_file_personal',
                null,
                Auth::user()->name . ' searched for "' . $fileName . '"',
                null,
                'Search',
                new Activity()
            );
        }

        return $files;
    }

    // Fetch list of sections to which folders/files can be moved/copied.
    public function movableFolderList(array $folders, DriveType $driveType, int $currentSectionId)
    {
        $all_children = new Collection();

        foreach ($folders as $folder) {
            $folder = Section::findOrFail($folder['id']);
            $all_children->push($this->getAllChildren($folder));
        }

        $not_in_folders = $all_children->flatten()->merge(collect($folders))->pluck('id');

        $data = Section::where([['type', $driveType::PERSONAL], ['id', '!=', $currentSectionId]])
            ->whereHas('access', function ($query) {
                $query->where('user_id', Auth::user()->id);
            })
            ->whereNotIn('id', $not_in_folders)
            ->with('parent')
            ->get();
        foreach ($data as $folder) {
            $folder->path = $this->getPath($folder);
        }

        return $data;
    }
}
