<?php

namespace Suiterus\Dms\Services\Repositories;

use Activity;
use stdClass;
use Exception;
use Illuminate\Http\JsonResponse;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Dms\Classes\DriveType;
use Suiterus\Dms\Models\Files\File;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Suiterus\Dms\Enums\Log\FolderLogType;
use App\Enums\Dms\DriveType as DmsDriveType;
use Illuminate\Database\Eloquent\Collection;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Contracts\Repositories\Folder;
use Suiterus\Dms\Models\CustomForm\CustomFormValue;
use Suiterus\Dms\Models\Repositories\SectionAccess;
use App\Models\AccessManagement\GroupManagement\Role;

class FolderService implements Folder
{

    use HasCustomLogs;

    /**
     * It creates a folder in the database
     * @param string folderName The name of the folder
     * @param string description
     * @param int parentId the id of the parent folder
     * @param int driveType 1 = Personal Drive, 2 = Shared Drive
     * @return JsonResponse A JsonResponse object.
     */
    public function createFolder(string $folderName, string $description, int $parentId, int $driveType, array $users, array $groups, $option): JsonResponse
    {
        DB::beginTransaction();
        try {

            if ($option == 'continue') {
                $existingFolder = Section::where([
                    ['name', $folderName], ['parent_id', $parentId], ['type', $driveType]
                ])->first();
                if ($existingFolder) {
                    $folderName = $this->generateUniqueFolderName($folderName, $parentId, $driveType);
                }
            }

            $section = Section::create([
                'name' => $folderName,
                'description' => $description,
                'type' => $driveType,
                'parent_id' => $parentId,
                'status' => 1,
                'created_by' => Auth::user()->id
            ]);

            if ($driveType == DmsDriveType::PUBLIC) {
                $this->logCustomMessage('folder_create', $section, 'The ' . $section->name . ' folder has been created by ' . Auth::user()->name, $section, FolderLogType::CREATE, new Activity());
                return response()->json([
                    'text' => 'Section has been created.',
                ]);
            }

            $section->access()->create([
                'user_id' => Auth::user()->id,
                'access_level' => 1,
            ]);

            if ($users || $groups) {
                $devs = Role::where(['name' => 'Developer'])->first();
                $devsId = $devs ? $devs->id : null;

                $isCreateDevAccess = $section->access()->where(['group_id' => $devsId])->count() == 0;

                if ($isCreateDevAccess) {
                    $section->access()->create([
                        'user_id' => 0,
                        'group_id' => $devs->first()->id,
                        'access_level' => 1,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }

                foreach ($users as $user) {
                    $section->access()->create([
                        'user_id' => $user,
                        'group_id' => 0,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }

                foreach ($groups as $group) {
                    $section->access()->create([
                        'user_id' => 0,
                        'group_id' => $group,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }

            DB::commit();
            $this->logCustomMessage('folder_create', $section, 'The ' . $section->name . ' folder has been created by ' . Auth::user()->name, $section, FolderLogType::CREATE, new Activity());
            return response()->json([
                'text' => 'Section has been created.',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function renameFolder(int $sectionId, string $folderName, ?string $description, int $parentId, ?string $option, int $driveType, array $users, array $groups): JsonResponse
    {
        DB::beginTransaction();
        try {
            if ($option == 'continue') {
                $existingFolder = Section::where([
                    ['name', $folderName], ['parent_id', $parentId], ['type', $driveType]
                ])->first();
                if ($existingFolder) {
                    $folderName = $this->generateUniqueFolderName($folderName, $parentId, $driveType);
                }
            }

            $section = Section::without([
                'access',
                'files',
                'author.roles',
                'author.permissions',
                'author.storage',
                'author.employeeMetaInfo',
                'author.supervisor',
                'author.exitInterview',
                'author.userProfilePicture',
                'author.profileBasicInfo',
            ])->find($sectionId);

            $oldSection = clone $section;
            $section = Section::find($sectionId);

            if ($driveType == DmsDriveType::PRIVATE) {
                foreach ($users as $user) {
                    $userHasSectionAccess = SectionAccess::where([
                        ['section_id', $sectionId],
                        ['user_id', $user],
                    ])->first();

                    if (!$userHasSectionAccess) {
                        $section->access()->create([
                            'user_id' => $user,
                            'group_id' => 0,
                            'access_level' => 2,
                            'shared_by' => Auth::id(),
                            'shared_date' => now()
                        ]);
                    }
                }

                foreach ($groups as $group) {
                    $groupHasSectionAccess = SectionAccess::where([
                        ['section_id', $sectionId],
                        ['group_id', $group],
                    ])->first();

                    if (!$groupHasSectionAccess) {
                        $section->access()->create([
                            'user_id' => 0,
                            'group_id' => $group,
                            'access_level' => 2,
                            'shared_by' => Auth::id(),
                            'shared_date' => now()
                        ]);
                    }
                }
            }

            $section->name = $folderName;
            $section->description = $description;
            $section->save();
            
            $section->attributes = collect($section);
            $section->old = collect($oldSection);
            DB::commit();
            if ($section->name != $oldSection->name) {
                $this->logCustomMessage('folder_rename', $section, 'The ' . $oldSection->name . ' folder has been renamed to ' . $folderName . ' by ' . Auth::user()->name, $section, FolderLogType::RENAME, new Activity());
                $sectionNameLog = $section->name;
            }
            if ($section->description != $oldSection->description) {
                $sectionNameLog = isset($sectionNameLog)  ? $sectionNameLog : $oldSection->name;
                $this->logCustomMessage('folder_update_description', $section, Auth::user()->name . ' updated the folder description for ' . $sectionNameLog, $section, FolderLogType::UPDATE_DESCRIPTION, new Activity());
            }
            return response()->json([
                'text'  => 'folder name/description has been updated',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    /**
     * It fetches all the folders from the database and then transforms the data to add a new property
     * to each folder
     * @param int page the number of items to be displayed per page
     * @param int parentSectionId The id of the parent folder
     * @param int driveType 1 = Private, 2 = Public
     * @return JsonResponse A JsonResponse object.
     */
    public function fetchFolder(int $page, int $parentSectionId, int $driveType): JsonResponse
    {
        $data = Section::where([['parent_id', $parentSectionId], ['type', $driveType]])
            ->orderBy('name', 'asc')
            ->paginate($page);

        if ($driveType == DmsDriveType::PRIVATE) {
            $data->transform(function ($item) {
                return $item = (new PrivateFolderService)->fetch_access_group($item);
            });
        }

        return response()->json([
            'data' => $data,
        ]);
    }

    /**
     * It fetches all the files in a folder and returns a json response
     * @param int sectionId The id of the section
     * @param int driveType 1 =&gt; Private, 2 =&gt; Public, 3 =&gt; Shared
     * @param int page the number of items to be displayed per page
     */
    public function fetchFolderFile(int $sectionId, int $driveType, int $page): JsonResponse
    {
        $data = File::where([['section_id', $sectionId], ['type', $driveType]])
            ->orderBy('name', 'asc')->without([
                'modifier',
                'extraFields',
                'access',
                'versions',
                'documentType'
            ])
            ->paginate($page);

        if ($driveType == DmsDriveType::PRIVATE) {
            $data->transform(function ($item) {
                return $item = (new PrivateFolderService)->fetchFileAccessGroup($item);
            });
        }

        return response()->json([
            'data' => $data,
        ]);
    }

    /**
     * It moves folders from one parent to another
     * @param array folders array of folders to be moved
     * @param int folderId The id of the folder where the folders will be moved.
     * @return JsonResponse An array of folders
     */
    public function moveFolder(array $folders, int $folderId): JsonResponse
    {
        DB::beginTransaction();
        try {

            foreach ($folders as $folder) {
                $data = Section::without([
                    'access',
                    'files',
                    'author',
                    'parent.access',
                    'parent.files',
                    'parent.author'
                ])->with(['parent'])->findOrFail($folder['id']);
                $oldSection = clone $data;
                $newParent = Section::without([
                    'access',
                    'files',
                    'author',
                ])->find($folderId);
                $data->parent_id = $folderId;
                $data->save();
                $data->new_parent = collect($newParent);
                $data->old_parent = collect($oldSection->parent);
                $this->logCustomMessage('folder_move', $data, 'The ' . $oldSection->name . ' folder has been moved from ' . $oldSection->parent->name . ' to ' . $newParent->name . ' by ' . Auth::user()->name, $data, FolderLogType::MOVE, new Activity());
            }

            DB::commit();
            return response()->json([
                'text'  =>  count($folders) . ' folder(s) has been moved.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    /**
     * It copies a folder and all its contents to another folder
     * @param array folders array of folders to be copied
     * @param int folderId The id of the folder where the folders will be copied.
     * @return JsonResponse A JsonResponse
     */
    public function copyFolder(array $folders, int $folderId): JsonResponse
    {
        DB::beginTransaction();
        try {
            foreach ($folders as $folder) {
                $id = $folder['id'];
                $data = Section::without([
                    'access',
                    'files',
                    'author',
                    'parent.access',
                    'parent.files',
                    'parent.author'
                ])->with(['parent'])->findOrFail($id);
                $oldSection = clone $data;
                $copy = $data->replicate();
                $copy->parent_id = $folderId;
                $newParent = Section::without([
                    'access',
                    'files',
                    'author',
                ])->find($folderId);
                $copy->save();
                $this->copy_access_folder($copy);
                $this->copy_files_of_folders($data->files, $copy);
                $this->copy_folder_recursive($data->children, $copy->id);
                $data->copy_from = collect($data->parent);
                $data->copy_to = collect($newParent);
                $this->logCustomMessage('folder_copy', $data, 'The ' . $oldSection->name . ' folder has been copied from ' . $oldSection->parent->name . ' to ' . $newParent->name . ' by ' . Auth::user()->name, $data, FolderLogType::COPY, new Activity());
            }
            DB::commit();
            return response()->json([
                'text'  =>  count($folders) . ' folders(s) has been copied.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();

            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    /**
     * It deletes a folder and all its subfolders and files
     *
     * @param array folders array of objects
     */
    public function deleteFolder(array $folders)
    {

        DB::beginTransaction();
        try {
            foreach ($folders as $folder) {
                $folder = (object) $folder;
                $deletedBy = Auth::user();
                $data = Section::withTrashed()->findOrFail($folder->id);
                $data->update(['status' => 3, 'deleted_by' => $deletedBy->id]);
                $data->files()->update(['status' => 3, 'deleted_by' => $deletedBy->id]);
                $this->deleteFolderFileVersions($data->files);
                $this->deleteSubFolder($data->children);
                $data->delete();
                $this->logCustomMessage('folder_trash', $data, 'The ' . $data->name . ' folder has been trashed by ' . $deletedBy->name, $data, FolderLogType::TRASHED, new Activity());
            }

            DB::commit();

            return response()->json([
                'text'  =>  count($folders) . ' folder(s) has been deleted.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }


    /**
     * It deletes all the versions of a file and then deletes the file itself.
     * @param stdClass children This is an array of folders that you want to delete.
     */
    public function deleteSubFolder(Collection $children)
    {
        foreach ($children as $child) {
            $child->update(['status' => 3, 'deleted_by' => Auth::user()->id]);
            $child->files()->update(['status' => 3, 'deleted_by' => Auth::user()->id]);
            $child->children()->delete();
            $this->deleteFolderFileVersions($child->files);
            $child->delete();
            if ($child->children->count() > 0) {
                $this->deleteSubFolder($child->children);
            }
        }
    }


    /**
     * It deletes all the versions of a file and then deletes the file itself
     * @param stdClass files This is an array of files that you want to delete.
     */
    public function deleteFolderFileVersions(Collection $files)
    {
        foreach ($files as $file) {
            $file->versions()->delete();
            $file->delete();
        }
    }

    /**
     * It creates a new access folder for the user who is currently logged in
     * @param Section folders This is the folder that is being copied.
     */
    public function copy_access_folder(Section $folders)
    {
        $folders->access()->create([
            'user_id' => Auth::user()->id,
            'access_level' => 1,
        ]);
    }

    /**
     * It copies files from one section to another
     * @param Collection files Collection of files
     * @param Section copy The section that is being copied
     */
    public function copy_files_of_folders(Collection $files, Section $copy)
    {
        if ($files->count() > 0) {
            $file_copies = [];
            foreach ($files as $file) {
                $file_copy = $file->replicate();
                $file_copy->section_id = $copy->id;
                array_push($file_copies, $file_copy);
            }
            $copy->files()->saveMany($file_copies);
            $this->copy_versions_of_files($file_copies, $files);
        }
    }

    /**
     * It copies the files and their versions from one section to another
     * @param file_copies array of file objects
     */
    public function copy_versions_of_files($file_copies)
    {
        foreach ($file_copies as $key => $copy) {
            if ($copy->versions->count() > 0) {
                $versions = [];
                $customForms = [];
                foreach ($copy->versions as $version) {
                    $version_copy = $version->replicate();
                    $version_copy->file_id = $copy->id;

                    $destination = Auth::user()->id . '/' . $copy->section_id . '/' . $version_copy->file_id . '/' . $version_copy->file_version . '/' . urlencode($version_copy->file_name);
                    Storage::disk(DriveType::parse($copy->type))->makeDirectory(dirname($destination));
                    Storage::disk(DriveType::parse($copy->type))->copy($version->path, $destination);

                    $version_copy->path = $destination;
                    array_push($versions, $version_copy);
                    array_push($customForms, $version['customFormValue']);
                }

                $copiedVersions = $copy->versions()->saveMany($versions);

                foreach ($copiedVersions as $copiedVersion) {
                    if (isset($copiedVersion['customFormValue'])) {
                        $customForm = $copiedVersion['customFormValue'];
                        CustomFormValue::create([
                            'custom_form_id' => $customForm['custom_form_id'],
                            'file_version_id' => $copiedVersion['id'],
                            'form_value' => $customForm['form_value']
                        ]);
                    }
                }
            }
        }
    }

    /**
     * It copies a folder and all of its subfolders and files
     * @param Collection children Collection of folders
     * @param int parent_id The id of the parent folder
     */
    public function copy_folder_recursive(Collection $children, int $parent_id)
    {
        foreach ($children as $key => $child) {
            $copy = $child->replicate();
            $copy->parent_id = $parent_id;
            $copy->save();
            $this->copy_access_folder($copy);
            $this->copy_files_of_folders($child->files, $copy);
            if ($child->children->count() > 0) {
                $this->copy_folder_recursive($child->children, $copy->id);
            }
        }
    }

    /**
     * The function `checkDuplicateFolder` checks for duplicate folders based on name, parent ID, and
     * drive type in a Section model and returns a JSON response indicating the result.
     *
     * @param request The `checkDuplicateFolder` function checks for a duplicate folder in the database
     * based on the provided parameters. The parameters are:
     *
     * @return The function `checkDuplicateFolder` is checking for a duplicate folder in the database
     * based on the provided request parameters. If a duplicate folder is found, it returns a JSON
     * response with the message 'Duplicate folder found', sets 'duplicate' to true, and includes the
     * name of the existing folder. If no duplicate folder is found, it returns a JSON response with
     * the message 'No duplicate folder found'
     */
    public function checkDuplicateFolder($request)
    {
        $currentFolder = Section::where([
            ['name', $request->name], ['parent_id', $request->parent_id], ['type', $request->drive_type]
        ])->first();

        if ($currentFolder) {
            if ($request->name !== $currentFolder->name) {
                $existingFolder = Section::where([
                    ['name', $request->name], ['parent_id', $request->parent_id], ['type', $request->drive_type]
                ])->first();
        
                if ($existingFolder) {
                    return response()->json([
                        'message' => 'Duplicate folder found.',
                        'duplicate' => true,
                        'name' => $existingFolder->name,
                    ]);
                }
            }
        }

        return response()->json([
            'message' => 'No duplicate folder found.',
            'duplicate' => false
        ]);
    }

    /**
     * The function fetches section access information for groups and users based on the provided
     * request ID.
     * 
     * @param request The `fetchHasSectionAccess` function takes a request object as a parameter. This
     * request object is used to retrieve section access information for a specific section ID.
     * 
     * @return The `fetchHasSectionAccess` function returns a JSON response containing data about
     * groups and users with access to a specific section. The data includes an array of groups and an
     * array of users who have access to the section based on the provided request ID.
     */
    public function fetchHasSectionAccess($request)
    {
        $groups = SectionAccess::where([
            ['section_id', $request->id], ['user_id', 0]
        ])->get();

        $users = SectionAccess::where([
            ['section_id', $request->id], ['group_id', 0]
        ])->get();

        return response()->json([
            'data' => [
                'groups' => $groups,
                'users' => $users
            ]
        ]);
    }

    /**
     * The function generates a unique folder name by appending a suffix and incrementing a count if a
     * folder with the same name already exists in the database.
     *
     * @param folderName The `folderName` parameter is the name of the folder for which you want to
     * generate a unique folder name.
     * @param parentId The `parentId` parameter in the `generateUniqueFolderName` function is used to
     * specify the parent ID of the folder for which a unique name needs to be generated. This ID is
     * typically used to identify the parent folder in a hierarchical structure, such as a file system
     * or a database.
     * @param driveType The `driveType` parameter in the `generateUniqueFolderName` function is used to
     * specify the type of the drive for which the unique folder name is being generated. It is used in
     * the database query to check for existing folders with the same name, parent ID, and drive type.
     * This helps
     *
     * @return The function `generateUniqueFolderName` returns a unique folder name by appending a
     * suffix ' - copy' to the original folder name and incrementing a count if a folder with the same
     * name already exists in the database for the given parent ID and drive type. The function ensures
     * that the returned folder name is unique by checking for existing folders with similar names and
     * incrementing the count if necessary.
     */
    private function generateUniqueFolderName($folderName, $parentId, $driveType)
    {
        $suffix = ' - copy';

        $count = 1;
        $newFolderName = $folderName . $suffix;
        while (Section::where([
            ['name', $newFolderName], ['parent_id', $parentId], ['type', $driveType]
        ])->exists()) {
            $newFolderName = $folderName . $suffix . ' (' . $count . ').';
            $count++;
        }

        return $newFolderName;
    }
}
