<?php

namespace Suiterus\Dms\Services\File;

use Activity;
use Carbon\Carbon;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Dms\Classes\DriveType;
use Suiterus\Dms\Models\Files\File;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Enums\Log\FileLogType;
use Suiterus\Dms\Enums\File\AccessLevel;
use Suiterus\Dms\Models\Approvals\Approval;
use Suiterus\Dms\Models\Repositories\Section;
use App\Models\AccessManagement\GroupManagement\Role;

class FileService
{

    use HasCustomLogs;

    private $fileVersion;

    /**
     * The function uploads a file and creates a new file record with various attributes, including the
     * file's section, user, name, type, document type, privacy, modified by, effectivity, and expiry,
     * and stores the file in the user's personal drive with a version number.
     * @param int section The section where the file will be uploaded to.
     * @param file The file parameter is the actual file that is being uploaded.
     * @param int type The type of the file, which is likely an integer value representing a specific
     * drive type (e.g. 1 for private, 2 for public, 3 for personal etc.).
     * @param int documentType The documentType parameter is an integer that represents the type of
     * document being uploaded. It could be a resume, a cover letter, a certificate, or any other type
     * of document. This parameter helps to categorize the uploaded files for easy retrieval and
     * management.
     * @param int privacy The privacy parameter is used to determine the visibility of the uploaded
     * file. It can have values such as confidential, or restricted. This parameter helps in controlling
     * who can access the uploaded file.
     * @return the created file object.
     */
    public function uploadFile(int $section, $file, int $type, int $documentType = null, int $privacy = null, $option)
    {
        $fileName = $file->getClientOriginalName();
        
        if ($option == 'replace') {
            $existingFile = File::where([
                ['name', $fileName], ['section_id', $section]
            ])->first();

            if ($existingFile) {
                $existingFile->update([
                    'name' => $fileName,
                    'type' => $type,
                    'document_type' => $documentType,
                    'privacy' => $privacy,
                    'updated_by' => Auth::user()->id,
                    'updated_at' => Carbon::parse(now())
                ]);
    
                $versionsCollection = collect($existingFile->versions);

                $latestVersion = $versionsCollection->sortByDesc('file_version')->first();

                $file_version_name = $latestVersion->file_version + 1 . '_' . $fileName;
                $path = $file->storeAs(
                    Auth::user()->id . '/' . $section . '/' . $existingFile->id . '/' . $latestVersion->file_version,
                    urlencode($file_version_name),
                    DriveType::parse($type)
                );

                $existingFile->versions()->create([
                    'file_name' => $file_version_name,
                    'file_version' => $latestVersion->file_version + 1,
                    'created_by' => Auth::user()->id,
                    'path' => $path
                ]);
    
                return $existingFile;
            }
        } else {
            $existingFile = File::where([
                ['name', $fileName], ['section_id', $section]
            ])->first();
            if ($existingFile) {
                $fileName = $this->generateUniqueFileName($fileName, $section); 
            }
        }
        
        $createdFile = File::create([
            'section_id' => $section,
            'user_id' => Auth::user()->id,
            'name' => $fileName,
            'type' => $type,
            'document_type' => $documentType,
            'privacy' => $privacy,
            'modified_by' => 0,
            'effectivity' => Carbon::now(),
            'expiry' => Carbon::now()->addYears(5),
        ]);
    
        // PERSONAL DRIVE
        // store to personal folder
        $version = '1';
        $file_version_name = $version . '_' . $fileName;
        $path = $file->storeAs(Auth::user()->id . '/' . $section . '/' . $createdFile->id . '/' . $version, urlencode($file_version_name), DriveType::parse($type));
        //file version
        $this->fileVersion = $createdFile->versions()->create([
            'file_name' => $file_version_name,
            'file_version' => $version,
            'created_by' => Auth::user()->id,
            'path' => $path
        ]);
    
        $createdFile->access()->create([
            'user_id' => Auth::user()->id,
            'access_level' => AccessLevel::EDITOR,
        ]);
    
        // developer access
        $devs = Role::where(['name' => 'Developer']);
        if ($devs->count()) {
            if ($createdFile->access()->where(['group_id' => $devs->first()->id])->count() == 0) {
                $createdFile->access()->create([
                    'user_id' => 0,
                    'group_id' => $devs->first()->id,
                    'access_level' => AccessLevel::EDITOR,
                    'shared_by' => Auth::id(),
                    'shared_date' => now()
                ]);
            }
        }
    
        $this->logCustomMessage(
            'file_upload',
            $createdFile,
            Auth::user()->name . ' has uploaded a file ' . $createdFile->name,
            $createdFile,
            FileLogType::UPLOAD,
            new Activity()
        );
    
        return $createdFile;
    }
    

    /**
     * The function gives file access to specified users and groups with a specified access level.
     * 
     * @param file The file object that we want to give access to.
     * @param array users An array of user IDs to whom the file access will be granted. If this
     * parameter is not provided, the file access will not be granted to any specific user.
     * @param array groups is an optional array of group IDs to which the file
     * access will be granted. The function loops through each group ID and creates a new record in the
     * `access` table with the group ID, access level, shared by user ID, and shared date. This allows
     * multiple users
     * @param int accessLevel accessLevel is an integer parameter that represents the level of access
     * granted to the users or groups for the given file. 1 = EDITOR, 2 = VIEWER
     */
    public function giveFileAccess($file, array $users = [], array $groups = [], int $accessLevel = 1)
    {
        foreach ($users as $user) {
            $file->access()->create([
                'user_id' => $user,
                'access_level' => $accessLevel,
                'shared_by' => Auth::id(),
                'shared_date' => now()
            ]);
        }

        foreach ($groups as $group) {
            $file->access()->create([
                'group_id' => $group,
                'access_level' => $accessLevel,
                'shared_by' => Auth::id(),
                'shared_date' => now()
            ]);
        }
    }

    /**
     * The function returns the version of a created file.
     * 
     * @return the value of the property ``.
     */
    public function getCreatedFileVersion()
    {
        return $this->fileVersion;
    }

    /**
     * The function `getRepositoryType` takes an integer input representing a repository type and
     * returns the corresponding string representation of that type.
     * 
     * @param type The `getRepositoryType` function takes a parameter `type` which represents the type
     * of repository. The function then checks the value of `type` using a switch statement and
     * assigns a corresponding string value based on the case. If the value of `type` does not match
     * any of the cases, it returns a JSON response with an error message stating 'Invalid access type'
     * 
     * @return The function `getRepositoryType` returns the repository type based on the input
     * parameter `type`. If the input `type` is 1, it returns 'private'; if it is 2, it returns
     * 'shared'; if it is 3, it returns 'personal'. If the input `type` does not match any of these
     * cases, it returns a JSON response with an error message stating 'Invalid access type'
     */
    public function getRepositoryType($type){
        switch ($type) {
            case 1:
                $type = 'private';
                break;
            case 2:
                $type = 'shared';
                break;
            case 3:
                $type = 'personal';
                break;
            default:
                return response()->json([
                    'status' => 'Invalid access type.',
                ], 406);
        }
        return $type;
    }

    /**
     * The function `getAccessType` in PHP converts a numeric access type to a corresponding string
     * representation.
     * 
     * @param type The `getAccessType` function takes a parameter which represents the access
     * type of a resource. The function then converts the numeric access type to a corresponding string
     * representation.
     * 
     * @return The function `getAccessType` returns the access type corresponding to the input
     * parameter ``. If the input `type` is 1, it returns 'private', if it is 2, it returns
     * 'public', and if it is 3, it returns 'personal'. If the input `type` does not match any of
     * these cases, it returns a JSON response with an error message stating 'Invalid access type.'
     */
    public function getAccessType($type)
    {
        switch ($type) {
            case 1:
                $type = 'private';
                break;
            case 2:
                $type = 'public';
                break;
            case 3:
                $type = 'personal';
                break;
            default:
                return response()->json([
                    'status' => 'Invalid access type.',
                ], 406);
        }
        return $type;
    }

    /**
     * The function getPath generates a file path based on input data and access level.
     * 
     * @param data The `data` parameter seems to be an object that contains information related to a
     * file. It has properties like `user`, `file`, and `file_version`.
     * @param access The `access` parameter in the `getPath` function seems to represent the type of
     * access level or permission required to access a specific file or resource. It is concatenated
     * with other parts of the path to generate the complete file path.
     * 
     * @return The function `getPath` is returning a string that represents a file path based on the
     * provided data and access parameters. The format of the returned path is 'dms/' followed by the
     * access parameter, the user ID from the data object, the section ID and file ID from the file
     * object within the data object, and finally the file version from the data object.
     */
    public function getPath($data, $access)
    {
        return 'dms/' . $access . '/' . $data->user->id . '/' . $data->file->section_id . '/' . $data->file->id . '/' . $data->file_version;
    }

    /**
     * The function `getBreadcrumbsPath` retrieves the breadcrumbs path for a given section ID and
     * drive type in PHP.
     * 
     * @param sectionId The `sectionId` parameter is used to specify the ID of the section for which
     * you want to retrieve the breadcrumbs path.
     * @param driveType The `driveType` parameter in the `getBreadcrumbsPath` function is used to
     * specify the type of drive for which the breadcrumbs path is being retrieved. It is likely used
     * to differentiate between different types of drives (e.g., local drive, network drive, cloud
     * drive) and customize the breadcrumbs
     * 
     * @return The `getBreadcrumbsPath` function is returning the result of calling the `breadcrumbs`
     * method on the `Section` model with the specified `sectionId` and `driveType` parameters.
     */
    public function getBreadcrumbsPath($sectionId, $driveType) 
    {
        return $this->breadcrumbs(Section::whereId($sectionId)->firstOrFail(), $driveType);
    }

    /**
     * This PHP function generates breadcrumbs based on a given set of items and a drive type.
     * 
     * @param items The `items` parameter in the `breadcrumbs` function seems to represent a
     * hierarchical structure of items, where each item has a `parent` property pointing to its parent
     * item. The function is intended to generate breadcrumbs based on this structure.
     * @param driveType The `driveType` parameter in the `breadcrumbs` function seems to be used to
     * determine the type of drive. It is passed to the `determineDriveType` method to get the name of
     * the drive type.
     * 
     * @return An array of breadcrumb items is being returned. Each item in the array contains
     * information such as id, name, and a disabled flag. The function also determines the drive type
     * based on the input parameter  and adds it as the last item in the array. The array is
     * then reversed before being returned.
     */
    public function breadcrumbs($items, $driveType)
    {
        $data = [];
        if ($items === null) {
            return $data;
        }
        $item = $items;
        $item['disabled'] = true;
        $data[] = $item;
        while ($item->parent !== null) {
            if ($item->parent === null) {
                break;
            }
            $x = $item['parent'];
            $x['disabled'] = false;
            $data[] = $x;
            $item = $item['parent'];
        }

        $data[] = [
            'id' => 0,
            'name' => $this->determineDriveType($driveType),
        ];
        return array_reverse($data);
    }

    /**
     * The function `determineDriveType` takes a drive type as input and returns the corresponding
     * drive type description.
     * 
     * @param driveType The `determineDriveType` function takes a parameter `` and returns a
     * string based on the value of the input. Here are the meanings of the drive types:
     * 
     * @return If the input parameter `` is 1, the function will return 'Private Drive'. If
     * it is 2, the function will return 'Shared Drive'. If it is 3, the function will return 'Personal
     * Drive'. If none of these cases match, the function will return nothing (null).
     */
    public function determineDriveType($driveType) {
        switch ($driveType) {
            case 1:
                return 'Private Drive';
            case 2:
                return 'Shared Drive';
            case 3:
                return 'Personal Drive';
            default:
                return;
        }
    }

    
    /**
     * The function generates a unique file name by appending a suffix and incrementing a count if a
     * file with the same name already exists in a specific section.
     * 
     * @param fileName The `fileName` parameter is the original name of the file for which you want to
     * generate a unique file name.
     * @param section The `generateUniqueFileName` function you provided generates a unique file name
     * by appending a suffix to the original file name if a file with the same name already exists in a
     * specific section.
     * 
     * @return The function `generateUniqueFileName` returns a unique file name based on the original
     * file name, section ID, and any existing files in the database with the same name and section ID.
     * The function appends a suffix "- copy" to the original file name and increments a count if a
     * file with the same name already exists in the database for the given section. The final unique
     * file name is returned.
     */
    public function generateUniqueFileName($fileName, $section)
    {
        $name = pathinfo($fileName, PATHINFO_FILENAME);
        $extension = pathinfo($fileName, PATHINFO_EXTENSION);
        
        $suffix = ' - copy';
        
        $count = 1;
        $newFileName = $name . $suffix . '.' . $extension;
        while (File::where([
            ['name', $newFileName], ['section_id', $section]
        ])->exists()) {
            $newFileName = $name . $suffix . ' (' . $count . ').' . $extension;
            $count++;
        }
        
        return $newFileName;
    }
}