<?php

namespace Suiterus\Dms\Services\File;

use Activity;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Imports\ContractsImport;
use Suiterus\Dms\Models\Repositories\Section;
use App\Models\AccessManagement\GroupManagement\Role;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Dms\Enums\File\AccessLevel;
use Suiterus\Dms\Enums\File\SectionStatus;
use Suiterus\Dms\Enums\Log\FolderLogType;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Suiterus\Dms\Models\CustomForm\CustomForm;
use Suiterus\Dms\Models\CustomForm\CustomFormValue;

class BatchUploadService
{

    use HasCustomLogs;

    private ?int $documentType = null;
    private ?int $privacy = null;
    private ?array $users = [];
    private ?array $groups = [];

    /**
     * Get the value of users
     */
    public function getUsers()
    {
        return $this->users;
    }

    /**
     * Set the value of users
     *
     * @return  self
     */
    public function setUsers($users)
    {
        $this->users = $users;

        return $this;
    }

    /**
     * Get the value of groups
     */
    public function getGroups()
    {
        return $this->groups;
    }

    /**
     * Set the value of groups
     *
     * @return  self
     */
    public function setGroups($groups)
    {
        $this->groups = $groups;

        return $this;
    }

    /**
     * Get the value of privacy
     */
    public function getPrivacy()
    {
        return $this->privacy;
    }

    /**
     * Set the value of privacy
     *
     * @return  self
     */
    public function setPrivacy($privacy)
    {
        $this->privacy = $privacy;

        return $this;
    }

    /**
     * Get the value of documentType
     */
    public function getDocumentType()
    {
        return $this->documentType;
    }

    /**
     * Set the value of documentType
     *
     * @return  self
     */
    public function setDocumentType($documentType)
    {
        $this->documentType = $documentType;

        return $this;
    }

    /**
     * Saves the file
     *
     * @param UploadedFile $file
     *
     * @return JsonResponse
     */
    public function saveFile(UploadedFile $file, String $relativePath = '', int $driveType, int $sectionId)
    {
        $this->traverseFileTree([$relativePath], [$file], $driveType, $sectionId);

        return response()->json([
            'path' => $relativePath,
            'name' => $file->getClientOriginalName(),
            'mime_type' => $file->getClientMimeType()
        ]);
    }



    /**
     * This function traverses a file tree and creates folders based on the given paths and files.
     * @param array paths An array of strings representing the paths of the folders to be traversed.
     * @param files The  parameter is an array of file objects that will be associated with the
     * folders being created in the traverseFileTree function. Each file object contains information
     * such as the file name, size, and type.
     * @param int driveType of the file, which is likely an integer value representing a specific
     * drive type (e.g. 1 for private, 2 for public, 3 for personal etc.).
     * @param int parentSectionId The ID of the parent section or folder in which the new folder or
     * file will be created.
     */
    public function traverseFileTree(array $paths, $files, int $driveType, int $parentSectionId)
    {
        $fileIndex = 0;

        foreach ($paths as $path) {
            $splittedPaths = explode('/', $path);
            $index = 0;
            $this->createFolder(
                $splittedPaths[0],
                $splittedPaths,
                basename($path),
                $files[$fileIndex],
                $driveType,
                $parentSectionId,
                $index
            );
            $fileIndex++;
        }
    }

    /**
     * This function creates a folder in a file system with a given name and path, and uploads a file
     * to it if the basename matches the file name.
     * @param string name The name of the folder to be created.
     * @param array splittedPaths An array of strings representing the path to the folder being
     * created, split by the directory separator ("/" or "\").
     * @param string basename The base name of the file being created.
     * @param file  is a file object that will be uploaded to the created folder.
     * @param int driveType of the file, which is likely an integer value representing a specific
     * drive type (e.g. 1 for private, 2 for public, 3 for personal etc.).
     * @param int parentId The ID of the parent folder where the new folder will be created. If not
     * specified, the default value is 0 which means the new folder will be created at the root level.
     * @param int index The index parameter is an integer that represents the current index of the
     * folder being created in the array of splittedPaths. It is used to keep track of the current
     * position in the array and to recursively call the createFolder function for each subfolder in
     * the path.
     * @return The function is recursively calling itself and returning the result of the recursive
     * call. The final return statement will return the result of the last recursive call.
     */
    public function createFolder(string $name, array $splittedPaths, string $basename, $file, int $driveType, int $parentId = 0, int $index = 0)
    {

        if ($basename == $name) {
            $option = 'continue';
            $fileService = new FileService();
            $uploadedFile = $fileService->uploadFile($parentId, $file, $driveType, $this->getDocumentType(), $this->getPrivacy(), $option);
            $this->createCustomForm($uploadedFile->name, $fileService->getCreatedFileVersion());
            if ($this->getUsers() || $this->getGroups()) {
                $fileService->giveFileAccess($uploadedFile, $this->getUsers(), $this->getGroups(), AccessLevel::VIEWER);
            }
            return;
        }

        $isSectionExist = Section::where([
            ['name', $name],
            ['parent_id', $parentId]
        ])->first();

        if (!$isSectionExist) {
            $section = Section::create([
                'name' => $name,
                'description' => '',
                'type' => $driveType,
                'parent_id' => $parentId,
                'status' => SectionStatus::ACTIVE,
                'created_by' => Auth::id(),
                'deleted_by' => null
            ]);

            $section->access()->create([
                'user_id' => Auth::user()->id,
                'access_level' => AccessLevel::EDITOR,
            ]);

            $devs = Role::where(['name' => 'Developer']);
            if ($devs->count()) {
                if (
                    $section
                    ->access()
                    ->where(['group_id' => $devs->first()->id])
                    ->count() == 0
                ) {
                    $section->access()->create([
                        'user_id' => 0,
                        'group_id' => $devs->first()->id,
                        'access_level' => AccessLevel::EDITOR,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }

            $this->logCustomMessage(
                'folder_upload',
                $section,
                'The ' . $section->name . ' folder has been uploaded by ' . Auth::user()->name,
                $section,
                FolderLogType::UPLOAD,
                new Activity()
            );
        }


        if (isset($section)) {
            $sectionId = $section->id;
        } else {
            $sectionId = $isSectionExist->id;
        }

        $index++;

        return $this->createFolder(
            $splittedPaths[$index],
            $splittedPaths,
            $basename,
            $file,
            $driveType,
            $sectionId,
            $index
        );
    }

    private function createCustomForm($file_name, $file_version)
    {   
        //Todo: optimize code for laoding excel file, try using filesystem function of php
        $custom_form = CustomForm::whereHas('documentType', function ($query) {
            $query->where('name', 'Contract Forms');
        })->first();
        
        $file_path = Storage::disk('contract_indexes')->path('Index File for Contracts_6-10-2023.xlsx');
        
        if (!$custom_form || !file_exists($file_path)) {
            return;
        }
        
        $excel_collection = (new ContractsImport)->toCollection($file_path);
        
        $filtered_data = $excel_collection->flatten(1)->filter(function ($value) use ($file_name) {
            return collect($value['filename'])->contains($file_name);
        })->first();
        
        if ($filtered_data) {
            $encoded_form_value = $this->encodeFormValueToJson($filtered_data);
            CustomFormValue::Create([
                'custom_form_id' => $custom_form->id,
                'file_version_id' => $file_version->id,
                'form_value' => $encoded_form_value,
            ]);
        }

        return;
    }
    
    private function encodeFormValueToJson($form_value) {
        $form_values = [
            [
                'type' => 'text',
                'label' => 'Contract no.',
                'value' => $form_value['contract_no'],
                'formID' => 0,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'Title',
                'value' => $form_value['title'],
                'formID' => 1,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'Supplier',
                'value' => $form_value['supplier'],
                'formID' => 2,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'End user',
                'value' => $form_value['enduser'],
                'formID' => 3,
                'required' => false
            ],
            [
                'type' => 'date',
                'label' => 'Effectivity date',
                'value' => $form_value['effectivity_date'],
                'formID' => 4,
                'required' => false
            ],
            [
                'type' => 'date',
                'label' => 'End of contract',
                'value' => $form_value['end_of_contract'],
                'formID' => 5,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'Category',
                'value' => $form_value['category'],
                'formID' => 6,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'Subcategory',
                'value' => $form_value['subcategory'],
                'formID' => 7,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'Remarks',
                'value' => $form_value['remarks'],
                'formID' => 8,
                'required' => false
            ],
            [
                'type' => 'text',
                'label' => 'Filename',
                'value' => $form_value['filename'],
                'formID' => 9,
                'required' => false
            ]
        ];
        return json_encode($form_values);
    }
}