<?php

namespace Suiterus\Dms\Controllers\Trashbin;

use Activity;
use Exception;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Dms\Classes\DriveType;
use Suiterus\Dms\Models\Files\File;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Models\Files\Forward;
use Illuminate\Support\Facades\Storage;
use Suiterus\Dms\Enums\Log\FileLogType;
use Suiterus\Dms\Enums\Log\FolderLogType;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Models\Files\FileAccess;
use Suiterus\Dms\Models\Approvals\Approval;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Models\Approvals\FileApprover;
use Suiterus\Dms\Models\Approvals\UserApprover;
use Suiterus\Dms\Models\Files\FileVersion as FV;
use Suiterus\Dms\Models\Signatory\FileSignatory;
use Illuminate\Support\Facades\Auth as FacadesAuth;
use Suiterus\Dms\Models\CustomForm\CustomFormValue;
use Suiterus\Dms\Models\Repositories\SectionAccess;
use Suiterus\Dms\Models\Signatory\FileSignatorySignee;
use Suiterus\Dms\Models\Signatory\FileSignatorySignature;
use Suiterus\Dms\Models\OtherAttachments\OtherAttachments;
use App\Models\AccessManagement\GroupManagement\ModelHasRole;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class TrashbinController extends Controller
{

	use HasCustomLogs;

	private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

	public function search_trash_folders(Request $req)
	{
		$paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
		$user_groups = ModelHasRole::where('model_id', FacadesAuth::user()->id)->pluck('role_id');
		$data = Section::onlyTrashed()
			->where([['name', 'LIKE', '%' . $req->keyword . '%'], ['status', 3]])
			->where(function ($query) use ($user_groups) {
				$query->whereHas('access', function ($query) use ($user_groups) {
					$query->where('user_id', FacadesAuth::user()->id)
						->orWhereIn('group_id', $user_groups);
				})
					->orWhere([['status', 3], ['type', 2]]);
			})
			->where(function ($query) {
				$query->whereHas('parent')
					->orWhere('parent_id', 0);
			})
			->with('deletedBy')->orderBy('deleted_at', 'desc')
			->orderBy('id', 'desc')->paginate($paginate);
		
		$this->logCustomMessage('search_trash_folder', null, FacadesAuth::user()->name . ' searched for "' . $req->keyword . '"', null, 'Search trash folder', new Activity());

		return response()->json([
			'data'  =>  $data
		]);
	}

	public function search_trash_files(Request $req)
	{
		$paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
		$user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
		$data = SF::onlyTrashed()
			->where([['name', 'LIKE', '%' . $req->keyword . '%'], ['status', 3]])
			->where(function ($query) use ($user_groups) {
				$query->whereHas('access', function ($query) use ($user_groups) {
					$query->where('user_id', Auth::user()->id)
						->orWhereIn('group_id', $user_groups);
				})
					->orWhere([['status', 3], ['type', 2]]);
			})
			->where(function ($query) {
				$query->whereHas('section')
					->orWhere('section_id', 0);
			})
			->with('deletedBy')->orderBy('deleted_at', 'desc')
			->orderBy('id', 'desc')->paginate($paginate);

			$this->logCustomMessage('search_trash_file', null, FacadesAuth::user()->name . ' searched for "' . $req->keyword . '"', null, 'Search trash file', new Activity());

		return response()->json([
			'data'  =>  $data
		]);
	}

	public function fetch_trash_folder(Request $req)
	{
		$paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
		$user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
		$data = Section::onlyTrashed()
			->where('status', 3)
			->where(function ($query) use ($user_groups) {
				$query->whereHas('access', function ($query) use ($user_groups) {
					$query->where('user_id', Auth::user()->id)
						->orWhereIn('group_id', $user_groups);
				})
					->orWhere([['status', 3], ['type', 2]]);
			})
			->where(function ($query) {
				$query->whereHas('parent')
					->orWhere('parent_id', 0);
			})
			->with('deletedBy')
			->orderBy('deleted_at', 'desc')
			->orderBy('id', 'desc')
			->paginate($paginate);

		return response()->json([
			'message' => 'Fetch successful.',
			'data' => $data,
		]);
	}

	public function fetch_trash_files(Request $req)
	{
		$paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
		$user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
		$data = SF::onlyTrashed()
			->where('status', 3)
			->where(function ($query) use ($user_groups) {
				$query->whereHas('access', function ($query) use ($user_groups) {
					$query->where('user_id', Auth::user()->id)
						->orWhereIn('group_id', $user_groups);
				})
					->orWhere([['status', 3], ['type', 2]]);
			})
			->where(function ($query) {
				$query->whereHas('section')
					->orWhere('section_id', 0);
			})
			->with('deletedBy')
			->orderBy('deleted_at', 'desc')
			->orderBy('id', 'desc')
			->paginate($paginate);

		return response()->json([
			'text' => 'Fetch successful.',
			'data' => $data
		]);
	}

	public function empty_trash_bin(Request $req)
	{
		DB::beginTransaction();
		try {
			SF::onlyTrashed()->forceDelete();
			Section::onlyTrashed()->forceDelete();

			return response()->json([
				'text' => 'Trash bin emptied successfully.',
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return response()->json([
				'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
				'message'   =>  $e->getMessage()
			], 500);
		}
	}

	public function delete_document(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'documents' => 'required|array',
			'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
		]);

		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}
		DB::beginTransaction();
		try {

			$this->delete_versions_of_files($req->documents);

			DB::commit();
			return response()->json([
				'text'  =>  count($req->documents) . ' file(s) has been deleted.',

			]);
		} catch (\Exception $e) {
			DB::rollback();

			return response()->json([
				'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
				'message'   =>  $e->getMessage()
			], 500);
		}
	}

	public function restore_document(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'documents' => 'required|array',
			'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
		]);

		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}

		DB::beginTransaction();
		try {
			foreach ($req->documents as $file) {
				$file = (object) $file;
				$file = SF::onlyTrashed()->without([
					'extraFields',
					'access',
					'versions',
				])->findOrFail($file->id);

				if ($file->status == 2) {
					$this->logCustomMessage('file_archive_restore', $file, 'The ' . $file->name . ' file has been restored from archive by ' . FacadesAuth::user()->name, $file, FileLogType::RESTOREARCHIVE, new Activity());
				} else {
					$this->logCustomMessage('file_trash_restore', $file, 'The ' . $file->name . ' file has been restored from trash by ' . FacadesAuth::user()->name, $file, FileLogType::RESTORETRASHBIN, new Activity());
				}

				$file->update([
					'deleted_by' => null,
					'status' => 1,
					'expiry' => Carbon::now()->addYears(5)
				]);
				$file->versions()->restore();
				$file->restore();
			}

			DB::commit();
			return response()->json([
				'text'  =>  count($req->documents) . ' file(s) has been restored.'

			]);
		} catch (\Exception $e) {
			DB::rollback();
			return response()->json([
				'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
				'message'   =>  $e->getMessage()
			], 500);
		}
	}

	public function restore_all(Request $req)
	{
		DB::beginTransaction();
		try {
			$file = SF::onlyTrashed()->get();
			$file->update([
				'deleted_by' => null,
				'status' => 1,
				'expiry' => Carbon::now()->addYears(5)
			]);
			$file->restore();

			$folder = Section::onlyTrashed()->get();
			$folder->update([
				'deleted_by' => null,
				'status' => 1,
			]);
			$folder->restore();

			FV::onlyTrashed()->restore();

			DB::commit();
			return response()->json([
				'text' => 'Files(s) and folders(s) restored successfully',
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return response()->json([
				'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
				'message'   =>  $e->getMessage()
			], 500);
		}
	}

	public function restore_folder(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'folders' => 'required|array',
			'folders.*.id' => 'required|numeric|exists:dms_db.sections,id',
		]);

		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}

		DB::beginTransaction();
		try {
			foreach ($req->folders as $folder) {
				$folder = (object) $folder;
				$data = Section::onlyTrashed()->findOrFail($folder->id);
				if ($data->status == 2) {
					$this->logCustomMessage('folder_archive_restore', $data, 'The ' . $data->name . ' folder has been restored from archive by ' . FacadesAuth::user()->name, $data, FolderLogType::RESTOREARCHIVE, new Activity());
				} else {
					$this->logCustomMessage('folder_trash_restore', $data, 'The ' . $data->name . ' folder has been restored from trash by ' . FacadesAuth::user()->name, $data, FolderLogType::RESTORETRASHBIN, new Activity());
				}
				$this->restore_versions_of_files($data->files()->onlyTrashed()->get());
				$this->restore_folder_recursive($data->children()->onlyTrashed()->get());
				$data->update(['status' => 1, 'deleted_by' => null]);
				$data->restore();
			}

			DB::commit();
			return response()->json([
				'text'  => count($req->folders) . ' folder(s) has been restored.',
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return response()->json([
				'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
				'message'   =>  $e->getMessage()
			], 500);
		}
	}
	public function restore_folder_recursive($children)
	{
		foreach ($children as $child) {
			$this->restore_versions_of_files($child->files()->onlyTrashed()->get());
			$child->update(['status' => 1, 'deleted_by' => null]);
			$child->restore();
			if ($child->children()->onlyTrashed()->get()->count() > 0) {
				$this->restore_folder_recursive($child->children()->onlyTrashed()->get());
			}
		}
	}

	public function restore_versions_of_files($files)
	{
		foreach ($files as $file) {
			$file->versions()->onlyTrashed()->restore();
			$file->update([
				'deleted_by' => null,
				'status' => 1,
				'expiry' => Carbon::now()->addYears(5)
			]);
			$file->restore();
		}
	}

	public function delete_folder(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'folders' => 'required|array',
			'folders.*.id' => 'required|numeric|exists:dms_db.sections,id',
		]);

		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}

		DB::beginTransaction();
		try {
			foreach ($req->folders as $folder) {
				$folder = (object) $folder;
				$data = Section::onlyTrashed()->findOrFail($folder->id);
				$this->delete_versions_of_files($data->files()->onlyTrashed()->get());
				$this->delete_folder_recursive($data->children()->onlyTrashed()->get());
				SectionAccess::where('section_id', $folder->id)->forceDelete();
				$data->forceDelete();
				$this->logCustomMessage('folder_delete', $data, 'The ' . $data->name . ' folder has been deleted from trashed by ' . FacadesAuth::user()->name, $data, FolderLogType::DELETE, new Activity());
			}

			DB::commit();
			return response()->json([
				'text'  =>  count($req->folders) . ' folder(s) has been deleted.',
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return response()->json([
				'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
				'message'   =>  $e->getMessage()
			], 500);
		}
	}

	public function delete_folder_recursive($children)
	{
		foreach ($children as $child) {
			$this->delete_versions_of_files($child->files()->onlyTrashed()->get());
			$child->forceDelete();
			if ($child->children()->onlyTrashed()->get()->count() > 0) {
				$this->delete_folder_recursive($child->children()->onlyTrashed()->get());
			}
		}
	}

	public function delete_versions_of_files($files)
	{
		try {
			foreach ($files as $file) {
				$file = (object) $file;
				$fileModel = File::onlyTrashed()->without([
					'extraFields',
					'access',
					'versions',
				])->where('id', $file->id)->first();

				$this->logCustomMessage('file_delete', $fileModel, 'The ' . $fileModel->name . ' file has been deleted from trashed by ' . FacadesAuth::user()->name, $fileModel, FileLogType::DELETE, new Activity());

				$fileVersionsObject = FV::onlyTrashed()->where('file_id', $file->id)->with(['file' => function ($query) {
					$query->onlyTrashed();
				}]);
				$fileVersions = $fileVersionsObject->latest()->get();

				$otherAttachments = OtherAttachments::where('file_id', $file->id)->get()->makeVisible('path');

				foreach ($fileVersions as $data) {
					if (Storage::disk(DriveType::parse($data->file->type))->has(dirname(dirname($data->path)))) {
						Storage::disk(DriveType::parse($data->file->type))->deleteDirectory(dirname(dirname($data->path)));
					}
					CustomFormValue::where('file_version_id', $data->id)->forceDelete();

					$fileSignatoryObject = FileSignatory::where('file_version_id', $data->id);

					$fileSignatories = $fileSignatoryObject->get();

					foreach ($fileSignatories as $fileSignatory) {
						FileSignatorySignee::where('file_signatory_id', $fileSignatory->id)->forceDelete();
						FileSignatorySignature::where('file_signatory_id', $fileSignatory->id)->forceDelete();
					}

					$approvalObject = Approval::where('file_version_id', $data->id);

					if ($approvalObject->first()) {
						$approvalId = $approvalObject->first()->id;

						$fileApproverObject = FileApprover::where('approval_id', $approvalId);
						$fileApproverId = $fileApproverObject->first()->id;

						UserApprover::where('file_approver_id', $fileApproverId)->forceDelete();

						$fileApproverObject->forceDelete();
						$approvalObject->forceDelete();
					}

					$fileSignatoryObject->forceDelete();
				}

				foreach ($otherAttachments as $otherAttachment) {
					$path = $otherAttachment['path'];
					if (Storage::disk('attachment')->exists(dirname($path))) {
						Storage::disk('attachment')->deleteDirectory(dirname($path));
					}
				}

				$data = SF::onlyTrashed()->findOrFail($file->id);

				Forward::where('file_id', $file->id)->forceDelete();
				FileAccess::where('file_id', $file->id)->forceDelete();
				$data->versions()->forceDelete();
				$data->otherAttachments()->forceDelete();
				$data->forceDelete();
			}
		} catch (Exception $e) {
			return response()->json([
				'error' => $e->getMessage()
			]);
		}
	}

	public function view_document(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'id' => 'required|numeric|exists:dms_db.files,id'
		]);
		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}

		$data = FV::onlyTrashed()->where('file_id', $req->id)->with(['file' => function ($query) {
			$query->onlyTrashed();
		}])->latest()->first();
		$access = $this->fileService->getAccessType($data->file->type);
		$data->path =  $this->fileService->getPath($data, $access);

		return response()->json([
			'message' => 'Fetch successful.',
			'data' => $data,
		]);
	}
	public function list_versions(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'id' => 'required|numeric|exists:dms_db.files,id'
		]);
		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}

		$data = FV::onlyTrashed()->where('file_id', $req->id)->orderBy('created_at', 'desc')->get();

		return response()->json([
			'message' => 'Fetch successful.',
			'data' => $data,
		]);
	}

	public function view_version(Request $req)
	{
		$valid = Validator::make($req->all(), [
			'id' => 'required|numeric|exists:dms_db.file_versions,id'
		]);

		if ($valid->fails()) {
			return response()->json([
				'errors' => $valid->errors(),
			], 400);
		}

		$data = FV::onlyTrashed()->whereId($req->id)->with(['file' => function ($query) {
			$query->onlyTrashed();
		}])->latest()->first();
		$access = $this->fileService->getAccessType($data->file->type);
		$data->path =$this->fileService->getPath($data, $access);

		return response()->json([
			'message' => 'Fetch successful.',
			'data' => $data,
			'access' => $access,
			'path' => $this->fileService->getPath($data, $access),
		]);
	}

	public function getFilePath($data, $access)
	{
		return 'dms/' . $access . '/' . $data->user->id . '/' . $data->file->section_id . '/' . $data->file->id;
	}
}
