<?php

namespace Suiterus\Dms\Controllers\Signatory;

use Activity;
use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Events\ApplySignature;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Dms\Classes\DriveType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Http\Classes\NotificationMessage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\FileAccess;
use Suiterus\Dms\Models\Files\FileVersion;
use Suiterus\Dms\Services\File\FileService;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Dms\Enums\Log\SignatoryLogType;
use Suiterus\Dms\Models\Signatory\FileSignatory;
use Suiterus\Dms\Models\Signatory\FileSignatorySignee;
use Suiterus\Dms\Models\Signatory\FileSignatorySignature;

class FileSignatorySigneeController extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function create(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_version_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.file_versions,id',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $file = FileVersion::findOrFail($request->file_version_id);
        $type = $this->fileService->getRepositoryType($file->file->type);
        $permission =  User::where('id', Auth::user()->id)
            ->whereHas('permissions', function ($query) use ($type) {
                $query->where('name', 'assign signatory ' . $type . ' drive');
            })->first();

        if (!$permission) {
            return response()->json([
                'errors' => ['You do not have the required authorization.'],
            ], 403);
        }
        DB::beginTransaction();
        try {
            $fileSignatory = FileSignatory::create([
                'user_id' => Auth::id(),
                'file_version_id' => $request->file_version_id,
                'is_any_order' => $request->is_any_order,
            ]);


            $fileVersion = FileVersion::without([
                'user',
                'comment',
                'forward',
                'customFormValue'
            ])->find($request->file_version_id);

            $sequence = 1;

            foreach ($request->signers as $signee) {
                FileAccess::create([
                    'user_id' => $signee['id'],
                    'file_id' => $file->file->id,
                    'group_id' => null,
                    'access_level' => 2,
                    'shared_by' => Auth::id(),
                    'shared_date' => now()
                ]);

                $fileSignatorySignee = FileSignatorySignee::create([
                    'file_signatory_id' => $fileSignatory->id,
                    'user_id' => $signee['id'],
                    'option' => json_encode($signee['options']),
                    'sequence' => $sequence,
                    'status' => $request->is_any_order || (!$request->is_any_order && $sequence == 1)  ? 2 : 1,
                ]);

                $signatoryCollection = FileSignatorySignee::find($fileSignatorySignee->id);
                $signatoryCollection->file_version = $fileVersion;

                $message = Auth::user()->name . ' assigned ' . $signee['name'] . ' to sign the file ' . $fileVersion->file_name;

                $this->logCustomMessage(
                    'file_version_sign',
                    $signatoryCollection,
                    $message,
                    $signatoryCollection,
                    SignatoryLogType::ASSIGN,
                    new Activity()
                );

                NotificationMessage::notifySenderAndReceiver('Assign Signatory', $signee['id'], Auth::id(), null, null, $fileSignatorySignee->id);

                $sequence++;
            }

            DB::commit();

            broadcast(new ApplySignature($request->file_version_id, false));

            return response()->json([
                'message' => 'Success'
            ], 200);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function fetchByFileVersion(Request $request)
    {
        return FileSignatorySignee::select('id', 'file_signatory_id', 'user_id', 'sequence', 'status', 'created_at')->whereHas('fileSignatory', function ($query) use ($request) {
            $query->where('file_version_id', $request->file_version_id);
        })->orderBy('sequence')->paginate(3);
    }

    public function fetchByAuthUser(Request $request)
    {
        $fileSignatorySignee =  FileSignatorySignee::where([
            ['user_id', Auth::id()],
        ])->whereHas('fileSignatory', function ($query) use ($request) {
            $query->where('file_version_id', $request->file_version_id);
        })->first();

        if (!$fileSignatorySignee) {
            return;
        }

        $fileSignatorySignee->option = json_decode($fileSignatorySignee->option);

        return $fileSignatorySignee;
    }

    public function fetchSigneeByID(Request $request)
    {
        $fileSignatorySignee =  FileSignatorySignee::where([
            ['id', $request->id],
        ])->with(['fileSignatory.fileVersion.file'])->first();

        if (!$fileSignatorySignee) {
            return;
        }

        $fileSignatorySignee->option = json_decode($fileSignatorySignee->option);

        return $fileSignatorySignee;
    }

    public function applySignature(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_path' => 'required',
            'encrypted_signature' => 'required',
            'signature' => 'required',
            'file_version_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.file_versions,id',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            broadcast(new ApplySignature($request->file_version_id, true));

            $fileVersion = FileVersion::where('id', $request->file_version_id)->without([
                'user',
                'comment',
                'forward',
                'customFormValue'
            ])->first();
            $file = $fileVersion->file;
            $filePath = dirname($fileVersion->path) . '/with-signature';
            $tmpFilePath = uniqid();

            if (Storage::disk(DriveType::parse($file->type))->exists($filePath)) {
                Storage::disk(DriveType::parse($file->type))->deleteDirectory($filePath);
            }

            $encryptedSignaturePath = Storage::disk('tmp_encrypt_document')->put($tmpFilePath, $request->encrypted_signature);
            $signaturePath = Storage::disk('tmp_encrypt_document')->put($tmpFilePath, $request->signature);

            $fileSignatory = FileSignatory::where('file_version_id', $request->file_version_id);
            $fileSignatoryID = $fileSignatory->first()->id;

            if (!($encryptedSignaturePath && $signaturePath && $fileSignatory)) {
                return response()->json([
                    'message' => 'An error occurred while signing the document.'
                ], 400);
            }

            FileVault::disk('tmp_encrypt_document')->encrypt($encryptedSignaturePath);
            FileVault::disk('tmp_encrypt_document')->encrypt($signaturePath);

            $encryptedDocument = Storage::disk('tmp_encrypt_document')->get($encryptedSignaturePath . '.enc');
            $encryptedSignedDocument = Storage::disk('tmp_encrypt_document')->get($signaturePath . '.enc');

            $newEncryptedSignaturePath = $filePath . '/' . basename($encryptedSignaturePath);
            $newSignaturePath = $filePath . '/' . basename($signaturePath);

            Storage::disk(DriveType::parse($file->type))->put($newEncryptedSignaturePath . '.enc', $encryptedDocument);
            Storage::disk(DriveType::parse($file->type))->put($newSignaturePath . '.enc', $encryptedSignedDocument);

            Storage::disk('tmp_encrypt_document')->deleteDirectory(dirname($encryptedSignaturePath));

            $signee = FileSignatorySignee::where([
                ['user_id', Auth::id()],
                ['file_signatory_id', $fileSignatoryID]
            ]);

            if ($fileSignatory->first()->is_any_order == 0) {

                $signee->update([
                    'status' => 3,
                ]);

                FileSignatorySignee::where([
                    ['sequence', $signee->first()->sequence + 1],
                    ['file_signatory_id', $fileSignatoryID]
                ])->update([
                    'status' => 2,
                ]);
            } else {
                $signee->update([
                    'status' => 3,
                ]);
            }

            $totalSignee = FileSignatorySignee::where([
                ['file_signatory_id', $fileSignatoryID]
            ])->count();

            $totalSigned = FileSignatorySignee::where([
                ['file_signatory_id', $fileSignatoryID],
                ['status', 3]
            ])->count();

            FileSignatorySignature::updateOrCreate(
                [
                    'file_signatory_id' => $fileSignatoryID
                ],
                [
                    'file_signatory_id' => $fileSignatoryID,
                    'document_encrypted_path' => $newEncryptedSignaturePath,
                    'document_signature_path' => $newSignaturePath,
                ]
            );

            NotificationMessage::notifySenderAndReceiver('Signed a Document', $fileSignatory->first()->user_id, Auth::id(), null, null, $signee->first()->id);

            if ($totalSignee == $totalSigned) {
                $fileSignatory->update([
                    'status' => 2
                ]);

                NotificationMessage::notifyReceiver('Document Signed Completed', $fileSignatory->first()->user_id, Auth::id(), null, null, $signee->first()->id);
            }

            broadcast(new ApplySignature($request->file_version_id, false));
            DB::commit();

            $signatoryCollection = FileSignatorySignee::where([
                ['user_id', Auth::id()],
                ['file_signatory_id', $fileSignatoryID]
            ])->first();

            $signatoryCollection->file_version = $fileVersion;

            $message = Auth::user()->name . ' signed the file ' . $fileVersion->file_name;

            $this->logCustomMessage(
                'file_version_signed',
                $signatoryCollection,
                $message,
                $signatoryCollection,
                SignatoryLogType::SIGN,
                new Activity()
            );
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function fetchSignatory()
    {
        return User::where('id', '!=', Auth::id())->get();
    }
}
