<?php

namespace Suiterus\Dms\Controllers\SectionManager;

use Activity;
use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Models\Files\Forward;
use Suiterus\Dms\Enums\Log\FileLogType;
use App\Http\Classes\NotificationMessage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Enums\Log\ForwardLogType;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Models\NotificationMessages;
use Suiterus\Dms\Models\Files\FileVersion as FV;
use Illuminate\Support\Facades\Auth as FacadesAuth;
use Suiterus\Dms\Controllers\NotificationMessagesController;
use App\Models\AccessManagement\GroupManagement\Role as GroupManagementRole;

class ForwardController extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function list_internal_user(Request $request)
    {
        $data = User::whereHas('employeeMetaInfo')
            ->with(['roles' => function ($query) {
                $query->whereNotIn('name', ['Super Admin', 'Admin', 'Developer']);
            }])
            ->get();

        return response()->json([
            'data' => $data
        ]);
    }

    public function list_internal_group(Request $request)
    {
        $data = Role::whereNotIn('name', ['Super Admin', 'Admin', 'Developer'])->get();

        return response()->json([
            'data' => $data
        ]);
    }

    public function forward_document(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|integer',
            'subject' => 'required|string',
            'message' => 'required|string',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $file = SF::findOrFail($request->id);

            $type = $this->fileService->getRepositoryType($file->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name', 'forward document ' . $type . ' drive');
                })->first();

            if (!$permission) {
                return response()->json([
                    'errors' => ['You do not have the required authorization.'],
                ], 403);
            }

            $groups = collect($request->groups);
            $members = User::whereHas('roles', function ($query) use ($groups) {
                $query->whereIn('name', $groups->pluck('name'));
            })->get();

            foreach ($groups->pluck('id') as  $group) {
                $file->access()->firstOrCreate([
                    'user_id' => 0,
                    'group_id' => $group,
                    'access_level' => 1,
                    'shared_by' => Auth::id(),
                    'shared_date' => now()
                ]);


                $members = User::whereHas('roles', function ($query) use ($groups) {
                    $query->whereIn('name', $groups->pluck('name'));
                })->get();

                foreach ($members as $member) {
                    $record = [
                        'file_id'    => $file->id,
                        'user_id'    => $member['id'],
                        'subject'    => $request->subject,
                        'message'    => $request->message,
                        'created_by' => Auth::user()->id,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ];
                    $forward = Forward::create($record);
                    $forwardCollection = Forward::find($forward->id);
                    NotificationMessage::notifySenderAndReceiver('Forward', $member['id'], Auth::user()->id, null, $forward->id);

                    $this->logCustomMessage(
                        'file_forward',
                        $forwardCollection,
                        FacadesAuth::user()->name . ' has forwarded a file ' . $file->name . ' to ' . $member['name'],
                        $forwardCollection,
                        ForwardLogType::FORWARD,
                        new Activity()
                    );
                }
            }

            $users = [];
            foreach ($request->users as $user) {
                $user = (object) $user;
                $userCollection = User::without([
                    'roles',
                    'permissions',
                    'storage',
                    'employeeMetaInfo',
                    'supervisor',
                    'exitInterview',
                    'userProfilePicture',
                    'profileBasicInfo'
                ])->find($user->id);
                $record = [
                    'file_id'    => $file->id,
                    'user_id'    => $user->id,
                    'subject'    => $request->subject,
                    'message'    => $request->message,
                    'created_by' => Auth::user()->id,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ];
                $forward = Forward::create($record);
                $forwardCollection = Forward::find($forward->id);
                NotificationMessage::notifySenderAndReceiver('Forward', $user->id, Auth::user()->id, null, $forward->id);
                array_push($users, $record);
                $this->logCustomMessage(
                    'file_forward',
                    $forwardCollection,
                    FacadesAuth::user()->name . ' has forwarded a file ' . $file->name . ' to ' . $userCollection->name,
                    $forwardCollection,
                    ForwardLogType::FORWARD,
                    new Activity()
                );
            }

            return response()->json([
                'text' => 'File forwarded successfully to ' . count($members) . ' user(s).'
            ]);
        } catch (Exception $ex) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
            ], 500);
        }
    }

    public function view_forwarded(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_id' => 'required|numeric|exists:dms_db.forwards,file_id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ]);
        }
        return response()->json([
            'data' => Forward::where('file_id', $request->file_id)
                ->where('user_id', Auth::user()->id)
                ->orWhere('created_by', Auth::user()->id)
                ->first()
        ]);
    }

    public function forward_history(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');

        $data = Forward::where('file_id', $request->id)->orderBy('created_at', 'desc')->with('file')->paginate($paginate);

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data,
        ]);
    }

    public function update_status(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'forward_id' => 'required|numeric|exists:dms_db.forwards,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $data = Forward::where('id', $req->forward_id)->first();

        if ($data->status != 1 && $data->user_id == Auth::id()) {
            $data->update([
                'status' => 1,
                'updated_at' => Carbon::now(),
            ]);
            $message = FacadesAuth::user()->name .
                ' has viewed the file ' .
                $data->file->name .
                ' forwarded by ' .
                $data->sender->name;

            $this->logCustomMessage(
                'file_forward_view',
                $data,
                $message,
                $data,
                ForwardLogType::VIEW,
                new Activity()
            );
            NotificationMessage::notifyReceiver('Viewed', $data->created_by, $data->user_id, null);
        }

        return response()->json([
            'message' => 'Update successful.',
            'data' => $data,
        ]);
    }
}
