<?php

namespace Suiterus\Dms\Controllers\SectionManager;

use Activity;
use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Dms\Classes\DriveType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Models\Files\FileType;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Models\Files\FileVersion;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Enums\Log\FileVersionLogType;
use Illuminate\Support\Facades\Auth as FacadesAuth;
use Suiterus\Dms\Models\CustomForm\CustomFormValue;
use Suiterus\Dms\Controllers\Configuration\AllowedFileTypesController;

class FileVersionController extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function allowed_type()
    {
        $allowed = FileType::where('status', '=', '1')->pluck('file_ext');
        $allowed = str_replace('"', "", $allowed);
        $allowed = str_replace(array('[', ']'), '', $allowed);
        $allowed = $allowed . ',' . strtolower($allowed);

        return $allowed;
    }

    public function file_version_upload(Request $request)
    {
        $allowedTypes = AllowedFileTypesController::getType();;
        $valid = Validator::make($request->all(), [
            'file' => 'required|max:' . env('MAX_FILE_UPLOAD_SIZE') . '|mimes:' . $allowedTypes
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $file = $request->file('file');
            $file_name = $file->getClientOriginalName();

            $file_data = SF::where('id', $request->id)->first();

            $type = $this->fileService->getRepositoryType($file_data->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name', 'upload new version ' . $type . ' drive');
                })->first();

            if (!$permission) {
                return response()->json([
                    'errors' => ['You do not have the required authorization.'],
                ], 403);
            }

            if ($file_data) {
                $version = FileVersion::where('file_id', $file_data->id)->latest()->first();
                $versionCustomFormValue = CustomFormValue::where('file_version_id', $version->id)->latest()->first();
                $new_version = $version['file_version'] + 1;
                $file_version_name = $new_version . '_' . $file_name;
                $access = $this->fileService->getAccessType($file_data->type);
                $path = $file->storeAs($file_data->user_id . '/' . $file_data['section_id'] . '/' . $file_data->id . '/' . $new_version,  urlencode($file_version_name), DriveType::parse($file_data->type));

                $data = FileVersion::create([
                    'file_id' =>  $file_data->id,
                    'file_name' => $file_version_name,
                    'file_version' => $new_version,
                    'created_by' => Auth::user()->id,
                    'path' => $path
                ]);

                if ($versionCustomFormValue) {
                    CustomFormValue::create([
                        'custom_form_id' => $versionCustomFormValue->custom_form_id,
                        'file_version_id' => $data->id,
                        'form_value' => $versionCustomFormValue->form_value,
                    ]);
                }

                $old_file_data = clone $file_data;
                $file_data->update(['name' => $file_name]);

                $fileVersionCollection = FileVersion::with(['file'])->without([
                    'file.extraFields',
                    'file.access',
                    'file.versions',
                ])->find($data->id);

                $fileVersionCollection->old = $old_file_data;
                $fileVersionCollection->attributes = $file_data;

                $message = Auth::user()->name . ' has uploaded a file version ' . $file_name . ' to ' . $old_file_data->name;

                $this->logCustomMessage(
                    'file_version_upload',
                    $fileVersionCollection,
                    $message,
                    $fileVersionCollection,
                    FileVersionLogType::UPLOAD,
                    new Activity()
                );
            } else {
                return response()->json([
                    'errors' => ['File name not found.'],
                ], 400);
            }

            DB::commit();
            return response()->json([
                'text' => 'New file version uploaded successfully.',
                'data' => $data
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function view_version(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|numeric|exists:dms_db.file_versions,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $data = FileVersion::where('id', $request->id)->withTrashed()->first();
        $access = $this->fileService->getAccessType($data->file->type);
        $data->path = $this->fileService->getPath($data, $access);
        $data->breadcrumbs = $this->fileService->getBreadcrumbsPath($data->file->section_id, $data->file->type);

        $this->logCustomMessage('view_file_version', $data, FacadesAuth::user()->name . ' viewed file version "' . $data->file_name . '"', $data, 'View a file version', new Activity());

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data
        ]);
    }

    public function version_list(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|numeric|exists:dms_db.files,id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $data = FileVersion::where('file_id', $request->id)->orderBy('created_at', 'desc')->get();

        $file = SF::findOrFail($request->id);
        $type = $this->fileService->getRepositoryType($file->type);
        $permission =  User::where('id', Auth::user()->id)
            ->whereHas('permissions', function ($query) use ($type) {
                $query->where('name', 'list file version ' . $type . ' drive');
            })->first();

        if (!$permission) {
            return response()->json([
                'errors' => ['You do not have the required authorization.'],
            ], 403);
        }

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data
        ]);
    }
}
