<?php

namespace Suiterus\Dms\Controllers;

use Activity;
use App\Enums\Dms\DriveType as DmsDriveType;
use Exception;
use Suiterus\Dms\Classes\UserAccess;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Models\Files\FileAccess;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Models\Files\FileVersion as FV;
use Suiterus\Dms\Models\Signatory\FileSignatory;
use App\Models\AccessManagement\GroupManagement\Role;
use Suiterus\Dms\Models\CustomForm\CustomFormValue as CFV;
use App\Models\AccessManagement\GroupManagement\ModelHasRole;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Illuminate\Support\Facades\File;
use Iman\Streamer\VideoStreamer;
use NcJoes\OfficeConverter\OfficeConverter;
use Suiterus\Dms\Classes\AllowedFileType;
use Suiterus\Dms\Classes\DriveType;
use Suiterus\Dms\Controllers\Configuration\AllowedFileTypesController;
use Suiterus\Dms\Enums\File\AccessLevel;
use Suiterus\Dms\Enums\Log\FileLogType;
use Suiterus\Dms\Enums\Log\FileVersionLogType;
use Suiterus\Dms\Models\CustomForm\CustomFormValue;
use Suiterus\Dms\Models\Files\FileVersion;
use Suiterus\Dms\Models\Signatory\FileSignatorySignature;
use Suiterus\Dms\Request\File\FileUploadRequest;
use Suiterus\Dms\Services\File\FileService;

class FileController extends Controller
{
    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function file_upload(FileUploadRequest $req)
    {
        DB::connection(env('DMS_DB_CONNECTION'))->beginTransaction();
        try {
            $file = $req->file('files');

            $file = $this->fileService->uploadFile(
                $req->section,
                $file,
                $req->type,
                $req->documentType,
                $req->privacy,
                $req->option,
            );

            $this->fileService->giveFileAccess($file, $req->users ?? [], $req->groups ?? [], AccessLevel::VIEWER);

            DB::connection(env('DMS_DB_CONNECTION'))->commit();
            return response()->json([
                'text' =>  $req->file('files')->getClientOriginalName() . ' have been successfully uploaded.',
                'data' => $req->all(),
            ]);
        } catch (Exception $e) {
            DB::connection(env('DMS_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function move_file(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric|exists:dms_db.sections,id',
            'documents' => 'required|array',
            'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {

            foreach ($req->documents as $key => $document) {
                $data = SF::findOrFail($document['id']);
                $oldFile = clone $data;
                foreach ($data->versions as $version) {
                    $fileVersion = FileVersion::where('id', $version['id']);
                    $fileSignatory = FileSignatory::where('file_version_id', $version['id'])->first();

                    $destination = Auth::id() . '/' . $req->id . '/' . $data->id . '/' . $version['file_version'] . '/' . urlencode($version['file_name']);

                    Storage::disk(DriveType::parse($req->drive_type))->makeDirectory(dirname($destination));

                    // also move the with-signature folder
                    Storage::disk(DriveType::parse($req->drive_type))->move($version->path, $destination);

                    if ($fileSignatory) {
                        Storage::disk(DriveType::parse($req->drive_type))->move(dirname($version->path) . '/' . 'with-signature',  dirname($destination) . '/' . 'with-signature');
                        $fileSignatoryID = $fileSignatory->id;
                        $fileSignature = FileSignatorySignature::where('file_signatory_id', $fileSignatoryID);
                        $fileSignatureObject = $fileSignature->first();
                        $fileSignature->updateOrCreate(
                            [
                                'file_signatory_id' => $fileSignatoryID
                            ],
                            [
                                'document_encrypted_path' =>  dirname($destination) . '/' . 'with-signature' . '/' .  basename($fileSignatureObject->document_encrypted_path),
                                'document_signature_path' => dirname($destination) . '/' . 'with-signature' . '/' .  basename($fileSignatureObject->document_signature_path)
                            ]
                        );
                    }

                    Storage::disk(DriveType::parse($req->drive_type))->deleteDirectory(dirname($version->path));

                    $fileVersion->update([
                        'path' => $destination
                    ]);
                }
                $newParent = Section::without([
                    'access',
                    'files',
                    'author',
                ])->find($req->id);
                $data->section_id = $req->id;
                $data->save();
                $data->new_parent = collect($newParent);
                $data->old_parent = collect($oldFile->section);
                $message = 'The ' . $oldFile->name . ' file has been moved from ' . $oldFile->section->name . ' to ' . $newParent->name . ' by ' . Auth::user()->name;
                $this->logCustomMessage('file_move', $data, $message, $data, FileLogType::MOVE, new Activity());
            }

            DB::commit();
            return response()->json([
                'text'  =>  count($req->documents) . ' files(s) has been moved.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function copy_file(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric|exists:dms_db.sections,id',
            'documents' => 'required|array',
            'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($req->documents as $key => $document) {
                $data = SF::with(['versions.customFormValue'])->findOrFail($document['id']);
                $oldFile = clone $data;
                $copy = $data->replicate();
                $copy->section_id = $req->id;
                $copy->save();

                $newParent = Section::without([
                    'access',
                    'files',
                    'author',
                ])->find($req->id);

                if ($data->versions->count() > 0) {
                    $versions = [];
                    $customForms = [];
                    foreach ($data->versions as $version) {
                        $version_copy = $version->replicate();
                        $version_copy->file_id = $copy->id;

                        $destination = Auth::user()->id . '/' . $copy->section_id . '/' . $version_copy->file_id . '/' . $version_copy->file_version . '/' . urlencode($version_copy->file_name);
                        Storage::disk(DriveType::parse($data->type))->makeDirectory(dirname($destination));
                        Storage::disk(DriveType::parse($data->type))->copy($version->path, $destination);

                        $version_copy->path = $destination;
                        array_push($versions, $version_copy);
                        array_push($customForms, $version['customFormValue']);
                    }

                    $copiedVersions = $copy->versions()->saveMany($versions);

                    foreach ($copiedVersions as $copiedVersion) {
                        if (isset($copiedVersion['customFormValue'])) {
                            $customForm = $copiedVersion['customFormValue'];
                            CustomFormValue::create([
                                'custom_form_id' => $customForm['custom_form_id'],
                                'file_version_id' => $copiedVersion['id'],
                                'form_value' => $customForm['form_value']
                            ]);
                        }
                    }
                }

                $data->new_parent = collect($newParent);
                $data->old_parent = collect($oldFile->section);
                $message = 'The ' . $oldFile->name . ' file has been copied from ' . $oldFile->section->name . ' to ' . $newParent->name . ' by ' . Auth::user()->name;
                $this->logCustomMessage('file_copy', $data, $message, $data, FileLogType::COPY, new Activity());
            }
            DB::commit();
            return response()->json([
                'text'  =>  count($req->documents) . ' files(s) has been copied.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function deleteDocument(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'documents' => 'required|array',
            'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }
        DB::beginTransaction();
        try {
            foreach ($req->documents as $file) {
                $file = (object) $file;
                $data = SF::findOrFail($file->id);
                $data->update(['status' => 3, 'deleted_by' => Auth::user()->id]);
                $data->versions()->delete();
                $data->delete();
                $this->logCustomMessage('file_trash', $data, 'The ' . $data->name . ' file has been trashed by ' . Auth::user()->name, $data, FileLogType::TRASHED, new Activity());
            }

            DB::commit();
            return response()->json([
                'text'  =>  count($req->documents) . ' file(s) has been deleted.',

            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function view_document(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric|exists:dms_db.files,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $data = FV::where('file_id', $req->id)->with(['file' => function ($query) {
            $query->withTrashed();
        }])->withTrashed()->latest()->first();
        $access = $this->fileService->getAccessType($data->file->type);
        $data->path = $this->fileService->getPath($data, $access);
        $data->breadcrumbs = $this->fileService->getBreadcrumbsPath($data->file->section_id, $data->file->type);

        $this->logCustomMessage(
            'view_file',
            $data ?? null,
            Auth::user()->name . ' viewed file ' . $data->file->name,
            $data ?? null,
            'View a file',
            new Activity()
        );

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data,
        ]);
    }

    public function file_approval_info(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric|exists:dms_db.file_versions,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $data = FV::where('id', $req->id)->with(['file' => function ($query) {
            $query->withTrashed();
        }])->withTrashed()->latest()->first();

        $access = $this->fileService->getAccessType($data->file->type);
        $data->path = $this->fileService->getPath($data, $access);
        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data,
        ]);
    }

    public function file_download(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'path' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $fileSignatory = FileSignatory::where('file_version_id', $req->file_version_id)->first();

        $file = FV::withTrashed()->findOrFail($req->file_version_id);

        $access = $this->fileService->getAccessType($file->file->type);
        $path = $this->fileService->getPath($file, $access);

        $mimeType = Storage::disk(DriveType::parse($file->file->type))->mimeType($file->path);
        if (!($fileSignatory && $fileSignatory->fileSignatorySignature)) {
            switch ($mimeType) {
                    // docx
                case 'application/vnd.openxmlformats-officedocument.wordprocessingml.document':
                case 'application/msword':
                case 'application/doc':
                case 'application/ms-doc':
                    // xlsx
                case 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet':
                case 'application/excel':
                case 'application/vnd.ms-excel':
                case 'application/x-excel':
                case 'application/x-msexcel':
                    // ppt
                case 'application/vnd.openxmlformats-officedocument.presentationml.presentation':
                case 'application/mspowerpoint':
                case 'application/powerpoint':
                case 'application/vnd.ms-powerpoint':
                case 'application/x-mspowerpoint':
                    $ftpFile = Storage::disk(DriveType::parse($file->file->type))->get($file->path);

                    $folderUniqid = uniqid();

                    $temporaryPath = $folderUniqid . '/' . urlencode($file->file_name);

                    Storage::disk('converted_office')->put($temporaryPath, $ftpFile);

                    $converter = new OfficeConverter(Storage::disk('converted_office')->path($temporaryPath), null, env('LIBRE_OFFICE_COMMAND'), false);

                    $uniqid = uniqid();

                    $convertedPath = $converter->convertTo($uniqid . '.pdf'); //generates pdf file in same directory as test-file.docx
                    // $converter->convertTo('output-file.html'); //generates html file in same directory as test-file.docx

                    Storage::disk('converted_office')->delete($temporaryPath);
                    return response()->download($convertedPath)->deleteFileAfterSend(true);
                    break;
                case 'image/tiff':
                case 'image/tif':
                    $directoryName = dirname($file->path);
                    $uuid = uniqid();
                    $filePath = Storage::disk(DriveType::parse($file->file->type))->path($file->path);
                    $convertedFilePath = Storage::disk(DriveType::parse($file->file->type))->path($directoryName . '/' . $uuid . '.pdf');
                    @exec(env('IMAGICK_COMMAND') . ' ' . $filePath . ' ' . $convertedFilePath);

                    return response()->download($convertedFilePath)->deleteFileAfterSend(true);
                    break;
                default:
                    return Storage::disk(DriveType::parse($file->file->type))->download($file->path);
                    break;
            }
        }

        $path = $fileSignatory->fileSignatorySignature->document_encrypted_path . '.enc';

        if (!Storage::disk(DriveType::parse($file->file->type))->has($path)) {
            abort(404);
        }

        $newPath = basename($path);

        Storage::disk('tmp_encrypt_document')->put($newPath, Storage::disk(DriveType::parse($file->file->type))->get($path));

        FileVault::disk('tmp_encrypt_document')->decrypt($newPath);

        return response()->download(Storage::disk('tmp_encrypt_document')->path(basename($fileSignatory->fileSignatorySignature->document_encrypted_path)))->deleteFileAfterSend(true);
    }

    public function playVideo(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'path' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $file = FV::findOrFail($req->file_version_id);

        $hasAccess = UserAccess::traverseFileTreeAccess($file->file_id, Auth::user()->roles->pluck('id'));

        if (!$hasAccess && $file->file->type != DmsDriveType::PUBLIC) {
            return abort(403);
        }

        $access = $this->fileService->getAccessType($file->file->type);
        $path = $this->fileService->getPath($file, $access);
        VideoStreamer::streamFile(Storage::disk(DriveType::parse($file->file->type))->path($file->path));
    }

    public function file_download_encrypted(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'file_version_id' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }
        $file = FV::without([
            'user',
            'comment',
            'forward',
            'customFormValue'
        ])->findOrFail($req->file_version_id);
        $type = $this->fileService->getRepositoryType($file->file->type);
        $permission =  User::where('id', Auth::user()->id)
            ->whereHas('permissions', function ($query) use ($type) {
                $query->where('name', 'download document ' . $type . ' drive');
            })->first();

        if (!$permission) {
            return response()->json([
                'errors' => ['You do not have the required authorization.'],
            ], 403);
        }

        $fileSignatory = FileSignatory::where('file_version_id', $req->file_version_id)->first();

        if ($fileSignatory && $fileSignatory->fileSignatorySignature) {
            $path = $fileSignatory->fileSignatorySignature->document_signature_path;
            $path = $path . '.enc';
        }

        if (isset($path)) {
            if (!Storage::disk(DriveType::parse($file->file->type))->has($path)) {
                abort(404);
            }

            $newPath = basename($path);

            Storage::disk('tmp_encrypt_document')->put($newPath, Storage::disk(DriveType::parse($file->file->type))->get($path));

            FileVault::disk('tmp_encrypt_document')->decrypt($newPath);

            if ($req->loggable) {
                $message = Auth::user()->name . ' has downloaded the file ' . $file->file_name;

                $this->logCustomMessage(
                    'file_version_download',
                    $file,
                    $message,
                    $file,
                    FileVersionLogType::DOWNLOAD,
                    new Activity()
                );
            }

            return response()->download(Storage::disk('tmp_encrypt_document')->path(basename($fileSignatory->fileSignatorySignature->document_signature_path)))->deleteFileAfterSend(true);
        }

        if ($req->loggable) {
            $message = Auth::user()->name . ' has downloaded the file ' . $file->file_name;

            $this->logCustomMessage(
                'file_version_download',
                $file,
                $message,
                $file,
                FileVersionLogType::DOWNLOAD,
                new Activity()
            );
        }

        return Storage::disk(DriveType::parse($file->file->type))->download($file->path);
    }

    public function update_document_version(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id'        => 'required|exists:dms_db.file_versions,id',
            'file_name' => 'required|min:2|max:255|string',
        ], [
            'id.exists' => 'File not found.'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {

            $file = FV::findOrFail($req->id);
            $latest_file_version = FV::where('file_id', $file->file_id)->latest()->first();
            $oldFileVersion = clone $latest_file_version;
            $main_file = SF::where('id', $file->file_id)->first();
            $access = $this->fileService->getAccessType($file->file->type);
            $path = $this->fileService->getPath($file, $access);

            $type = $this->fileService->getRepositoryType($file->file->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name', 'update document ' . $type . ' drive');
                })->first();

            if (!$permission) {
                return response()->json([
                    'errors' => ['You do not have the required authorization.'],
                ], 403);
            }

            if ($file->file_name != $req->file_name) {
                $newPath = dirname($file->path) . urlencode($req->file_name);
                Storage::disk(DriveType::parse($main_file->type))->move($file->path, $newPath);
            }

            if ($latest_file_version->file_version == $file->file_version) {
                $main_file->update([
                    'name' => $req->file_name
                ]);
            }

            $file->update([
                'file_name' => $req->file_name,
                'path' => $newPath ?? $file->path
            ]);

            $main_file->update([
                'modified_by' => Auth::user()->id
            ]);

            CFV::where('file_version_id', $req->id)->update([
                'form_value' => $req->custom_forms,
            ]);

            $file->attributes = collect($file);
            $file->old = collect($oldFileVersion);


            $this->logCustomMessage('file_update', $file, 'The ' . $oldFileVersion->file_name . ' file has been updated by ' . Auth::user()->name, $file, FileLogType::UPDATE, new Activity());

            DB::commit();
            return response()->json([
                'text' => 'File Updated!',
                'data' => $file
            ]);
        } catch (Exception $ex) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
            ], 500);
        }
    }

    public function update_document(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id'        => 'required|exists:dms_db.files,id',
            'file_name' => 'required|min:2|max:255|string',
        ], [
            'id.exists' => 'File not found.'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $fileName = $req->file_name;

            if ($req->option == 'continue') {
                $existingFile = SF::where([
                    ['name', $fileName], ['section_id', $req->section_id]
                ])->first();
                if ($existingFile) {
                    $fileName = $this->fileService->generateUniqueFileName($fileName, $req->section_id);
                }
            }
            $file = SF::without([
                'author.roles',
                'author.permissions',
                'author.storage',
                'author.employeeMetaInfo',
                'author.supervisor',
                'author.exitInterview',
                'author.userProfilePicture',
                'author.profileBasicInfo',
                'modifier',
                'extraFields',
                'access',
                'versions',
                'documentType'
            ])->findOrFail($req->id);

            $oldFile = clone $file;
            $latest_file_version = FV::where('file_id', $file->id)->latest()->first();

            $access = $this->fileService->getAccessType($latest_file_version->file->type);
            $path = $this->fileService->getPath($latest_file_version, $access);

            $type = $this->fileService->getRepositoryType($file->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name', 'update document ' . $type . ' drive');
                })->first();

            if (!$permission) {
                return response()->json([
                    'errors' => ['You do not have the required authorization.'],
                ], 403);
            }

            $newFileName = $latest_file_version->file_version . '_' . $fileName;

            $newPath = dirname($latest_file_version->path) . '/' . $newFileName;

            Storage::disk(DriveType::parse($file->type))->move($latest_file_version->path, $newPath);

            $latest_file_version->update([
                'file_name' => $newFileName,
                'path' => $newPath
            ]);

            $file->update([
                'name' => $fileName,
                'modified_by' => Auth::user()->id,
            ]);

            DB::commit();
            $file->attributes = collect($file);
            $file->old = collect($oldFile);
            $this->logCustomMessage('file_rename', $file, 'The ' . $oldFile->name . ' file has been renamed to ' . $fileName . ' by ' . Auth::user()->name, $file, FileLogType::RENAME, new Activity());
            return response()->json([
                'text' => 'File Updated!',
                'data' => $file
            ]);
        } catch (Exception $ex) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
            ], 500);
        }
    }

    public function trash_document(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|exists:dms_db.files,id',
        ], [
            'id.exists' => 'File not found.'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $file = SF::findOrFail($req->id);

            $type = $this->fileService->getRepositoryType($file->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name', 'delete document ' . $type . ' drive');
                })->first();

            if (!$permission) {
                return response()->json([
                    'errors' => ['You do not have the required authorization.'],
                ], 403);
            }

            $file->update([
                'deleted_by' => Auth::user()->id,
                'modified_by' => Auth::user()->id,
                'status' => 3,
            ]);

            $file->versions()->delete();
            $file->delete();

            DB::commit();
            return response()->json([
                'text'  =>  'File has been deleted.',
                'data' => $file
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function view_recently_uploaded(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->without(['user'])->pluck('role_id');

        $data = SF::select('id', 'name', 'type')
            ->whereHas('access', function (Builder $query) use ($user_groups) {
                $query->where('user_id', Auth::user()->id)
                    ->orWhereIn('group_id', $user_groups);
            })->without([
                'author',
                'modifier',
                'extraFields',
                'access',
                'versions',
                'documentType'
            ])
            ->latest()->paginate($paginate);

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data
        ]);
    }

    public function fetch_updated_files(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = SF::with('section.id')->select('id', 'name', 'updated_at', 'type')
            ->whereHas('access', function (Builder $query) use ($user_groups) {
                $query->where('user_id', Auth::user()->id)
                    ->orWhereIn('group_id', $user_groups);
            })->paginate($paginate);

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data
        ]);
    }


    public function fetch_updated_folders(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');

        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->without(['user'])->pluck('role_id');
        $sections = Section::whereHas('access', function (Builder $query) use ($user_groups) {
            $query->where('user_id', Auth::user()->id)
                ->orWhereIn('group_id', $user_groups);
        })->without(['access', 'files', 'author'])->paginate($paginate);

        return response()->json([
            'data' => $sections,
            "raw_data" => $sections,
            'parent' => null,
            'breadcrumbs' => [
                [
                    'id' => 0,
                    'name' => 'Repositories',
                ],
            ],
        ]);
    }


    public function fetch_groups(Request $req)
    {
        $data = Role::whereHas('roleHasSystem', function (Builder $query) {
            $query->where('system_id', 1);
        })->get();

        $currentGroupAccess = Role::whereHas('fileAccess', function (Builder $query) use ($req) {
            $query->where('file_id', $req->file_id);
        })->get();

        return response()->json([
            'data' => $data,
            'currentGroupAccess' => $currentGroupAccess,
        ]);
    }

    public function fetch_users(Request $req)
    {
        $data = User::whereNotIn('name', ['Developer Account', 'System Account'])->without(['roles', 'permissions', 'supervisor', 'employeeMetaInfo', 'storage'])->get(['id', 'name']);
        $file = SF::whereId($req->file_id)->first();
        $groupIds = $file->access()->pluck('group_id');
        $fileId = $req->file_id;
        $currentUserAccess = User::whereNotIn('id', [1, 2])
            ->whereHas('file_access', function (Builder $query) use ($groupIds, $fileId) {
                $query->whereIn('id', $groupIds)->orWhere('file_id', $fileId);
            })->get(['id', 'name']);

        return response()->json([
            'data' => $data,
            'currentUserAccess' => $currentUserAccess,
        ]);
    }

    public function configure_access(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'file_id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $data = SF::whereId($req->file_id)->first();

            $initial_group = $data->access->whereNotNull('group')
                ->whereNotIn('group.name', ['Developer', 'Admin', 'Super Admin'])
                ->pluck('group_id')->toArray();

            $initial_user = $data->access->whereNotIn('user_id', [1, 2, 0])->pluck('user_id')->toArray();

            $userAccess = collect($req->userAccess)->pluck('id')->toArray();
            $groupAccess = collect($req->groupAccess)->pluck('id')->toArray();
            $old_users = $this->compare_array($initial_user, $userAccess);
            $old_groups = $this->compare_array($initial_group, $groupAccess);

            FileAccess::whereIn('user_id', $old_users)
                ->orWhereIn('group_id', $old_groups)
                ->forceDelete();

            if (!empty($req->userAccess)) {
                foreach ($userAccess as $user) {
                    $data->access()->firstOrCreate([
                        'user_id' => $user,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }

            if (!empty($req->groupAccess)) {
                foreach ($groupAccess as $group) {
                    $data->access()->firstOrCreate([
                        'user_id' => 0,
                        'group_id' => $group,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                'message' => 'File access has been configured.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function compare_array($array1, $array2)
    {
        return array_diff($array1, $array2);
    }

    public function check_duplicate_file(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'file_names' => 'required|array',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $responses = [];

        foreach ($req->file_names as $index => $file_name) {
            $existingFile = SF::where([
                ['name', $file_name], ['section_id', $req->section_id]
            ])->first();

            if ($existingFile) {
                $hasTransaction = false;

                $fileApproval = FileVersion::where('file_id', $existingFile->id)->whereHas('approval', function ($query) {
                    $query->whereNotIn('status', [1, 2]);
                })->first();

                if ($fileApproval) {
                    $hasTransaction = true;
                }

                $response = [
                    'index' => $index,
                    'name' => $file_name,
                    'duplicate' => true,
                    'has_transaction' => $hasTransaction,
                    'id' => $existingFile->id,
                    'message' => 'Duplicate file found.'
                ];
            } else {
                $response = [
                    'name' => $file_name,
                    'duplicate' => false,
                    'message' => 'No duplicate file found.'
                ];
            }

            $responses[] = $response;
        }

        return response()->json($responses);
    }

    public function check_rename_duplicate_file(Request $req)
    {

        $existingFile = SF::where([
            ['name', $req->file_name], ['section_id', $req->section_id]
        ])->first();

        if ($existingFile) {
            $hasTransaction = false;

            $fileApproval = FileVersion::where('file_id', $existingFile->id)->whereHas('approval', function ($query) {
                $query->whereNotIn('status', [1, 2]);
            })->first();

            if ($fileApproval) {
                $hasTransaction = true;
            }

            $response = [
                'name' => $req->file_name,
                'duplicate' => true,
                'has_transaction' => $hasTransaction,
                'id' => $existingFile->id,
                'message' => 'Duplicate file found.'
            ];
        } else {
            $response = [
                'name' => $req->file_name,
                'duplicate' => false,
                'message' => 'No duplicate file found.'
            ];
        }

        return response()->json($response);
    }
}
