<?php

namespace Suiterus\Dms\Controllers\SectionManager;

use Activity;
use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Models\Files\FileComment;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Enums\Log\FileVersionLogType;
use Suiterus\Dms\Models\Files\FileVersion as FV;

class FileCommentController extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function file_comment_upload(Request $request)
    {
        $valid = Validator::make(
            $request->all(),
            [
                'id' => 'required|numeric|exists:dms_db.file_versions,id',
                'body' => 'required|string',
            ],
            [
                'id.exists' => 'File not found.',
            ]
        );

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $file = FV::where('id', $request->id)->first();

            $type = $this->fileService->getRepositoryType($file->file->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name', 'file comment ' . $type . ' drive');
                })->first();

            if (!$permission) {
                return response()->json([
                    'errors' => ['You do not have the required authorization.'],
                ], 403);
            }

            $data = $file->comment()->create([
                'file_id' => $request->id,
                'body' => $request->body,
                'created_by' => Auth::user()->id,
            ]);

            $fileCommentCollection = FileComment::with(['versions.file'])->find($data->id);

            $message = Auth::user()->name . ' has commented on the file ' . $file->file_name;

            $this->logCustomMessage(
                'file_version_comment',
                $fileCommentCollection,
                $message,
                $fileCommentCollection,
                FileVersionLogType::COMMENT,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text' => 'Comment uploaded successfully.',
                'data' => $data,
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function file_comment_list(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|numeric|exists:dms_db.file_versions,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        $file = FV::where('id', $request->id)->first();

        $type = $this->fileService->getRepositoryType($file->file->type);
        $permission =  User::where('id', Auth::user()->id)
            ->whereHas('permissions', function ($query) use ($type) {
                $query->where('name', 'list file comment ' . $type . ' drive');
            })->first();

        if (!$permission) {
            return response()->json([
                'errors' => ['You do not have the required authorization.'],
            ], 403);
        }

        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        $data = FileComment::where('file_version_id', $request->id)
            ->orderBy('created_at', 'desc')
            ->without('user')
            ->paginate($paginate);

        return response()->json([
            'message' => 'Fetch successful.',
            'data' => $data,
        ]);
    }
}
