<?php

namespace Suiterus\Dms\Controllers\Repositories;

use Activity;
use App\Enums\Dms\DriveType;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Suiterus\Dms\Contracts\Repositories\FolderListing;
use Suiterus\Dms\Contracts\Repositories\RootFolderListing;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFile;
use Suiterus\Dms\Contracts\Repositories\SearchAbleFolder;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Request\Repositories\CreateFolderRequest;
use Suiterus\Dms\Request\Repositories\DeleteFolderRequest;
use Suiterus\Dms\Request\Repositories\MoveFolderRequest;
use Suiterus\Dms\Request\Repositories\RenameFolderRequest;
use Suiterus\Dms\Services\Repositories\FolderService;

class FolderController extends Controller
{

    use HasCustomLogs;

    public function folderView(Request $req)
    {
        $data = Section::where('id', $req->id)->get(['name', 'description', 'id', 'type', 'parent_id'])->first();

        $this->logCustomMessage(
            'view_folder',
            $data ?? null,
            Auth::user()->name . ' viewed folder ' . $data->name,
            $data ?? null,
            'View a folder',
            new Activity()
        );

        return response()->json([
            'data' => $data,
            'breadcrumbs' => $this->breadcrumbs(Section::whereId($req->id)->firstOrFail(), $req->drive_type)
        ]);
    }

    public function breadcrumbs($items, $driveType)
    {
        $data = [];
        if ($items === null) {
            return $data;
        }
        $item = $items;
        $item['disabled'] = true;
        $data[] = $item;
        while ($item->parent !== null) {
            if ($item->parent === null) {
                break;
            }
            $x = $item['parent'];
            $x['disabled'] = false;
            $data[] = $x;
            $item = $item['parent'];
        }

        $data[] = [
            'id' => 0,
            'name' => $this->determineDriveType($driveType),
        ];
        return array_reverse($data);
    }

    public function determineDriveType($driveType) {
        switch ($driveType) {
            case DriveType::PERSONAL:
                return 'Personal Drive';
            case DriveType::PRIVATE:
                return 'Private Drive';
            case DriveType::PUBLIC:
                return 'Shared Drive';
            default:
                return;
        }
    }

    public function move_folder(MoveFolderRequest $req, FolderService $service)
    {
        return $service->moveFolder($req->folders, $req->id);
    }

    public function copy_folder(MoveFolderRequest $request, FolderService $service)
    {
        return $service->copyFolder($request->folders, $request->id);
    }

    public function deleteFolder(DeleteFolderRequest $request, FolderService $service)
    {
        
        return $service->deleteFolder($request->folders);
    }

    public function listRootFolder(Request $req, RootFolderListing $folder)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        return $folder->listRootFolder($req->id ?? 0, new DriveType, $paginate);
    }

    public function searchFolder(Request $req, SearchAbleFolder $search)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        return $search->searchFolder($req->id ?? 0, $req->keyword ?? '', new DriveType, $paginate);
    }

    public function searchFile(Request $req, SearchAbleFile $search)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        return $search->searchFile($req->id ?? 0, $req->keyword ?? '', new DriveType, $paginate);
    }

    public function movableFolderList(array $folders, DriveType $driveType, int $currentSectionId, FolderListing $folder)
    {
        return $folder->movableFolderList($folders, $driveType, $currentSectionId);
    }

    public function createRootFolder(CreateFolderRequest $request, FolderService $service)
    {
        return $service->createFolder($request->name, $request->description ?? '', $request->parent_id ?? 0, $request->drive_type, $request->users ?? [], $request->groups ?? [], $request->option);
    }
    
    public function fetchFolder(Request $request, FolderService $service)
    {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        return $service->fetchFolder($paginate, $request->id, $request->drive_type);
    }

    public function fetchFolderFile(Request $request, FolderService $service)
    {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        return $service->fetchFolderFile($request->id, $request->drive_type, $paginate);
    }

    public function renameFolder(RenameFolderRequest $request, FolderService $service)
    {
        return $service->renameFolder($request->id, $request->folder_name, $request->folder_description, $request->parent_id, $request->option, $request->drive_type, $request->users ?? [], $request->groups ?? []);
    }

    public function checkDuplicateFolder(Request $request, FolderService $service) 
    {
        return $service->checkDuplicateFolder($request);
    }

    public function fetchHasSectionAccess(Request $request, FolderService $service)
    {
        return $service->fetchHasSectionAccess($request);
    }
}
