<?php

namespace Suiterus\Dms\Controllers\OtherAttachments;

use Activity;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Suiterus\Dms\Models\Files\FileType;
use Illuminate\Support\Facades\Validator;
use Iman\Streamer\VideoStreamer;
use NcJoes\OfficeConverter\OfficeConverter;
use Suiterus\Dms\Classes\UserAccess;
use Suiterus\Dms\Controllers\Configuration\AllowedFileTypesController;
use Suiterus\Dms\Enums\Log\AttachmentLogType;
use Suiterus\Dms\Models\Files\FileVersion;
use Suiterus\Dms\Models\OtherAttachments\OtherAttachments;

class OtherAttachmentsController extends Controller
{

    use HasCustomLogs;

    public function create(Request $request)
    {
        $allowedTypes = AllowedFileTypesController::getType();;
        $valid = Validator::make(
            $request->all(),
            [
                'files' => 'required|array',
                'files.*' => 'required|max:' . env('MAX_FILE_UPLOAD_SIZE') . '|mimes:' . $allowedTypes,
            ],
            [
                'files.*.required' => 'File is empty'
            ],
        );
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($request->file('files') as $file) {
                $path = Auth::user()->id . '/' . $request->file_id;
                $filePath = Storage::disk('attachment')->put($path, $file);

                $otherAttachment = OtherAttachments::create([
                    'file_id' => $request->file_id,
                    'user_id' => Auth::user()->id,
                    'name' => $file->getClientOriginalName(),
                    'path' => $filePath,
                    'updated_by' => Auth::user()->id
                ]);

                $otherAttachmentCollection = OtherAttachments::with(['file'])->find($otherAttachment->id);

                $message = Auth::user()->name . ' has uploaded an attachment file ' . $file->getClientOriginalName() . ' to ' . $otherAttachmentCollection->file->name;

                $this->logCustomMessage(
                    'file_attachment_upload',
                    $otherAttachmentCollection,
                    $message,
                    $otherAttachmentCollection,
                    AttachmentLogType::UPLOAD,
                    new Activity()
                );
            }

            DB::commit();
            return response()->json([
                'text' =>  count($request->file('files')) . ' file(s) have been successfully uploaded.',
                'data' => $request->all()
            ]);
        } catch (Exception $e) {
            DB::rollback();
            Storage::disk('local')->delete($filePath);
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function fetchById(Request $request)
    {

        $otherAttachment = OtherAttachments::where('id', $request->file_other_attachment_id)->first();
        if ($otherAttachment) {
            $this->logCustomMessage('view_file_attachment', $otherAttachment, Auth::user()->name . ' viewed file attachment ' . $otherAttachment->name, $otherAttachment, 'View a file attachment', new Activity());
        }
        return $otherAttachment;
    }

    public function fetch(Request $request)
    {
        return OtherAttachments::where('file_id', $request->file_id)->latest()->paginate(2);
    }

    public function download(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_other_attachment_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.file_other_attachments,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        $otherAttachment = OtherAttachments::find($request->file_other_attachment_id);
        $mimeType = Storage::disk('attachment')->mimeType($otherAttachment->path);
        switch ($mimeType) {
                // docx
            case 'application/vnd.openxmlformats-officedocument.wordprocessingml.document':
            case 'application/msword':
            case 'application/doc':
            case 'application/ms-doc':
                // xlsx
            case 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet':
            case 'application/excel':
            case 'application/vnd.ms-excel':
            case 'application/x-excel':
            case 'application/x-msexcel':
                // ppt
            case 'application/vnd.openxmlformats-officedocument.presentationml.presentation':
            case 'application/mspowerpoint':
            case 'application/powerpoint':
            case 'application/vnd.ms-powerpoint':
            case 'application/x-mspowerpoint':
                // $converter = new OfficeConverter(storage_path('app/public/1_Access Management.docx'), null, 'soffice', false);
                $ftpFile = Storage::disk('attachment')->get($otherAttachment->path);

                $folderUniqid = uniqid();

                $temporaryPath = $folderUniqid . '/' . basename($otherAttachment->path);

                Storage::disk('converted_office')->put($temporaryPath, $ftpFile);

                $converter = new OfficeConverter(Storage::disk('converted_office')->path($temporaryPath), null, env('LIBRE_OFFICE_COMMAND'), false);

                $uniqid = uniqid();

                $convertedPath = $converter->convertTo($uniqid . '.pdf'); //generates pdf file in same directory as test-file.docx
                // $converter->convertTo('output-file.html'); //generates html file in same directory as test-file.docx

                Storage::disk('converted_office')->delete($temporaryPath);
                return response()->download($convertedPath)->deleteFileAfterSend(true);
                break;
            default:
                return Storage::disk('attachment')->download($otherAttachment->path);
                break;
        }
    }

    public function fileDownload(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_other_attachment_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.file_other_attachments,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'error' => $valid->errors()
            ], 400);
        }

        $otherAttachment = OtherAttachments::with(['file'])->find($request->file_other_attachment_id);

        $message = Auth::user()->name . ' has downloaded an attachment file ' . $otherAttachment->name . ' to ' . $otherAttachment->file->name;

        $this->logCustomMessage(
            'file_attachment_download',
            $otherAttachment,
            $message,
            $otherAttachment,
            AttachmentLogType::DOWNLOAD,
            new Activity()
        );

        return Storage::disk('attachment')->download($otherAttachment->path);
    }

    public function playVideo(Request $request)
    {

        $otherAttachment = OtherAttachments::find($request->file_other_attachment_id);

        VideoStreamer::streamFile(Storage::disk('attachment')->path($otherAttachment->path));
    }

    public function allowed_type()
    {
        $allowed = FileType::where('status', '=', '1')->pluck('file_ext');
        $allowed = str_replace('"', "", $allowed);
        $allowed = str_replace(array('[', ']'), '', $allowed);
        $allowed = $allowed . ',' . strtolower($allowed);

        return $allowed;
    }
}
