<?php

namespace Suiterus\Dms\Controllers\GenerateLink;

use Activity;
use Exception;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Dms\Models\Files\File;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Models\GenerateLink\GenerateFileLink;

class GenerateFileLinkController extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function create(Request $request)
    {
        $file = File::findOrFail($request->file_id);
        $type = $this->fileService->getRepositoryType($file->type);
        $permission =  User::where('id', Auth::user()->id)
            ->whereHas('permissions', function ($query) use ($type) {
                $query->where('name', 'configure document ' . $type . ' drive');
            })->first();

        if (!$permission) {
            return response()->json([
                'errors' => ['You do not have the required authorization.'],
            ], 403);
        }
        DB::connection(env('DMS_DB_CONNECTION'))->beginTransaction();

        try {
            $generateFileLink = GenerateFileLink::where('file_id', $request->file_id)->first();
            $values = [
                'file_id' => $request->file_id,
                'is_viewable' => $request->is_viewable ?? true,
                'created_by' => Auth::id()
            ];
            if (!$generateFileLink) {
                $values = [
                    'file_id' => $request->file_id,
                    'generated_file_link' => Str::orderedUuid(),
                    'is_viewable' => $request->is_viewable ?? true,
                    'created_by' => Auth::id()
                ];
            }

            $generateFileLink = GenerateFileLink::updateOrCreate(
                ['file_id' => $request->file_id],
                $values
            );

            if (!$generateFileLink) {
                $this->logCustomMessage('generate_file_link', $generateFileLink, Auth::user()->name . ' created a public link for file ' . $generateFileLink->file->name . '. The link is: [link]', $generateFileLink, 'Generate file link', new Activity());
            } else if ($generateFileLink) {
                $isViewAbleText = $request->is_viewable ? 'shared' : 'unshared';
                $logType = $request->is_viewable ? 'shared' : 'unshared';
                $this->logCustomMessage($logType . '_file_link', $generateFileLink, Auth::user()->name . ' ' .  $isViewAbleText . ' a public link for file ' . $generateFileLink->file->name, $generateFileLink, ucfirst($logType) . ' file link', new Activity());
            }

            DB::connection(env('DMS_DB_CONNECTION'))->commit();
            return response()->json([
                'text' => 'Generated File Link has been created',
                'link' => $generateFileLink
            ]);
        } catch (Exception $e) {
            DB::connection(env('DMS_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors' => ['There is a problem creating generated file link'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    public function fetch(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'uuid' => 'required',
            'type' => 'required|in:' . 'file'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $viewable = 1;

        $generateFileLink = GenerateFileLink::where([
            ['generated_file_link', $request->uuid],
            ['is_viewable', $viewable]
        ])->first();
        $generateFileLink = json_decode($generateFileLink);

        if (!$generateFileLink) {
            abort(404);
        }

        $access = $this->fileService->getAccessType($generateFileLink->file->type);

        $generateFileLink->path = $this->getPath($generateFileLink, $access);

        return response()->json([
            'viewable' => true,
            'link' => $generateFileLink
        ], 200);
    }

    public function fetchByFileId(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.files,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $viewable = 1;

        $generateFileLink = GenerateFileLink::where([
            ['file_id', $request->file_id],
            ['is_viewable', $viewable]
        ])->first();
        $generateFileLink = json_decode($generateFileLink);

        if (!$generateFileLink) {
            abort(404);
        }

        $access = $this->fileService->getAccessType($generateFileLink->file->type);
        $generateFileLink->path = $this->getPath($generateFileLink, $access);

        return $generateFileLink;
    }

    public function download(Request $request)
    {
        $viewable = 1;

        $generateFileLink = GenerateFileLink::where([
            ['file_id', $request->file_id],
            ['is_viewable', $viewable]
        ])->first();
        $generateFileLink = json_decode($generateFileLink);

        if (!$generateFileLink) {
            abort(404);
        }

        return Storage::download($request->path);
    }

    public function getPath($data, $access)
    {
        return 'dms/' . $access . '/' . $data->file->latest_version->user->id . '/' . $data->file->section_id . '/' . $data->file->id . '/' . $data->file->latest_version->file_version . '/' . $data->file->latest_version->file_name;
    }
}
