<?php

namespace Suiterus\Dms\Controllers\GenerateLink;

use Activity;
use Exception;
use ZipArchive;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use RecursiveIteratorIterator;
use RecursiveDirectoryIterator;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\FileVersion;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Models\Files\File as FilesFile;
use Suiterus\Dms\Models\GenerateLink\GenerateDriveLink;

class GenerateDriveLinkController extends Controller
{
    use HasCustomLogs;
    
    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function create(Request $request)
    {
        $section = Section::findOrFail($request->section_id);
        $type = $this->fileService->getRepositoryType($section->type);
            $permission =  User::where('id', Auth::user()->id)
                ->whereHas('permissions', function ($query) use ($type) {
                    $query->where('name','folder configure '.$type.' drive');
                })->first();
            
        if(!$permission){
            return response()->json([
                'errors' => ['You do not have the required authorization.'],
            ], 403);
        }
        DB::beginTransaction();

        try {
            $generateDriveLink = GenerateDriveLink::where('section_id', $request->section_id)->first();
            $values = [
                'section_id' => $request->section_id,
                'is_viewable' => $request->is_viewable ?? true,
                'created_by' => Auth::id()
            ];
            if (!$generateDriveLink) {
                $values = [
                    'section_id' => $request->section_id,
                    'generated_drive_link' => Str::orderedUuid(),
                    'is_viewable' => $request->is_viewable ?? true,
                    'created_by' => Auth::id()
                ];
            }
            $generateDriveLink = GenerateDriveLink::updateOrCreate(
                ['section_id' => $request->section_id],
                $values
            );

            if (!$generateDriveLink) {
                $this->logCustomMessage('generate_folder_link', $generateDriveLink, Auth::user()->name . ' created a public link for folder ' . $generateDriveLink->section->name . '. The link is: [link]', $generateDriveLink, 'Generate folder link', new Activity());
            } else if ($generateDriveLink) {
                $isViewAbleText = $request->is_viewable ? 'shared' : 'unshared';
                $logType = $request->is_viewable ? 'shared' : 'unshared';
                $this->logCustomMessage($logType . '_folder_link', $generateDriveLink, Auth::user()->name . ' ' .  $isViewAbleText . ' a public link for folder ' . $generateDriveLink->section->name, $generateDriveLink, ucfirst($logType) . ' folder link', new Activity());
            }

            DB::commit();
            return response()->json([
                'text' => 'Drive Link has been created',
                'link' => $generateDriveLink
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['There is a problem creating drive link'],
                'msg' => $e->getMessage()
            ], 500);
        }
    }

    public function fetch(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'uuid' => 'required',
            'type' => 'required|in:' . 'drive'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $viewable = 1;

        $generateDriveLink = GenerateDriveLink::where([
            ['generated_drive_link', $request->uuid],
            ['is_viewable', $viewable]
        ])->first();
        $generateDriveLink = json_decode($generateDriveLink);

        if (!$generateDriveLink) {
            abort(404);
        }

        return response()->json([
            'viewable' => true,
            'link' => $generateDriveLink
        ], 200);
    }

    public function fetchBySectionId(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'section_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.sections,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $viewable = 1;

        $generateDriveLink = GenerateDriveLink::where([
            ['section_id', $request->section_id],
            ['is_viewable', $viewable]
        ])->first();
        $generateDriveLink = json_decode($generateDriveLink);

        if (!$generateDriveLink) {
            abort(404);
        }

        return $generateDriveLink;
    }

    public function download(Request $request)
    {
        try {
            $viewable = 1;

            $generateDriveLink = GenerateDriveLink::where([
                ['section_id', $request->section_id],
                ['is_viewable', $viewable]
            ])->first();
            $generateDriveLink = json_decode($generateDriveLink);

            if (!$generateDriveLink) {
                abort(404);
            }

            $parentSection = Section::where('id', $request->section_id)->firstOrFail();
            $childSections = Section::where('parent_id', $request->section_id)->get();

            $uuid = Str::orderedUuid();
            $zippableFoldersPathName = 'downloadable-drive-external-folders-to-zip';
            $temporaryParentDirectory = $zippableFoldersPathName . '/' . $uuid . '/' . $parentSection->name;

            Storage::disk('local')->makeDirectory($temporaryParentDirectory);

            $files = FilesFile::where('section_id', $request->section_id)->get();

            foreach ($files as $file) {
                $latestFileVersion = FileVersion::where('file_id', $file['id'])->firstOrFail();
                $access = $this->fileService->getAccessType($latestFileVersion->file->type);
                $filePath = $this->fileService->getPath($latestFileVersion, $access) . '/' . $latestFileVersion->file_name;

                Storage::disk('local')->copy($filePath, $temporaryParentDirectory . '/' . $latestFileVersion->file_name);
            }

            $this->createChildSection($childSections, $temporaryParentDirectory);

            // Get real path for our folder
            $rootPath = storage_path('app/' . $temporaryParentDirectory);

            $zipUuid = Str::orderedUuid();

            if (!File::exists('app/downloadable-drive-external-zip')) {
                Storage::disk('local')->makeDirectory('downloadable-drive-external-zip');
            }

            //zip file path
            $zipFilePath = storage_path('app/downloadable-drive-external-zip/' . $zipUuid);

            // Initialize archive object
            $zip = new ZipArchive();
            $zip->open($zipFilePath, ZipArchive::CREATE | ZipArchive::OVERWRITE);

            // Create recursive directory iterator
            /** @var SplFileInfo[] $files */
            $files = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($rootPath),
                RecursiveIteratorIterator::LEAVES_ONLY
            );

            foreach ($files as $name => $file) {

                // Skip directories (they would be added automatically)
                if (!$file->isDir()) {
                    // Get real and relative path for current file
                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen($rootPath) + 1);
                    // Add current file to archive
                    $zip->addFile($filePath, $relativePath);
                }
            }

            // Zip archive will be created only after closing object
            $zip->close();

            Storage::disk('local')->deleteDirectory($zippableFoldersPathName . '/' . $uuid);

            return response()->download($zipFilePath)->deleteFileAfterSend(true);
        } catch (Exception $e) {
            return response()->json([
                'message' => "There are no files or directories in the folder, or the file doesn't exist.",
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * It takes an array of sections, and a temporary directory, and creates a directory for each
     * section in the temporary directory, and then copies all the files in each section into the
     * directory
     * 
     * @param sections an array of sections that you want to download
     * @param temporaryParentDirectory The directory where the files will be stored.
     * 
     * @return Nothing.
     */
    public function createChildSection($sections, $temporaryParentDirectory)
    {
        foreach ($sections as $section) {
            $temporaryChildDirectory = $temporaryParentDirectory . '/' . $section['name'];
            Storage::disk('local')->makeDirectory($temporaryParentDirectory . '/' . $section['name']);

            $files = FilesFile::where('section_id', $section['id'])->get();

            foreach ($files as $file) {
                $latestFileVersion = FileVersion::where('file_id', $file['id'])->firstOrFail();
                $access = $this->fileService->getAccessType($latestFileVersion->file->type);
                $filePath = $this->fileService->getPath($latestFileVersion, $access) . '/' . $latestFileVersion->file_name;

                Storage::disk('local')->copy($filePath, $temporaryChildDirectory . '/' . $latestFileVersion->file_name);
            }

            $childSections = Section::where('parent_id', $section['id'])->get();
            if (count($childSections) > 0) {
                $this->createChildSection($childSections, $temporaryChildDirectory);
            }
        }

        return;
    }

    public function getSectionPath($access, $createdBy, $sectionId)
    {
        return 'dms/' . $access . '/' . $createdBy . '/' . $sectionId;
    }
}
