<?php

namespace Suiterus\Dms\Controllers\Dashboard;

use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Dms\Models\Repositories\SectionAccess;
use Suiterus\Dms\Models\Repositories\Section;
use App\Http\Controllers\Controller;
use Spatie\Permission\Models\Role;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Models\Files\FileField as SFF;
use Suiterus\Dms\Models\Files\FileType;
use Suiterus\Dms\Models\Files\FileAccess;
use Illuminate\Http\Request;
use App\Models\User;
use Carbon\Carbon;
use Validator;
use Exception;
use Auth;
use DB;
use Suiterus\Dms\Classes\UserAccess;
use Suiterus\Dms\Models\Files\File;

class DashboardController extends Controller
{

    public function file_upload(Request $req)
    {
        $allowedTypes = $this->allowed_type();
        $valid = Validator::make(
            $req->all(),
            [
                'files' => 'required|max:' . env('MAX_FILE_UPLOAD_SIZE') . '|mimes:' . $allowedTypes,
            ],
            [
                'files.required' => 'File is empty.',
                'files.mimes' => 'Invalid file type.',
            ]
        );

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $file = $req->file('files');
            $data = SF::create([
                'section_id' => $req->section,
                'user_id' => Auth::user()->id,
                'name' => $file->getClientOriginalName(),
                'type' => 3,
                'document_type' => $req->document_type,
                'privacy' => $req->privacy,
                'modified_by' => 0,
                'effectivity' => Carbon::now(),
                'expiry' => Carbon::now()->addYears(5),
            ]);




            // PERSONAL DRIVE
            // store to personal folder
            $version = '1';
            $file_version_name = $version . '_' . $file->getClientOriginalName();
            $file->storeAs('dms/personal/' . Auth::user()->id . '/' . $req->section . '/' . $data->id . '/' . $version, $file_version_name);
            //file version
            $data->versions()->create([
                'file_name' => $file_version_name,
                'file_version' => $version,
                'created_by' => Auth::user()->id,
            ]);
            // user access
            $data->access()->create([
                'user_id' => Auth::user()->id,
                'access_level' => 1,
            ]);

            // developer access
            $devs = Role::where(['name' => 'Developer']);
            if ($devs->count()) {
                if ($data->access()->where(['group_id' => $devs->first()->id])->count() == 0) {
                    $data->access()->create([
                        'user_id' => 0,
                        'group_id' => $devs->first()->id,
                        'access_level' => 1,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }

            $users = json_decode(json_encode($req->users));
            $groups = json_decode(json_encode($req->groups));

            if ($users) {
                foreach ($users as $user) {
                    $data->access()->create([
                        'user_id' => $user,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }

            if ($groups) {
                foreach ($groups as $group) {
                    $data->access()->create([
                        'group_id' => $group,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                }
            }



            DB::commit();
            return response()->json([
                'text' =>  $req->file('files')->getClientOriginalName() . ' have been successfully uploaded.',
                'data' => $req->all(),
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function allowed_type()
    {
        $allowed = FileType::where('status', '=', '1')->pluck('file_ext');
        $allowed = str_replace('"', "", $allowed);
        $allowed = str_replace(array('[', ']'), '', $allowed);
        $allowed = $allowed . ',' . strtolower($allowed);

        return $allowed;
    }
    public function fetch_allowed_types()
    {
        return response()->json([
            'data'  => FileType::where('status', '=', '1')->pluck('file_ext')
        ]);
    }

    public function fetch_user_with_license()
    {
        return response()->json([
            'data' => User::whereHas('license')->get()
        ]);
    }

    public function check_access_dashboard(Request $request)
    {
        $user = User::where('id', Auth::user()->id)->first();
        $roles = $user->roles->pluck('id');

        $file = File::find($request->file_id);

        // meaning deleted or not existing
        if (!$file) {
            return response()->json([
                'message' => 'No access',
            ], 404);
        }

        if ($file->type == 2) {
            return response()->json([
                'message' => 'Has access',
            ], 200);
        }

        $fileAccess = UserAccess::traverseFileTreeAccess($request->file_id, $roles);

        if ($fileAccess) {
            return response()->json([
                'message' => 'Has access',
            ], 200);
        } else {
            return response()->json([
                'message' => 'No access',
            ], 404);
        }
    }
    public function check_section_access(Request $request)
    {
        $user = User::where('id', Auth::user()->id)->first();

        $roles = $user->roles->pluck('id');

        $sectionAccess = UserAccess::traverseFolderTreeAccess($request->section_id, $roles);

        if ($sectionAccess) {
            return response()->json([
                'message' => $user->name . ' has section access'
            ], 200);
        } else {
            return response()->json([
                'message' => $user->name . ' has no section access'
            ], 404);
        }
    }
}
