<?php

namespace Suiterus\Dms\Controllers\Archive;

use DB;
use Activity;
use Exception;
use Validator;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Suiterus\Dms\Enums\Log\FolderLogType;
use Suiterus\Dms\Models\Files\File as SF;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Models\Repositories\Section;
use Suiterus\Dms\Models\Files\FileVersion as FV;
use Illuminate\Support\Facades\Auth as FacadesAuth;
use Suiterus\Dms\Models\Repositories\SectionAccess;
use App\Models\AccessManagement\GroupManagement\ModelHasRole;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class ArchiveController extends Controller
{

    use HasCustomLogs;
    
    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function archiveFiles()
    {
        SF::where([['status', 1], ['expiry', '<', Carbon::now()]])->each(function ($files) {
            $files->versions()->delete();
            $files->delete();
            $files->update(['status' => 2]);
        });
    }

    public function search_archive_folders(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = Section::onlyTrashed()
            ->where([['name', 'LIKE', '%' . $req->keyword . '%'], ['status', 2]])
            ->where(function ($query) use ($user_groups) {
                $query->whereHas('access', function ($query) use ($user_groups) {
                    $query->where('user_id', Auth::user()->id)
                        ->orWhereIn('group_id', $user_groups);
                })
                    ->orWhere([['status', 2], ['type', 2]]);
            })
            ->where(function ($query) {
                $query->whereHas('parent')
                    ->orWhere('parent_id', 0);
            })
            ->orderBy('name', 'asc')->paginate($paginate);

            $this->logCustomMessage('search_archive_folder', null, FacadesAuth::user()->name . ' searched for "' . $req->keyword . '"', null, 'Search archive folder', new Activity());

        return response()->json([
            'data'  =>  $data
        ]);
    }

    public function search_archive_files(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = SF::onlyTrashed()
            ->where([['name', 'LIKE', '%' . $req->keyword . '%'], ['status', 2]])
            ->where(function ($query) use ($user_groups) {
                $query->whereHas('access', function ($query) use ($user_groups) {
                    $query->where('user_id', Auth::user()->id)
                        ->orWhereIn('group_id', $user_groups);
                })
                    ->orWhere([['status', 2], ['type', 2]]);
            })
            ->where(function ($query) {
                $query->whereHas('section')
                    ->orWhere('section_id', 0);
            })
            ->orderBy('name', 'asc')->paginate($paginate);

            $this->logCustomMessage('search_archive_file', null, FacadesAuth::user()->name . ' searched for "' . $req->keyword . '"', null, 'Search archive file', new Activity());

        return response()->json([
            'data'  =>  $data
        ]);
    }

    public function listing_archive_files(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = SF::onlyTrashed()
            ->where('status', 2)
            ->where(function ($query) use ($user_groups) {
                $query->whereHas('access', function ($query) use ($user_groups) {
                    $query->where('user_id', Auth::user()->id)
                        ->orWhereIn('group_id', $user_groups);
                })
                    ->orWhere([['status', 2], ['type', 2]]);
            })
            ->where(function ($query) {
                $query->whereHas('section')
                    ->orWhere('section_id', 0);
            })
            ->orderBy('name', 'asc')
            ->paginate($paginate);
        $data->transform(function ($item) {
            return $item = $this->get_file_path($item);
        });
        return response()->json([
            'data' => $data,
        ]);
    }
    
    public function get_file_path($item)
    {   
        $access = $this->fileService->getAccessType($item->type);
        $path = 'dms/' . $access . '/' . Auth::user()->id . '/' . $item->section_id . '/' . $item->id;
        $item->path = $path;
        return $item;
    }


    public function listing_archive_folders(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user_groups = ModelHasRole::where('model_id', Auth::user()->id)->pluck('role_id');
        $data = Section::onlyTrashed()
            ->where('status', 2)
            ->where(function ($query) use ($user_groups) {
                $query->whereHas('access', function ($query) use ($user_groups) {
                    $query->where('user_id', Auth::user()->id)
                        ->orWhereIn('group_id', $user_groups);
                })
                    ->orWhere([['status', 2], ['type', 2]]);
            })
            ->where(function ($query) {
                $query->whereHas('parent')
                    ->orWhere('parent_id', 0);
            })
            ->orderBy('name', 'asc')
            ->paginate($paginate);

        return response()->json([
            'data' => $data,
        ]);
    }

    public function delete_archive_files(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'documents' => 'required|array',
            'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($req->documents as $file) {
                $file = (object) $file;
                SF::onlyTrashed()->where([['id', $file->id], ['status', 2]])->update(['status' => 3]);
            }

            DB::commit();
            return response()->json([
                'text' =>  count($req->documents) . ' file(s) has been move to trash.',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function restore_archive_files(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'documents' => 'required|array',
            'documents.*.id' => 'required|numeric|exists:dms_db.files,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($req->documents as $file) {
                $file = (object) $file;
                $data = SF::onlyTrashed()->findOrFail($file->id);
                $data->update([
                    'status' => 1,
                    'expiry' => Carbon::now()->addYears(5)
                ]);
                $data->versions()->onlyTrashed()->restore();
                $data->restore();
            }

            DB::commit();
            return response()->json([
                'text' =>  count($req->documents) . ' file(s) has been restore',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong in our system. Please contact the developer to fix it.'],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function restore_archive_folders(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'folders' => 'required|array',
            'folders.*.id' => 'required|numeric|exists:dms_db.sections,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($req->folders as $folder) {
                $folder = (object) $folder;
                $data = Section::onlyTrashed()->findOrFail($folder->id);
                $this->restore_files($data->files()->onlyTrashed()->get());
                $this->restore_folder_recursive($data->children()->onlyTrashed()->get());
                $data->update(['status' => 1]);
                $data->restore();
            }

            DB::commit();
            return response()->json([
                'text'  =>  count($req->folders) . ' folder(s) has been restored.'
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }
    public function restore_folder_recursive($children)
    {
        foreach ($children as $child) {
            $this->restore_files($child->files()->onlyTrashed()->get());
            $child->update(['status' => 1]);
            $child->restore();
            if ($child->children()->onlyTrashed()->get()->count() > 0) {
                $this->restore_folder_recursive($child->children()->onlyTrashed()->get());
            }
        }
    }
    public function restore_files($files)
    {
        foreach ($files as $file) {
            $file->versions()->onlyTrashed()->restore();
            $file->update([
                'status' => 1,
                'expiry' => Carbon::now()->addYears(5)
            ]);
            $file->restore();
        }
    }

    public function manual_archive_folder(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|numeric|exists:dms_db.sections,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $data = Section::findOrFail($req->id);
            $data->update(['status' => 2]);
            $data->files()->update(['status' => 2]);
            $this->archive_versions_of_files($data->files);
            $this->manual_archive_folder_recursive($data->children);
            $data->delete();
            $this->logCustomMessage('folder_archive', $data, 'The '. $data->name . ' folder has been archived by ' . FacadesAuth::user()->name, $data, FolderLogType::ARCHIVE, new Activity());

            DB::commit();
            return response()->json([
                'text'  =>  'Folder has been move to archived.',
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Something went wrong in our system. Please contact the developer to fix it.'],
                'message'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function manual_archive_folder_recursive($children)
    {
        foreach ($children as $key => $child) {
            $child->update(['status' => 2]);
            $child->files()->update(['status' => 2]);
            $child->children()->delete();
            $this->archive_versions_of_files($child->files);
            $child->delete();
            if ($child->children->count() > 0) {
                $this->manual_archive_folder_recursive($child->children);
            }
        }
    }

    public function archive_versions_of_files($files)
    {
        foreach ($files as $key => $file) {
            $file->versions()->delete();
            $file->delete();
        }
    }
}
