<?php

namespace Suiterus\Dms\Controllers\Approvals;

use Activity;
use Exception;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Suiterus\Dms\Models\Files\File;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Classes\NotificationMessage;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Exports\ApprovalsExport;
use Suiterus\Dms\Models\Files\FileVersion;
use Suiterus\Dms\Enums\Log\ApprovalLogType;
use Suiterus\Dms\Models\Approvals\Approval;
use Suiterus\Dms\Services\File\FileService;
use Suiterus\Dms\Models\CustomForm\CustomForm;
use Illuminate\Pagination\LengthAwarePaginator;
use Suiterus\Dms\Models\Approvals\FileApprover;
use Suiterus\Dms\Models\Approvals\UserApprover;
use Suiterus\Dms\Enums\Approvals\ApprovalStatus;
use Suiterus\Dms\Models\CustomForm\CustomFormValue;
use Suiterus\Dms\Enums\Approvals\FileApproverStatus;
use Suiterus\Dms\Enums\Approvals\UserApproverStatus;

class UserApproverController extends Controller
{

    use HasCustomLogs;

    private $fileService;

    public function __construct(FileService $fileService)
    {
        $this->fileService = $fileService;
    }

    public function updateFileStatus(Request $request)

    {
        $valid = Validator::make($request->all(), [
            'file_approver_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.file_approvers,id',
            'receiver_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'status' => 'required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }
        DB::beginTransaction();
        try {
            UserApprover::where([
                ['user_id', Auth::id()],
                ['file_approver_id', $request->file_approver_id]
            ])->update([
                'remark_type_id' => $request->remark_type_id ?? null,
                'remark_description' => $request->remark_description ?? null,
                'status' => $request->status
            ]);
            $userApprover = UserApprover::where([
                ['user_id', Auth::id()],
                ['file_approver_id', $request->file_approver_id]
            ])->with(['fileApprover.approval.file'])->first();

            if ($userApprover) {
                // status = 1 is not used as it became a remark when declining a document but remains as a part of status
                $fileApprover = FileApprover::where('id', $request->file_approver_id)->with(['approval'])->first();

                $this->updateUserApproverStatus($request, $fileApprover, $userApprover);
            }

            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function updateMultipleStatusByID(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'user_approver_ids' => 'required|array',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }
        DB::beginTransaction();
        try {
            foreach ($request->user_approver_ids as $id) {
                $userApproverModel = UserApprover::where([
                    ['user_id', Auth::id()],
                    ['id', $id]
                ]);

                $userApproverModel->update([
                    'status' => $request->status
                ]);

                $userApprover = $userApproverModel->with(['fileApprover.approval'])->first();

                $fileApproverID = $userApprover->file_approver_id;

                $fileApprover = FileApprover::where('id', $fileApproverID)->with(['approval'])->first();


                $this->updateUserApproverStatus($request, $fileApprover, $userApprover);
            }
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }

    public function fetch(Request $request)


    {
        $userApprover = json_decode(json_encode(UserApprover::where('id', $request->user_approver_id)->with(['fileApprover', 'fileApprover.approval', 'fileApprover.approval.file'])->first()));
        if ($userApprover->user_id == Auth::id() || $userApprover->file_approver->approval->user_id == Auth::id()) {
            $fileType = $userApprover->file_approver->approval->file->file->type;
            $fileVersionId = $userApprover->file_approver->approval->file_version_id;
            $fileVersion = FileVersion::where('id', $fileVersionId)->first();
            $access = $this->fileService->getAccessType($fileType);
            $userApprover->file_approver->approval->file->path = $this->fileService->getPath($fileVersion, $access);
            return $userApprover;
        }
        return response()->json([
            'message' => 'Page not found'
        ], 404);
    }

    public function approvalHistory(Request $request)

    {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        return UserApprover::whereHas('fileApprover', function ($query) use ($request) {
            $query->whereHas('approval', function ($query) use ($request) {
                $query->where([
                    ['file_version_id', $request->file_id],
                    ['user_id', $request->user_id]
                ]);
            });
        })->with(['user', 'remark', 'fileApprover'])->paginate($paginate);
    }

    public function searchApproval(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        $data = UserApprover::whereHas('fileApprover', function ($query) {
            $query->whereHas('approval', function ($query) {
                $query->whereHas('file', function ($query) {
                    $query->whereHas('file', function ($query) {
                        $query->whereNull('deleted_at');
                    });
                });
            });
        })->when($request->type == 'approval', function ($query) {
            $query->where([
                ['user_id', Auth::id()],
                ['status', 5]
            ])->orWhere([
                ['status', 4]
            ])->whereHas('fileApprover', function ($query) {
                $query->where('is_approving', true)->orWhere('status', 1);
            });
        })->when($request->type == 'submission', function ($query) {
            $query->whereHas('fileApprover', function ($query) {
                $query->whereHas('approval', function ($query) {
                    $query->where('user_id', Auth::id());
                });
            });
        })->when($request->type == 'endorsed', function ($query) {
            $query->where(function ($query) {
                $query->where([['user_id', Auth::id()], ['status', 2]])
                    ->whereHas('fileApprover', function ($query) {
                        $query->where('status', 1)
                            ->orWhere('is_approving', 0)
                            ->whereHas('approval', function ($query) {
                                $query->where([['user_id', Auth::id()], ['status', 1]]);
                            });
                    });
            })->orWhere(function ($query) {
                $query->where('status', 2)
                    ->whereHas('fileApprover', function ($query) {
                        $query->whereHas('approval', function ($query) {
                            $query->where([['user_id', Auth::id()], ['status', 1]]);
                        });
                    });
            });
        })->whereHas('fileApprover', function ($query) use ($request) {
            $query->whereHas('approval', function ($query) use ($request) {
                $query->whereHas('file', function ($query) use ($request) {
                    $query->where('file_name', 'LIKE', '%' . $request->keyword . '%')
                        ->when($request->status != null, function ($query) use ($request) {
                            $query->where('is_obsolete', $request->status);
                        })->when($request->document_type != null, function ($query) use ($request) {
                            $query->whereHas('file', function ($query) use ($request) {
                                $query->whereHas('documentType', function ($query) use ($request) {
                                    $query->where('name', $request->document_type);
                                });
                            });
                        });
                })->when($request->date != null, function ($query) use ($request) {
                    $array = explode(',', $request->date);
                    $startDate = Carbon::parse($array[0])->startOfDay();
                    $endDate = Carbon::parse($array[1])->endOfDay();
                    $query->whereBetween('created_at', [$startDate, $endDate]);
                });
            });
        })->with(['user', 'fileApprover', 'fileApprover.approval', 'fileApprover.approval.file', 'fileApprover.approval.file.customFormValue'])->paginate($paginate);

        if ($request->type == 'endorsed') {
            $array = [];
            $collection = $data->unique(function ($item) {
                return $item->fileApprover->approval->file_version_id;
            });
            foreach ($collection as $value) {
                $array[] = $value;
            }
            return new \Illuminate\Pagination\LengthAwarePaginator(
                $array,
                $data->total(),
                $data->perPage($paginate),
                $data->currentPage(),
                [
                    'path' => \Request::url(),
                    'query' => [
                        'page' => $data->currentPage()
                    ]
                ]
            );
        }
        return $data;
    }

    public function generateCSV(Request $request)
    {
        $data = UserApprover::where(function ($query) {
            $query->where([['user_id', Auth::id()], ['status', 2]])
                ->whereHas('fileApprover', function ($query) {
                    $query->where('status', 1)
                        ->orWhere('is_approving', 0)
                        ->whereHas('approval', function ($query) {
                            $query->where([['user_id', Auth::id()], ['status', 1]]);
                        });
                });
        })->orWhere(function ($query) {
            $query->where('status', 2)
                ->whereHas('fileApprover', function ($query) {
                    $query->whereHas('approval', function ($query) {
                        $query->where([['user_id', Auth::id()], ['status', 1]]);
                    });
                });
        })->whereHas('fileApprover', function ($query) use ($request) {
            $query->whereHas('approval', function ($query) use ($request) {
                $query->whereHas('file', function ($query) use ($request) {
                    $query->where('file_name', 'LIKE', '%' . $request->keyword . '%')
                        ->when($request->status != null, function ($query) use ($request) {
                            $query->where('is_obsolete', $request->status);
                        })->when($request->document_type != null, function ($query) use ($request) {
                            $query->whereHas('file', function ($query) use ($request) {
                                $query->whereHas('documentType', function ($query) use ($request) {
                                    $query->where('name', $request->document_type);
                                });
                            });
                        });
                })->when($request->date != null, function ($query) use ($request) {
                    $array = explode(',', $request->date);
                    $startDate = Carbon::parse($array[0])->startOfDay();
                    $endDate = Carbon::parse($array[1])->endOfDay();
                    $query->whereBetween('created_at', [$startDate, $endDate]);
                });
            });
        })->with(['user', 'fileApprover', 'fileApprover.approval', 'fileApprover.approval.file', 'fileApprover.approval.file.customFormValue'])->get();

        if ($data->isEmpty()) {
            return response()->json([
                'message' => 'No endorsed documents found'
            ], 404);
        }
        $array = [];
        $collection = $data->unique(function ($item) {
            return $item->fileApprover->approval->file_version_id;
        });
        foreach ($collection as $value) {
            $array[] = $value;
        }
        return (new ApprovalsExport(count($array), $array))->download('approvals.xlsx');
    }

    public function updateFileApproval(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'file_version_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.file_versions,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $data = FileVersion::where('id', $request->file_version_id)->withTrashed()->first();

            if ($data->is_obsolete == 0) {
                $data->is_obsolete = true;
                $data->save();
            } else if ($data->is_obsolete == 1) {
                $data->is_obsolete = false;
                $data->save();
            };

            DB::commit();
            return response()->json([
                'text' => 'File has been updated.',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return $e;
        }
    }

    public function updateUserApproverStatus($request, $fileApprover, $userApprover)
    {
        if ($request->status == UserApproverStatus::DECLINED) {
            $fileApprover->status = FileApproverStatus::DECLINED;
            if ($fileApprover->reject == 1) {
                $fileApprover->is_approving = false;
                $previousFileApprover = FileApprover::where([
                    ['approval_id', $fileApprover->approval_id],
                    ['sequence', $fileApprover->sequence - 1]
                ])->first();
                $previousFileApprover->is_approving = true;
                $previousFileApprover->status = FileApproverStatus::IN_PROGRESS;
                $previousFileApprover->save();
                $previousUserApprovers = UserApprover::where('file_approver_id', $previousFileApprover->id);
                $previousUserApprovers->update([
                    'status' => UserApproverStatus::IN_PROGRESS
                ]);
                $previousUserApprovers = $previousUserApprovers->get();
                NotificationMessage::notifySenderAndReceiver('Declined', $request->receiver_id, Auth::id(), $userApprover->id);
                $index = 0;
                foreach ($previousUserApprovers as $previousUserApprover) {
                    if ($index == 0) {
                        NotificationMessage::notifySenderAndReceiver('Returned a Document', $previousUserApprover['user_id'], Auth::id(), $previousUserApprover->id);
                    } else {
                        NotificationMessage::notifyReceiver('Returned a Document', $previousUserApprover['user_id'], Auth::id(), $previousUserApprover->id);
                    }
                }
            } else {
                $fileApprovers = FileApprover::where('approval_id', $fileApprover->approval_id);
                $fileApprovers->where('status', '!=', FileApproverStatus::APPROVED)->update([
                    'status' => FileApproverStatus::DECLINED
                ]);
                
                $approver = true; 
                
                foreach ($fileApprovers->get() as $fileApprover) {
                    $setStatus = $approver ? UserApproverStatus::DECLINED : UserApproverStatus::NA;

                    if ($approver) {
                        UserApprover::where([
                            ['file_approver_id', '=', $fileApprover['id']],
                            ['status', '!=', UserApproverStatus::APPROVED]
                        ])->update([
                            'status' => $setStatus
                        ]);
                        $approver = false;
                    } else {
                        UserApprover::where([
                            ['file_approver_id', '=', $fileApprover['id']],
                            ['status', '!=', UserApproverStatus::APPROVED]
                        ])->update([
                            'status' => $setStatus
                        ]);
                    }
                }

                $approval = Approval::where('id', $fileApprover->approval_id)->first();
                $approval->status = ApprovalStatus::DECLINED;
                $approval->save();

                NotificationMessage::notifySenderAndReceiver('Declined', $userApprover->fileApprover['approval']['user_id'], Auth::id(), $userApprover->id);
            }
            $fileApprover->save();
        } else if ($request->status == UserApproverStatus::APPROVED) {
            $userApproverLength = UserApprover::where('file_approver_id', $fileApprover)->count();

            $approveUserApproverStatus = UserApprover::where([
                ['file_approver_id', $fileApprover],
                ['status', UserApproverStatus::APPROVED]
            ])->count();

            if ($userApproverLength == $approveUserApproverStatus) {
                $numberOfApprovers = UserApprover::where([
                    ['file_approver_id', $fileApprover->id],
                ])->get()->count();
                $numberOfApproveStatus = UserApprover::where([
                    ['file_approver_id', $fileApprover->id],
                    ['status', $request->status]
                ])->get()->count();

                if ($numberOfApprovers <= $numberOfApproveStatus) {
                    $fileApprover->status = FileApproverStatus::APPROVED;
                    $fileApprover->save();

                    $numberOfFileApprover = FileApprover::where([
                        ['approval_id', $fileApprover->approval_id],
                    ])->get()->count();

                    $numberOfFileApprove = FileApprover::where([
                        ['approval_id', $fileApprover->approval_id],
                        ['status', FileApproverStatus::APPROVED],
                    ])->get()->count();

                    if ($numberOfFileApprover <= $numberOfFileApprove || $fileApprover->approve == 2) {
                        $approval = Approval::where('id', $fileApprover->approval_id)->first();
                        $approval->status = ApprovalStatus::APPROVED;
                        $approval->save();

                        $approval = Approval::where('id', $fileApprover->approval_id)->with(['file'])->first();

                        FileVersion::where([
                            ['file_id', $approval->file->file_id],
                            ['id', '!=', $approval->file_version_id]
                        ])->update([
                            'is_obsolete' => true
                        ]);


                        // needs another set of notification messages when document is approved as a whole.
                        // NotificationMessage::notifyReceiver('Approved', $request->receiver_id, $approval->id);
                    } else if ($fileApprover->approve == 1) {
                        $fileApprover = FileApprover::where('id', $request->file_approver_id)->first();
                        $fileApprover->is_approving = false;
                        $fileApprover->save();

                        $nextFileApprover = FileApprover::where([
                            ['approval_id', $fileApprover->approval_id],
                            ['sequence', $fileApprover->sequence + 1]
                        ])->with(['approval'])->first();
                        $nextFileApprover->is_approving = true;
                        $nextFileApprover->save();
                        UserApprover::where('file_approver_id', $nextFileApprover->id)->update([
                            'status' => UserApproverStatus::IN_PROGRESS
                        ]);
                        $nextUserApprover = UserApprover::where('file_approver_id', $nextFileApprover->id)->get();
                        foreach ($nextUserApprover as $user) {
                            NotificationMessage::notifySenderAndReceiver('Approval', $user['user_id'], Auth::id(), $user['id']);
                        }
                    }
                }
                NotificationMessage::notifySenderAndReceiver('Approved', $userApprover->fileApprover['approval']['user_id'], Auth::id(), $userApprover->id);
            }
        } else if ($request->status == UserApproverStatus::REVIEWED) {
            NotificationMessage::notifySenderAndReceiver('Reviewed', $userApprover->fileApprover['approval']['user_id'], Auth::id(), $userApprover->id);
        }

        $logType = '';

        switch ($request->status) {
            case UserApproverStatus::DECLINED:
                $logType = 'file_approval_decline';
                break;
            case UserApproverStatus::APPROVED:
                $logType = 'file_approval_approve';
                break;
            default:
                $logType = 'file_approval_review';
                break;
        }

        $fileName = $fileApprover->approval->file->file_name;

        $message = Auth::user()->name . ' has reviewed the file ' . $fileName . ' submitted by Developer Account';

        $this->logCustomMessage(
            $logType,
            $userApprover,
            $message,
            $userApprover,
            ApprovalLogType::REVIEW,
            new Activity()
        );
    }

    public function viewApprovalMessage(Request $request)
    {
        return UserApprover::where('user_id', Auth::id())->whereHas('fileApprover', function ($query) use ($request) {
            $query->whereHas('approval', function ($query) use ($request) {
                $query->where([['file_version_id', $request->file_version_id], ['user_id', Auth::id()]]);
            });
        })->with(['fileApprover.approval'])->first();
    }
}
