<?php

namespace Suiterus\Dms\Controllers\ApprovalTemplate;

use Activity;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Enums\Log\AdminApprovalTemplate as LogAdminApprovalTemplate;
use Suiterus\Dms\Models\ApprovalTemplate\AdminApprovalTemplate;

class AdminApprovalTemplateController extends Controller
{

    use HasCustomLogs;

    public function create(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'template_name' => 'required',
            'template' => 'required',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }
        DB::connection(env('DMS_DB_CONNECTION'))->beginTransaction();
        try {
            $old = AdminApprovalTemplate::find($request->id);
            $template = AdminApprovalTemplate::updateOrCreate(
                [
                    'id' => $request->id
                ],
                [
                    'user_id' => Auth::id(),
                    'name' => $request->template_name,
                    'template' => json_encode($request->template),
                ]
            );
            $updated = AdminApprovalTemplate::find($request->id);
            if ($request->id) {
                $template->old = collect($old);
                $template->attributes = collect($updated);
                $this->logCustomMessage(
                    'update_admin_approval_template',
                    $template,
                    Auth::user()->name . ' updated an approval template named ' . $template->name,
                    $template,
                    LogAdminApprovalTemplate::UPDATE,
                    new Activity()
                );
            }else {
                $this->logCustomMessage(
                    'create_admin_approval_template',
                    $template,
                    Auth::user()->name . ' created an approval template named ' . $template->name,
                    $template,
                    LogAdminApprovalTemplate::CREATE,
                    new Activity()
                );
            }

            DB::connection(env('DMS_DB_CONNECTION'))->commit();
        } catch (Exception $e) {
            DB::connection(env('DMS_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                'msg' => $e->getMessage(),
            ], 500);
        }
    }


    public function paginate(Request $request)
    {

        if ($request->name) {
            $this->logCustomMessage(
                'search_admin_approval_template',
                null,
                Auth::user()->name . ' searched for approval template named "' . $request->name . '"',
                null,
                LogAdminApprovalTemplate::SEARCH,
                new Activity()
            );
        }

        return AdminApprovalTemplate::when($request->user_id, function ($query) use ($request) {
            $query->where('user_id', $request->user_id);
        })->when($request->name, function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->name . '%');
        })->paginate(!$request->is_only_three ? env('DEFAULT_PAGECOUNT') : 3);
    }

    public function updateStatus(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.admin_approval_templates,id',
            'status' => 'required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::connection(env('DMS_DB_CONNECTION'))->beginTransaction();
        try {
            $old = AdminApprovalTemplate::find($request->id);
            AdminApprovalTemplate::where([
                ['id', $request->id]
            ])->update([
                'status' => $request->status
            ]);
            $updated = AdminApprovalTemplate::find($request->id);

            $updated->old = collect($old);
            $updated->attributes = collect($updated);

            $isActive = $request->status == 1 ? 'active' : 'inactive';

            $this->logCustomMessage(
                'update_admin_approval_template_status',
                $updated,
                Auth::user()->name . ' marked "' . $updated->name . '" as ' . $isActive,
                $updated,
                LogAdminApprovalTemplate::UPDATE_STATUS,
                new Activity()
            );            

            DB::connection(env('DMS_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  'Template status updated.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('DMS_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating new Template.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function search(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return AdminApprovalTemplate::where('name', 'LIKE', '%' . $request->name . '%')->paginate($paginate);
    }

    public function delete(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'ids' => 'required'
        ]);

        if ($valid->fails()) {
            return response()->json([
                $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            foreach ($request->ids as $id) {
                $template = AdminApprovalTemplate::find($id);

                $this->logCustomMessage(
                    'delete_admin_approval_template',
                    $template,
                    Auth::user()->name . ' deleted an approval template named ' . $template->name,
                    $template,
                    LogAdminApprovalTemplate::DELETE,
                    new Activity()
                );
            }

            AdminApprovalTemplate::whereIn('id', $request->ids)->delete();
            DB::commit();

            return response()->json([
                'text'  =>  'Template deleted.'
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['There is a problem in delete a template.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }
}
