<?php

namespace Suiterus\Dms\Controllers\AccessManagement;

use Activity;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\AccessManagement\GroupManagement\Role;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Dms\Models\Files\File;
use Suiterus\Dms\Models\Files\FileAccess;

class FileAccessController extends Controller
{

    use HasCustomLogs;

    public function create(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'file_id' => 'required|exists:' . env('DMS_DB_CONNECTION') . '.files,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'message' => $valid->errors()
            ], 400);
        }

        DB::connection(env('DMS_DB_CONNECTION'))->beginTransaction();
        try {

            $fileRelation = ['file' => function ($query) {
                $query->without([
                    'author',
                    'modifier',
                    'extraFields',
                    'access',
                    'versions',
                    'documentType'
                ]);
            }];

            $members = FileAccess::with($fileRelation)->where([
                ['file_id', $request->file_id],
                ['group_id', '==', null]
            ])->get()->pluck('user_id')->toArray();

            $groups = FileAccess::with($fileRelation)->where([
                ['file_id', $request->file_id],
                ['user_id', '==', 0]
            ])->get()->pluck('group_id')->toArray();

            $deletedMembers = FileAccess::with($fileRelation)->whereNotIn('user_id', $request->users ?? [])->where([
                ['group_id', '==', null],
                ['file_id', $request->file_id],
                ['user_id', '!=', Auth::id()]
            ])->get();

            $deletedGroups = FileAccess::with($fileRelation)->whereNotIn('group_id', $request->groups ?? [])->where([
                ['group_id', '!=', 1],
                ['file_id', $request->file_id],
                ['user_id', '!=', Auth::id()]
            ])->where('user_id', '==', 0)->get();

            foreach ($deletedMembers as $member) {
                $this->logCustomMessage(
                    'revoke_user_file_access',
                    $member,
                    'Access to file "' . $member['file']['name'] . '" was revoked for ' . $member['user']['name'] . ' by ' . Auth::user()->name,
                    $member,
                    'Revoke an access to a file',
                    new Activity()
                );
            }

            foreach ($deletedGroups as $group) {
                $this->logCustomMessage(
                    'revoke_group_file_access',
                    $group,
                    'Access to file "' . $group['file']['name'] . '" was revoked for ' . $group['group']['name'] . ' by ' . Auth::user()->name,
                    $group,
                    'Revoke an access to a file',
                    new Activity()
                );
            }

            FileAccess::where([
                ['file_id', $request->file_id],
                ['user_id', '!=', Auth::id()]
            ])->delete();

            if (!$request->is_disabled) {
                foreach ($request->users as $user) {
                    $fileAccess = FileAccess::create([
                        'file_id' => $request->file_id,
                        'user_id' => $user,
                        'group_id' => 0,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                    if (!in_array($user, $members) && $user != Auth::id()) {
                        $fileAccess = FileAccess::with($fileRelation)->find($fileAccess->id);
                        $this->logCustomMessage(
                            'grant_user_file_access',
                            $fileAccess,
                            $fileAccess['user']['name'] . ' was granted access to file "' . $fileAccess['file']['name'] . '" by ' . Auth::user()->name,
                            $fileAccess,
                            'Grant an access to a file',
                            new Activity()
                        );
                    }
                }

                foreach ($request->groups as $group) {
                    $fileAccess = FileAccess::create([
                        'file_id' => $request->file_id,
                        'user_id' => 0,
                        'group_id' => $group,
                        'access_level' => 2,
                        'shared_by' => Auth::id(),
                        'shared_date' => now()
                    ]);
                    if (!in_array($group, $groups)) {
                        $fileAccess = FileAccess::with($fileRelation)->find($fileAccess->id);
                        $this->logCustomMessage(
                            'grant_group_file_access',
                            $fileAccess,
                            $fileAccess->group->name . ' was granted access to file "' . $fileAccess->file->name . '" by ' . Auth::user()->name,
                            $fileAccess,
                            'Grant an access to a file',
                            new Activity()
                        );
                    }
                }
            }

            DB::connection(env('DMS_DB_CONNECTION'))->commit();
            return response()->json([
                'message' => 'Success'
            ], 200);
        } catch (Exception $e) {
            DB::connection(env('DMS_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning internal access.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function fetchGroupByFileId(Request $request)
    {
        return Role::whereHas('fileAccess', function ($query) use ($request) {
            $query->where([
                ['file_id', $request->file_id],
            ]);
        })->has('roleHasSystem')->get();
    }

    public function fetchUserByFileId(Request $request)
    {
        return User::whereHas('file_access', function ($query) use ($request) {
            $query->where([
                ['file_id', $request->file_id],
            ]);
        })->get();
    }

    public function listSharedFiles(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        $authUserRoles = Auth::user()->roles->pluck('id');
        return File::where('user_id', '!=', Auth::id())->whereHas('access', function ($query) use ($authUserRoles) {
            $query->where('user_id', Auth::id())->orWhereIn('group_id', $authUserRoles);
        })->without(['access', 'versions', 'extraFields'])->with(['sharedBy'])->orderBy('name', 'ASC')->orderBy('created_at', 'DESC')->paginate($paginate);
    }
}
