<?php

namespace Suiterus\Adg\Traits;

/**
 * This function encapsulates basic information about an employee from the provided object.
 * It initializes the employee's info, such as name, gender, address, and basic salary, from the given object.
 * @param Object $employee - @see User.php model
 * NOTE: When fetching user collection, please add the following to your query to avoid looping when querying users:
 * "->with('employeeExtraField','salary.publicSalary.salaryGrade','salary.privateSalary')->without('salary.user')"
 */

trait EmployeeTrait
{
    protected ?String $address = "";
    protected ?String $last_name = "";
    protected ?String $first_name = "";
    protected ?String $middle_name = "";
    protected ?String $suffix = "";
    protected ?String $date_of_birth = "";
    protected ?String $gender = "";
    protected ?float $basic_salary = 0.00;
    protected ?String $employee_id = "";
    protected $employee_meta_info = [];

    public function initializeEmployeeInfo($employee)
    {
        $eefc = $employee->employeeExtraField->pluck('EmployeeExtraFieldColumn')->flatten();
        if ($eefc->isNotEmpty()) {

            // assign values to return
            $this->last_name = $eefc->where('field_name', "last_name")->pluck('field_value')->first();
            $this->first_name = $eefc->where('field_name', "first_name")->pluck('field_value')->first();
            $this->middle_name = $this->setMiddleName($eefc->where('field_name', "middle_name")->pluck('field_value')->first());
            $this->date_of_birth = $eefc->where('field_name', "date_of_birth")->pluck('field_value')->first();
            $this->suffix = $eefc->where('field_name', "name_extension")->pluck('field_value')->first() == "N/A" ? "" : $eefc->where('field_name', "name_extension")->pluck('field_value')->first();
            $this->basic_salary = 0.00;
            if ($employee->salary) {
                $value = $employee->salary->privateSalary ? $employee->salary->privateSalary->salary : $employee->salary->publicSalary->salaryGrade->value;
                $this->basic_salary = floatval(str_replace(",", "", $value));
            }
            $this->setGender($eefc);
            $this->setAddress($employee);
            $this->employee_id = $employee->employeeMetaInfo->employee_id;
            $this->employee_meta_info = $employee->employeeMetaInfo;
        } else {
            $this->address = "";
            $this->last_name = "";
            $this->first_name = "";
            $this->middle_name = "";
            $this->suffix = "";
            $this->date_of_birth = "";
            $this->gender = "";
            $this->basic_salary = 0.00;
            $this->employee_id = "";
        }
    }

    public function getEmployeeMetaInfo(){
        return $this->employee_meta_info;
    }

    public function getFullName()
    {
       return $this->last_name . ', ' . $this->first_name . ' ' . $this->middle_name . ' ' . $this->suffix;
    }

    private function setGender($eefc)
    {
        $binary =  $eefc->where('field_name', "sex")->pluck('field_value')->first();
        $non_binary =  $eefc->where('field_name', "sex_details")->pluck('field_value')->first();
        if ($binary) {
            $this->gender = $binary;
        } else if ($non_binary) {
            $this->gender = $non_binary;
        }
    }

    private function setAddress($user)
    {
        $residential_address =  $user->employeeExtraField
            ->where('table_name', 'residential_address')
            ->pluck('employeeExtraFieldColumn')
            ->flatten()
            ->pluck('field_value')
            ->toArray();
        $address = implode(" ", array_filter($residential_address, fn ($value) => $value !== "N/A"));
        $this->address = $address;
    }

    public function setMiddleName($middle_name = ''){
            if($middle_name){
                if($middle_name == 'N/A'){
                    $middle_name = '';
                }
            }

            return $middle_name;
    }
}
