<?php

namespace Suiterus\Adg\Services;

use App\Enums\AttendanceLegend;
use App\Enums\AttendanceType;
use App\Enums\ScheduleBasis;
use App\Enums\Status;
use App\Enums\ZKTeco\ZKTecoPunchType;
use App\Exceptions\BiometricConnectException;
use App\Models\User;
use Suiterus\Adg\Models\Timekeeping\BiometricRfidUser;
use Carbon\Carbon;
use ErrorException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Rats\Zkteco\Lib\Helper\Util;
use Rats\Zkteco\Lib\ZKTeco;
use Suiterus\Adg\Models\SM\Shift;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Models\Timekeeping\Fingerprint;
use App\Enums\Biometrics\BiometricType;
use Illuminate\Support\Facades\Http;

/**
 * Service used for the integrated functions of NKTI - HRIS and ZKTeco device
 * The library used for the integration is rats/zkteco - https://github.com/raihanafroz/zkteco
 * Biometrics, RFID/Proximity, and Attendance integration are included in the library
 */

class ZKTecoService {

    private $zk;

    public function __construct($ip = null, $port = 4370){
        $this->zk = new ZKTeco($ip, $port);
    }

    /* ----------------- User Details Functions ------------------- */

    /**
     * Create new user with biometrics and/or proximity in the device through the Biometrics/RFID module
     * This may also include updating the user's details if exists in the device
     * Setting a user also includes setting the card number
     */
    public function setUser($user, $sync = false, $card_no = 0, $card_date_issued = null, $unique_id = null) {

        $user_record = BiometricRfidUser::where('user_id', $user->id)->first();
        $user_id = null;

        // Check if record exists in the local database
        if($user_record === null) {
            $card_date_issued = $card_date_issued == null ? date('Y-m-d') : $card_date_issued;

            
            if($unique_id == null || $unique_id == '') {
                $unique_id = $user->employeeMetaInfo->employee_id;
            }

            // Get all the users from the device and sort to get the last id
            
            // Make the latest id
            $user_id = $user->id;
            $user_record = BiometricRfidUser::create([
                'user_id'           => $user->id,
                'device_user_id'    => $user_id,
                'unique_id'         => $unique_id,
            ]);
            $user_record->card()->create([
                'rfid'  => $card_no,
                'card_number' => $card_no == 0 ? null : $card_no,
                'status' => Status::ACTIVE,
                'date_issued' => $card_date_issued,
            ]);
            $user_record->biometrics()->updateOrCreate(
                [
                    'type' => BiometricType::FINGERPRINT, 
                    'biometric_number' => 0, 
                    'status' => 1, 
                    'date_issued' => date('Y-m-d'),
                    'created_by' => Auth::id()
                ]
            );
        } else {
            $user_id = $user_record->device_user_id;
            $unique_id = $unique_id ? $unique_id : $user_record->unique_id;
            $user_record->card->card_number = $card_no == 0 ? null : $card_no;
            $user_record->card->date_issued = $card_date_issued;
            $user_record->card->save();
        }
        
        if($sync) {
            $this->zk->setUser($user_id, $unique_id, $user->name, '', Util::LEVEL_USER, $card_no);
        }

        return true;
    }

    /**
     * Re-set user from system to device with same details
     * Used for re-syncing of details from system to device
     */
    public function reSetUser($user) {
        $device_details = $user->biometric_rfid_record; 
        $this->zk->setUser($device_details->device_user_id, $device_details->unique_id, $user->name, '', Util::LEVEL_USER, intval(rtrim($device_details->card->card_number)));
    }

    /**
     * Get the user from device by unique id and save the device_user_id by comparing
     */
    public function syncUserFromDevice() {
        $users = $this->zk->getUser();
        $users = array_values($users);

        foreach($users as $user) {
            $biometric_user = BiometricRfidUser::where('unique_id', $user['userid'])->first(); 
            if(!$biometric_user) {
                $system_emp_id = $user['userid']; // device user ID should be EMPLOYEE ID in system to be synced
                $employee = User::whereHas('employeeMetaInfo', function ($query) use ($system_emp_id) {
                    $query->where('employee_id',$system_emp_id);
                })->first();

                if($employee){
                    $unique_id = $employee->employeeMetaInfo->employee_id;
                    $user_record = BiometricRfidUser::create([
                        'user_id'           => $employee->id,
                        'device_user_id'    => $user['uid'],
                        'unique_id'         => $unique_id,
                    ]);
                    $user_record->card()->create([
                        'rfid'  => $user['cardno'] == '0000000000 ' ? 0 : $user['cardno'],
                        'card_number' => $user['cardno'] == '0000000000 ' ?  null : $user['cardno'],
                        'status' => Status::ACTIVE,
                        'date_issued' => date('Y-m-d'),
                    ]);
                    $user_record->biometrics()->updateOrCreate(
                        [
                            'type' => BiometricType::FINGERPRINT, 
                            'biometric_number' => 0, 
                            'status' => 1, 
                            'date_issued' => date('Y-m-d'),
                            'created_by' => Auth::id()
                        ]
                    );
                    //comment this line of code for faster syncing from device.
                    //$this->zk->setUser($user_record->device_user_id, $unique_id, $employee->name, '', Util::LEVEL_USER, $user_record->card->card_number);
                }
                
            }else {
                $biometric_user->update([
                    'device_user_id' => $user['uid'],
                ]);
                $biometric_user->card->update([
                    'card_number'   => $user['cardno'] == '0000000000 ' ?  null : $user['cardno'],
                ]);
            };
           
        }

        return true;
    }

    /**
     * Get the fingerprint records from the device and synchronize with the HRIS database
     * Uses the getFingerprint function and finds the fingerprints of a selected user
     *   
     */
    public function getUserFingerprints($user) {

        $biometric_rfid_record = $user->biometric_rfid_record;

        $fingerprints = $this->zk->getFingerprint($biometric_rfid_record->device_user_id);

        $biometrics = $biometric_rfid_record->biometrics; 

        if($biometrics === null) {
            $biometrics = $biometric_rfid_record->biometrics()->create([
                'type' => BiometricType::FINGERPRINT, 
                'biometric_number' => 0, 
                'status' => 1, 
                'date_issued' => date('Y-m-d'),
                'created_by' => Auth::id()
            ]);
        }

        $found_fingers = [];
        foreach($fingerprints as $finger => $data) {
            $biometrics->fingerprints()->updateOrCreate(
                [
                    'finger_number' => $finger,
                ],
                [
                    'fingerprint_template' => $data, 
                ]
            );
            array_push($found_fingers, $finger);
        }

        $biometrics->fingerprints()->whereNotIn('finger_number', $found_fingers)->delete();
        return true;

    }

    /**
     * Sync fingerprints of users stored in the system to the device
     * For next development - currently not yet needed
     */
    public static function syncFingerprints($user, $token) {
        Http::zktecoFlask()->accept('application/json')
            ->withHeaders(['Authorization' => $token])
            ->post('/per-user-sync-finger',['id' => $user->id])
            ->throw()->json();
        /**
         * IMPORTANT NOTE: 
         * Syncing of fingerprints from system to device is possible, but using a different library
         * since this library does not support it yet. Please find a different library that supports
         * uploading fingerprints on the device. Codes for the library will only be needed to change 
         * here at this service/class. 
         */
    }

    public static function syncAllFingerprints($token){
        Http::zktecoFlask()->accept('application/json')
            ->withHeaders(['Authorization' => $token])
            ->post('/sync-all-finger')
            ->throw()->json();
    }

    /**
     * Synchronize the card number of the user from the device if ever changed
     * 
     */
    public function syncCard($user) {

        $biometric_rfid_record = $user->biometric_rfid_record;
        if ($biometric_rfid_record->card->card_number) {
            $this->zk->setUser($biometric_rfid_record->device_user_id, $biometric_rfid_record->unique_id, $user->name, '', Util::LEVEL_USER, intval(rtrim($biometric_rfid_record->card->card_number)));
        }
        $users = $this->zk->getUser();
        $users = array_values($users);
        
        $found_user_index = array_search($biometric_rfid_record->unique_id, array_column($users, 'userid'));

        $selected_user = $users[$found_user_index];
        $biometric_rfid_record->card->card_number = $selected_user['cardno']  == '0000000000 ' ? null : $selected_user['cardno'];
        $biometric_rfid_record->status = Status::ACTIVE;

        $biometric_rfid_record->card->save();

        return true;
    }

    /**
     * Set Card as inactive
     * This sets the user's card number to 0 to disable the card in the zkteco devicec
     */
    public function setCardInactive($user) {
        
        $biometric_rfid_record = $user->biometric_rfid_record;
        
        $user_id = $biometric_rfid_record->device_user_id;
        $unique_id = $biometric_rfid_record->unique_id;
        $this->zk->setUser($user_id, $unique_id, $user->name, '', Util::LEVEL_USER, 0);

    }

    /**
     * Set Card as active 
     * This gets the stored card number from the system database and passes it into the device
     */
    public function setCardActive($user) {

        $biometric_rfid_record = $user->biometric_rfid_record;

        $users = $this->zk->getUser();
        $users = array_values($users);
        
        $found_user_index = array_search($biometric_rfid_record->card->card_number, array_column($users, 'cardno'));

        $card_number = $biometric_rfid_record->card->card_number;

        if($card_number !== null && ($found_user_index !== false && $found_user_index !== null)) {
            $biometric_rfid_record->card->card_number = null;
            $biometric_rfid_record->card->save();
            throw new InvalidArgumentException('Oops! A similar card number was registered in the system for the user. Please try registering a new card and syncing it again with the system.');
        }

        $user_id = $biometric_rfid_record->device_user_id;
        $unique_id = $biometric_rfid_record->unique_id;
        $this->zk->setUser($user_id, $unique_id, $user->name, '', Util::LEVEL_USER, intval($card_number));

    }

    /*------------------ Attendance Functions -----------------------*/

    /**
     * Get attendance records from device and sync with own database
     */
    public function syncAttendance($device_name = null)
    {

        $this->disable();
        $attendance_records = $this->zk->getAttendance();
        $attendance_records = array_values($attendance_records);

        // Loop through the records first
        // Find a clock-in first first the employee and record it as a new attendance record in the system
        // If there are multiple clock-ins found simultaneously, only record the first one
        // If a clock out is found, record it as clock out for the current attendance record without a clock-out

        $user_attendance = [];
        foreach ($attendance_records as $attendance) {
            $user_attendance[$attendance['id']][] = $attendance;
        }

        $count = 0;
        foreach ($user_attendance as $key => $user_record) {
            $user = BiometricRfidUser::where('unique_id', $key)->first();
            if ($user === null) {
                continue;
            }
            $user = $user->user;
            foreach ($user_record as $record) {
                /** Case scenario:
                 * 1. Multiple Check-ins is accepted as long as new check-in is greater than 12 hours on scheduled out of existing clock in
                 * 2. break in and break out is accepted and nullable
                 */

                //logic to avoid multiple punch and avoid wrong timestamp like when device time is not sync realtime
                $last_timestamp = $user->attendance()->latest('time_in')->first();
                $current_timestamp = Carbon::parse($record['timestamp']);
                $last_timestamp = $last_timestamp ? $this->getLastTimeStamp($last_timestamp) : null;
                if ($last_timestamp && $current_timestamp->lte($last_timestamp->copy()->addMinutes(5))) {
                    continue;
                }

                $attendance = $user->attendance()->where('time_out', null)->latest('time_in')->first();
                if ($attendance) {
                    if ($attendance->timetable_basis || $attendance->shift) {
                        $this->recordWithSchedule($user, $attendance, $record, $device_name);
                    } else {
                        $this->recordWithoutSchedule($user, $attendance, $record, $device_name);
                    }
                    continue;
                }

                $this->recordTimeIn($user, $record['timestamp'], $record['state'], $device_name);
            }
            $count++;
        }

        $datetime = date('Y-m-d H:i:s');
        echo "$datetime - Attendance synced for $count employees\n";

        $this->zk->clearAttendance();
        $this->enable();
    }

    private function getLastTimeStamp($attendance) {
        if ($attendance->time_out) {
            return Carbon::parse($attendance->time_out);
        } 
        if ($attendance->break_out) {
            return Carbon::parse($attendance->break_out);
        }
        if ($attendance->break_in) {
            return Carbon::parse($attendance->break_in);
        }
        if ($attendance->time_in) {
            return Carbon::parse($attendance->time_in);
        }
    }

    /* ---------------- Device Power Functions ---------------- */

    // Check device connection
    public function checkDeviceConnection() {
        try{
            fsockopen($this->zk->_ip, $this->zk->_port, $errorno, $errorstring, 1);
            if($errorno !== 0) {
                throw new BiometricConnectException(__('responses.zkteco.connection-failed'));
            }
        } catch (ErrorException $error){
            return false;
        }
        return $this->zk->connect();
    }

    // Disconnect Device
    public function disconnect() {
        $this->zk->disconnect();
    }

    // Shut down device
    public function shutdown() {
        $this->checkDeviceConnection();
        $this->zk->shutdown();
        $this->zk->disconnect();
    }
    
    // Restart Device
    public function restart() {
        $this->checkDeviceConnection();
        $this->zk->restart();
        $this->zk->disconnect();
    }

    // Sleep
    public function sleep() {
        $this->checkDeviceConnection();
        $this->zk->sleep(); 
        $this->zk->disconnect();
    }

    // Test Voice
    public function testVoice() {
        $this->checkDeviceConnection();
        $this->zk->testVoice(); 
        $this->zk->disconnect();
    }

    public function disable() {
        $this->zk->disableDevice(); 
    }

    public function enable() {
        $this->zk->enableDevice(); 
    }
    /* ---------------- Data Functions --------------------- */

    public function syncTime() {
        $this->enable(); 
        $this->zk->setTime(Carbon::now()->format('Y-m-d H:i:s'));
        $this->disable(); 
    }

    private function recordWithSchedule($user, $attendance, $record, $device_name) {
        $time_in = $attendance->time_in;
        $time_out = $attendance->time_out;
        $break_in = $attendance->break_in;
        $break_out = $attendance->break_out;
        $time_stamp = Carbon::parse($record['timestamp']);

        $scheduled_time_in = Carbon::parse($attendance->scheduled_time_in)->setDateFrom($time_in);
        $scheduled_time_out = Carbon::parse($attendance->scheduled_time_out)->setDateFrom($time_in);
        $scheduled_break_in = null;
        $scheduled_break_out = null;

        if ($scheduled_time_out->lt($scheduled_time_in)) {
            //Add day if time_out crossed midnight
            $scheduled_time_out->addDay();
        }

        if ($time_stamp->lte(Carbon::parse($time_in))) {
            return;
        }
        //march 5 2024 6:00pm + 12 hours == 5am march 6  is greater than march 4 2024 3am
        if ($time_stamp->gte($scheduled_time_out->copy()->addHours(12))) {
            $this->recordTimeIn($user, $record['timestamp'], $record['state'], $device_name);
            return;
        }

        $total_hours = $scheduled_time_in->diffInHours($scheduled_time_out);
        if ($total_hours <= 4) {
            $this->recordTimeOut($user, $attendance, $record['timestamp'], $record['state'], $device_name);
            return;
        }
        // find the midpoint for break time entries;
        $result = $scheduled_time_in->copy()->addSeconds($scheduled_time_in->diffInSeconds($scheduled_time_out) / 2);

        if ($attendance->scheduled_break_in && $attendance->scheduled_break_out) {
            $scheduled_break_in = Carbon::parse($attendance->scheduled_break_in)->setDateFrom($time_in);
            $scheduled_break_out = Carbon::parse($attendance->scheduled_break_out)->setDateFrom($time_in);
            if ($scheduled_break_out->lt($scheduled_break_in)) {
                $scheduled_break_out->addDay();
            }
            $result = $scheduled_break_out;
        }

        if ($break_in == null && $break_out == null && $time_out == null && $time_stamp <= $result) {
            $this->recordBreakIn($user, $attendance, $record['timestamp'], $record['state'], $device_name);
        } elseif ($break_in && $break_out == null && $time_out == null) {
            $this->recordBreakOut($user, $attendance, $record['timestamp'], $record['state'], $device_name);
        } elseif ($time_out == null) {
            $this->recordTimeOut($user, $attendance, $record['timestamp'], $record['state'], $device_name);
        }
    }

    private function recordWithoutSchedule($user, $attendance, $record, $device_name) {
        $time_in = Carbon::parse($attendance->time_in);
        $time_stamp = Carbon::parse($record['timestamp']);
        //record as time-in if the last attendance's time in is greater than 12 hours

        if ($time_stamp->lte($time_in)) {
            return; 
        }

        if ($time_stamp->gte($time_in->copy()->addHours(12))) {
            $this->recordTimeIn($user, $record['timestamp'], $record['state'], $device_name);
            return;
        }

        $this->recordTimeOut($user, $attendance, $record['timestamp'], $record['state'], $device_name);
    }

    /**
     * Record the time-in
     */
    private function recordTimeIn($user, $time_in, $state, $device_name) {

        // Check for temporary schedule
        $employee_schedule = $user->temporary_schedule;
        $roster_shift = $this->getRosterShift($user, Carbon::parse($time_in)->format('Y-m-d'));
        // If no temporary schedule, check if has active main schedule
        if($roster_shift === null) {
            if ($employee_schedule === null) {
                $employee_schedule = $this->checkActiveSchedule($user);
            }
        }

        // Get the schedule and timetable record for the attendance of the employee
        // Compare the dates to check which in the schedule template is the current date
        $day_of_week = date('l');
        $schedule = $employee_schedule === null ? null : $employee_schedule->schedule_template()->where('day', $day_of_week)->first();

        // Get the timetable where the time-in is within the timetable range
        $time_in_time = strtotime($time_in);
        $sched_timetable = $schedule !== null ? ($schedule->timetables()->whereTime('time_out', '>=', date('H:i:s', $time_in_time))->orderBy('time_in', 'asc')->first()) : null;

        $attendance = Attendance::create([
            "user_id"       => $user->id,
            "date_in"       => date('Y-m-d', strtotime($time_in)),
            "total"         => 0,
            "time_in"       => $time_in,
            'device_in' => $device_name,
            'schedule_day_id' => $schedule !== null ? $schedule->id : null, 
            'timetable_id'  => $sched_timetable !== null ? $sched_timetable->id : null,
            'shift_id'      => $roster_shift !== null ? $roster_shift->id : null,
            "type"          => AttendanceType::EMPLOYEE,
            'time_in_state' => $state,
            "created_by"    => $user->id,
            "updated_by"    => $user->id,
        ]);

        if ($attendance->scheduled_max_time_in) {
            if ($this->checkIfLate($time_in, $attendance->scheduled_max_time_in)) {
                $attendance->legends()->create([
                    'legend' => AttendanceLegend::LATE
                ]);
            }
        }
    }

    private function recordBreakIn($user, $attendance, $timestamp, $state, $device_name)
    {
        $attendance->update([
            'break_in' => $timestamp,
            'updated_by' => $user->id,
            'device_break_in' => $device_name,
            'break_in_state' => $state
        ]);

        if ($attendance->scheduled_break_in) {
            if ($this->checkIfUnderTime($timestamp, $attendance->scheduled_break_in)) {
                $attendance->legends()->create([
                    'legend' => AttendanceLegend::UNDERTIME
                ]);
            }
        }
    }

    private function recordBreakOut($user, $attendance, $timestamp, $state, $device_name)
    {
        $attendance->update([
            'break_out' => $timestamp,
            'updated_by' => $user->id,
            'device_break_out' => $device_name,
            'break_out_state' => $state
        ]);

        if ($attendance->scheduled_break_out) {
            if ($this->checkIfLate($timestamp, $attendance->scheduled_break_out)) {
                $attendance->legends()->create([
                    'legend' => AttendanceLegend::LATE
                ]);
            }
        }
    } 

    /**
     * Record the time-in
     */
    private function recordTimeOut($user, $attendance, $time_out, $state, $device_name) {
        // Compute for the total time between the in and out
        $total = Carbon::parse($time_out)->diff($attendance->time_in)->format('%H:%I');
        $break_in = null;
        $break_out = null;

        if ($attendance->scheduled_time_out) {
            if ($this->checkIfUnderTime($time_out, $attendance->scheduled_time_out)) {
                $attendance->legends()->create([
                    'legend' => AttendanceLegend::UNDERTIME
                ]);
            }
        }

        if ($attendance->break_in && $attendance->break_out) {

            $break_in = Carbon::parse($attendance->break_in);
            $break_out = Carbon::parse($attendance->break_out);

            $total = Carbon::parse($total);
            $hours_diff = $break_in->diffInSeconds($break_out);
            $hours_diff = Carbon::parse(gmdate('H:i', $hours_diff));
            $total->subHours($hours_diff->format('H'));
            $total->subMinutes($hours_diff->format('i'));
            $total = $total->format('G:i');

        } elseif ($attendance->scheduled_break_in && $attendance->scheduled_break_out) {

            $break_in = $attendance->scheduled_break_in ? Carbon::parse($attendance->scheduled_break_in)->setDateFrom($attendance->time_in) : null;
            $break_out = $attendance->scheduled_break_out ? Carbon::parse($attendance->scheduled_break_out)->setDateFrom($attendance->time_in) : null;
            $attendance_time_in = Carbon::parse($attendance->time_in);

            if ($break_out->lt($break_in)) {
                $break_out->addDay();
            }

            if ($attendance_time_in->lte($break_in->copy()->subHour()) && Carbon::parse($time_out)->gte($break_out->copy()->addHour())) {
                $total = Carbon::parse($total);
                $hours_diff = $break_in->diffInSeconds($break_out);
                $hours_diff = Carbon::parse(gmdate('H:i', $hours_diff));
                $total->subHours($hours_diff->format('H'));
                $total->subMinutes($hours_diff->format('i'));
                $total = $total->format('G:i');              
            }
        }

        $attendance->update([
            'total'         => $total,
            'time_out'      => $time_out,
            'device_out' => $device_name,
            'time_out_state'=> $state,
            'updated_by'    => $user->id,
        ]);
    }

    private function checkIfLate($time1, $time2)  {
        $time1 = strtotime(date('H:i:s', strtotime($time1)));
        $time2 = strtotime($time2);

        if (strtotime($time1) > strtotime($time2)) {
            return true;
        }
        return false;
    }

    private function checkIfUnderTime($time1, $time2) {
        $time1 = strtotime(date('H:i:s', strtotime($time1)));
        $time2 = strtotime($time2);

        if (strtotime($time1) < strtotime($time2)) {
            return true;
        }
        return false;
    }
    /**
     * internal function that checks if the employee has an active schedule
     */
    private function checkActiveSchedule($user) {
        $today = date('Y-m-d');
        $employee_schedule = $user->employeeSchedules()->without('user')->first();

        if($employee_schedule === null) {
            return null;
        }
        return $employee_schedule->schedule_template;
    }

    private function getRosterShift($user, $currentdate) {
        $shift = Shift::whereHas('employeeShift',function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user){
                $query->where('user_id',$user->id);
            });
        })
        ->orWhereHas('headEmployeeShift', function($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id);
            });
        })->first();
    
        return $shift;
    }
}