<?php 

namespace Suiterus\Adg\Services\TalentAcquisition;

use Carbon\Carbon;
use App\Enums\Status;
use App\Enums\ClearanceStatus;
use App\Models\User;
use Exception;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\Promotion\PromotionApplication;
use Suiterus\Adg\Models\Reports\CostPerHireReports;
use Suiterus\Adg\Models\Reports\TimeSinceLastPromotionReports;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Hrjp\Models\Position_has_salary;
use Suiterus\Hrjp\Models\Application;
use Suiterus\Hrjp\Models\ItemCode;

class TalentAcquisitionService
{
    public function fetchNewHireTurnoverRate($startDate, $endDate, $keyword) 
    {
        $inactiveUsers = $this->fetchNewHireTurnoverRateInactiveUsers($startDate, $endDate);
        $activeUsers = $this->fetchNewHireTurnoverRateActiveUsers($startDate, $endDate);
        
        $mergedRecords = $inactiveUsers->merge($activeUsers);

        if ($keyword) {
            $mergedRecords = $this->searchNewHireTurnoverRate($mergedRecords, $keyword);
        }

        $data = $this->newHireTurnoverRateGroupByDepartmentAndDivision($mergedRecords);
        
        return $data;
    }

    public function fetchNewHireTurnoverRateInactiveUsers($startDate, $endDate)
    {
        return EmployeeMetaInfo::whereHas('user', function ($query) use ($startDate, $endDate) {
            $query->whereHas('clearance', function ($query) use($startDate, $endDate){
                $query->where('status', ClearanceStatus::APPROVED)
                ->whereBetween('start_date', [$startDate, $endDate]);
            });
        })->whereHas('user', function ($query) {
            $query->where('status', Status::INACTIVE);
        })
        ->with('user', function ($query) {
            $query->without('roles', 'permissions', 'supervisor', 'employeeMetaInfo', 'storage', 'user_supervisor');
        })->get();
    }

    public function fetchNewHireTurnoverRateActiveUsers($startDate, $endDate)
    {
        return EmployeeMetaInfo::whereHas('user', function ($query) use($startDate, $endDate) {
            $query->where('status', Status::ACTIVE)
            ->whereBetween('date_hired', [$startDate, $endDate]);
        })->with('user', function ($query) {
            $query->without('roles', 'permissions', 'supervisor', 'employeeMetaInfo', 'storage', 'user_supervisor');
        })->get();
    }

    private function newHireTurnoverRateGroupByDepartmentAndDivision($records)
    {
        return $records->groupBy(function ($record) {
            return $record->department->name ?? 'Unknown Department';
        })->map(function ($departmentRecords, $departmentName) {
            return $departmentRecords->groupBy(function ($record) {
                return $record->division->name ?? 'Unknown Division';
            })->map(function ($divisionRecords, $divisionName) use ($departmentName) {
                $countInactiveUser = $divisionRecords->where('user.status', Status::INACTIVE)->count();
                $countActiveUser = $divisionRecords->where('user.status', Status::ACTIVE)->count();
                
                $total = ($countActiveUser == 0) ? '0' : number_format(($countInactiveUser / $countActiveUser) * 100, 2) . '%';
    
                return [
                    'department' => $departmentName,
                    'division' => $divisionName,
                    'inactive_users' => $countInactiveUser,
                    'active_users' => $countActiveUser,
                    'total' => $total,
                ];
            });
        });
    }

    private function searchNewHireTurnoverRate($records, $keyword)
    {
        return $records->filter(function ($record) use ($keyword) {
            return stripos($record['department'], $keyword) !== false
                || stripos($record['division'], $keyword) !== false;
        });
    }

    public function countNewHireTurnoverRateInactiveUsers($startDate, $endDate)
    {
        return EmployeeMetaInfo::whereHas('user', function ($query) use ($startDate, $endDate) {
            $query->whereHas('clearance', function ($query) use($startDate, $endDate){
                $query->where('status', ClearanceStatus::APPROVED)
                ->whereBetween('start_date', [$startDate, $endDate]);
            });
        })->whereHas('user', function ($query) {
            $query->where('status', Status::INACTIVE);
        })
        ->with('user', function ($query) {
            $query->without('roles', 'permissions', 'supervisor', 'employeeMetaInfo', 'storage', 'user_supervisor');
        })->count();
    }

    public function countNewHireTurnoverRateActiveUsers($startDate, $endDate)
    {
        return EmployeeMetaInfo::whereHas('user', function ($query) use($startDate, $endDate) {
            $query->where('status', Status::ACTIVE)
            ->whereBetween('date_hired', [$startDate, $endDate]);
        })->with('user', function ($query) {
            $query->without('roles', 'permissions', 'supervisor', 'employeeMetaInfo', 'storage', 'user_supervisor');
        })->count();
    }

    public function fetchTimeToHireRate($startDate, $endDate, $keyword) 
    {
        $data = [];

        $fetchJobs = Position_has_salary::whereBetween('post_start', [$startDate, $endDate]);

        if ($keyword) {
            $fetchJobs->where(function ($query) use ($keyword) {
                $query->whereHas('position', function ($query) use ($keyword) {
                    $query->where('title', 'like', "%$keyword%");
                })
                ->orWhereHas('itemCodes', function ($query) use ($keyword) {
                    $query->where('item_code', 'like', "%$keyword%");
                });
            });
        }

        $jobs = $fetchJobs->get();

        foreach ($jobs as $job) {
            $employee = $this->fetchTimeToHireEmployee($job->position_id, $job->item_code_id);

            if ($employee) {
                $remark = $employee->user->serviceRecord->first()->remark;
                $startDate = $employee->user->serviceRecord->first();
                $dateHired = $remark->id == 1 ? Carbon::parse($employee->date_hired)->format('F j, Y') : Carbon::parse($startDate->start_date)->format('F j, Y');
                $dateJobOpened = Carbon::parse($job->post_start);
                $slots = $job->slots;
                $position = $employee->position->title;
                $itemCode = $employee->itemCode->item_code;
                $timeToHire = ($dateJobOpened->diffInDays($dateHired) == 1) ? $dateJobOpened->diffInDays($dateHired) . ' day' : $dateJobOpened->diffInDays($dateHired) . ' days';
    
                $data[] = [
                    'job_post_id' => $job->id,
                    'slots' => $slots,
                    'position' => $position,
                    'item_code' => $itemCode,
                    'date_job_opened' => Carbon::parse($dateJobOpened)->format('F j, Y'),
                    'date_hired' => Carbon::parse($dateHired)->format('F j, Y'),
                    'time_to_hire' => $timeToHire,
                ];
            }
        }

        return $data;
    }

    public function fetchTimeToHireEmployee($positionId, $itemCodeId) 
    {
        $original = 1;
        $promoted = 2;

        return EmployeeMetaInfo::where([
            ['position_id', $positionId], ['item_code_id', $itemCodeId]
        ])->whereHas('user', function ($query) use ($original, $promoted) {
            $query->whereHas('serviceRecord', function ($query) use ($original, $promoted) {
                $query->whereIn('remark_id', [$original, $promoted]);
            });
        })->with(['user.serviceRecord' => function ($query) use ($original, $promoted) {
            $query->whereIn('remark_id', [$original, $promoted])->orderBy('created_at', 'desc')->take(1)->with('remark');
        }])->first();
    }

    public function createCostPerHireReport($positionId)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {

            $jobPost = Position_has_salary::where('position_id', $positionId)->with('expenses')->get();
            $directExpensesTotal = 0;
            $indirectExpensesTotal = 0;
            $costPerHireTotal = 0;

            foreach ($jobPost as $jp) {
                $directExpenses = $jp->expenses->where('type', 1)->sum('amount');
                $indirectExpenses = $jp->expenses->where('type', 2)->sum('amount');
                $directExpensesTotal += $directExpenses;
                $indirectExpensesTotal += $indirectExpenses;
                $applicationCount = Application::where("phs_id", $jp->id)->count();
                $promotionApplicationCount = PromotionApplication::where('phs_id', $jp->id)->count();
                $candidates = $applicationCount + $promotionApplicationCount;
                $candidates = $candidates > 0 ? $candidates : 1;
                $costPerHireTotal += ($directExpenses + $indirectExpenses) / $candidates;
            }

            $data = CostPerHireReports::create([
                'position_id' => $positionId,
                'no_job_post' => $jobPost->count(),
                'direct_expenses_total' => $directExpensesTotal,
                'indirect_expenses_total' => $indirectExpensesTotal,
                'cost_per_hire_total' => $costPerHireTotal,
                'created_by' => Auth::id(),
            ]);

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return $data;
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return $e->getMessage();
        }

    }

    public function fetchCostPerHireTable($request)
    {
        $data = CostPerHireReports::when(isset($request->keyword), function ($query) use ($request) {
            $query->where('id', 'LIKE', '%' . $request->keyword . '%');
        })->when(isset($request->position_id), function ($query) use ($request) {
            $query->where('position_id', $request->position_id);
        })->get();

        $data->map(function ($item) {
            $item->item_codes = ItemCode::whereHas('positionHasSalary', function ($query) use ($item) {
                $query->where('position_id', $item->position_id);
            })->get();
            return $item;
        });

        // Paginate the modified collection
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');
        $page = request()->get('page', 1); // Get the current page number from the request, default to 1 if not provided
        $paginatedData = new \Illuminate\Pagination\LengthAwarePaginator(
            $data->forPage($page, $paginate),
            $data->count(),
            $paginate,
            $page,
            ['path' => request()->url(), 'query' => request()->query()]
        );

        return $paginatedData;
    }

    public function createTimeSinceLastPromotionReportData($department_id, $from = null, $to = null)
    {
        $user = User::whereHas('employeeMetaInfo', function ($query) use ($department_id) {
            $query->where('department_id', $department_id);
        });
        //$promoted_employees = 0;
        $remarkPromotedId = 2;
        $remarkOriginalId = 1;

        $totalEmployees = $user->count();

        $promotedEmployees = $user->whereHas('serviceRecord', function ($query) use ($from, $to, $remarkPromotedId) {
            $query->where('remark_id', $remarkPromotedId)
                ->when($from && $to, function ($query) use ($from, $to) {
                    $query->whereDate('start_date', '>=', $from)
                        ->whereDate('start_date', '<=', $to);
            });
        })->with([
                    'serviceRecord' => function ($query) use ($from, $to, $remarkPromotedId, $remarkOriginalId) {
                        $query->where(function ($query) use ($remarkPromotedId, $remarkOriginalId) {
                            $query->where('remark_id', $remarkPromotedId)
                                ->orWhere('remark_id', $remarkOriginalId);
                        })->whereDate('start_date', '>=', $from)
                            ->whereDate('start_date', '<=', $to)
                            ->orderBy('created_at', 'asc');
                    }
                ])->get();
        
        $averageMonths = 0;
        $totalMonths = 0;
        $totalPromotedEmployees = count($promotedEmployees);
        foreach ($promotedEmployees as $employee) {
            $monthsPerEmployee = 0;
            foreach ($employee->serviceRecord as $key => $recordPerUser) {
                $latestPromotion = Carbon::parse($recordPerUser->start_date);
                if (isset($employee->serviceRecord[$key + 1]->start_date)) {
                    $lastPromotion = Carbon::parse($employee->serviceRecord[$key + 1]->start_date);
                } else {
                    $lastPromotion = Carbon::parse($recordPerUser->start_date);
                }
                $monthsPerEmployee += $latestPromotion->diffInMonths($lastPromotion);
            }
            if ($monthsPerEmployee > 0 && $employee->serviceRecord->count() > 0) {
                $monthsPerEmployee = number_format($monthsPerEmployee / $employee->serviceRecord->count(), 2);
            }
            $totalMonths += $monthsPerEmployee;
        }

        if ($totalPromotedEmployees > 0 && $totalMonths > 0) {
            $averageMonths = number_format($totalMonths / $totalPromotedEmployees, 2);
        }

        $data = TimeSinceLastPromotionReports::Create([
            'department_id' => $department_id,
            'total_employees' => $totalEmployees,
            'promoted_employees' => $promotedEmployees->count(),
            'average_month' => $averageMonths . ' months',
            'start_date' => $from,
            'end_date' => $to,
            'created_by' => Auth::id(),
        ]);

        return $data;
    }

    public function fetchTimeSinceLastPromotionTable($request)
    {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');
        $data = TimeSinceLastPromotionReports::when(isset($request->department_id), function ($query) use ($request) {
            $query->where('department', $request->department);
        })->when(isset($request->keyword), function ($query) use ($request) {
            $query->whereHas('department', function ($query) use ($request) {
                $query->where('name', 'LIKE', '%' . $request->keyword . '%');
            }); 
        })
        ->when(isset($request->from) && isset($request->to), function ($query) use ($request) {
            $query->whereDate('start_date', '>=', $request->from)
            ->whereDate('end_date', '<=', $request->to);
        })->paginate($paginate);

        return $data;
    }

    public function fetchTimeSinceLastPromotionData($request, $forExport = false)
    {
        $remarkPromotedId = 2;
        $remarkOriginalId = 1;
        $data = TimeSinceLastPromotionReports::findOrFail($request->id);

        $users = User::whereHas('employeeMetaInfo', function ($query) use ($data) {
            $query->where('department_id', $data->department_id);
        })->whereHas('serviceRecord', function ($query) use ($data, $remarkPromotedId) {
            $query->where('remark_id', $remarkPromotedId)
                ->whereDate('start_date', '>=', $data->start_date)
                ->whereDate('start_date', '<=', $data->end_date);
        })->with([
                    'serviceRecord' => function ($query) use ($data, $remarkPromotedId, $remarkOriginalId) {
                        $query->where(function ($query) use ($remarkPromotedId, $remarkOriginalId) {
                            $query->where('remark_id', $remarkPromotedId)
                                ->orWhere('remark_id', $remarkOriginalId);
                        })->whereDate('start_date', '>=', $data->start_date)
                            ->whereDate('start_date', '<=', $data->end_date)
                            ->orderBy('created_at', 'asc');
                    }
                ])->get();

        $employeesData = [];
        $totalMonths = 0;
        $averageMonths = 0;
        $promotedEmployees = count($users);
        foreach ($users as $user) {
            $monthsPerEmployee = 0;
            foreach ($user->serviceRecord as $key => $recordPerUser) {
                $latestPromotion = Carbon::parse($recordPerUser->start_date);
                if (isset($user->serviceRecord[$key + 1]->start_date)) {
                    $lastPromotion = Carbon::parse($user->serviceRecord[$key + 1]->start_date);
                } else {
                    $lastPromotion = Carbon::parse($recordPerUser->start_date);
                }
                $monthsPerEmployee += $latestPromotion->diffInMonths($lastPromotion);
            }

            if ($monthsPerEmployee > 0 && $user->serviceRecord->count() > 0) {
                $monthsPerEmployee = number_format($monthsPerEmployee / $user->serviceRecord->count(), 2);
            }
            $totalMonths += $monthsPerEmployee;

            $employeesData[] = [
                'employee_id' => $user->employeeMetaInfo->employee_id,
                'employee_name' => $user->name,
                'date_hired' => $user->employeeMetaInfo->date_hired,
                'date_of_promotion' => $latestPromotion->format('Y-m-d'),
                'time_since_last_promotion' => $monthsPerEmployee .' months',
            ];
        }
        
        if ($promotedEmployees > 0 && $totalMonths > 0) {
            $averageMonths = number_format($totalMonths / $promotedEmployees, 2);
        }

        if ($forExport) {
            return $employeesData;
        }

        $employeesData = new Collection($employeesData);
        $employeesData = $employeesData->when(isset($request->keyword), function ($query) use ($request) {
            return $query->filter(function ($item) use ($request) {
                return stripos($item['employee_name'], $request->keyword) !== false || strpos($item['employee_id'], $request->keyword) !== false;
            });
        })->toArray();

        // $newElement = ['average_month' => $averageMonths];
        // $employeesData = array_merge($newElement, $employeesData);

        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');
        $currentPage = LengthAwarePaginator::resolveCurrentPage();
        $pagedData = array_slice($employeesData, ($currentPage - 1) * $paginate, $paginate);
        $data = new LengthAwarePaginator($pagedData, count($employeesData), $paginate, $currentPage);
        $data->setPath(\Request::url());
        return $data;
    }
}
