<?php

namespace Suiterus\Adg\Services\Roster;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use Suiterus\Adg\Models\SM\Holiday;
use Suiterus\Adg\Models\Timekeeping\Roster\Roster;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDay;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Services\Attendance\AttendanceService;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterLegend;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterPartition;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterStaffPerDay;
use Suiterus\Adg\Controllers\Timekeeping\Roster\RosterController;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeesPerGroup;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadRequiredHours;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeGroupShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeRequiredHours;

class RosterService
{
    public static $roster;
    public static $rosterGroups;
    private $employeeGroup;

    public static function createRoster($data)
    {
        $data = Roster::create($data);
        self::$roster = $data;
        return new self();
    }

    public static function setRoster($id)
    {
        $data = Roster::find($id);
        self::$roster = $data;
        return new self();
    }

    public static function getRoster()
    {
        return self::$roster;
    }

    /**
     * The function `createGroups` creates groups for employees in a roster.
     *
     * @param groups The `createGroups` function takes an array of groups as a parameter. Each group in
     * the array should have a `group_id` and a `description` key. The function then creates a new
     * group for each item in the array and associates it with the roster's employee groups.
     *
     * @return An instance of the class that contains the `createGroups` method is being returned.
     */
    public static function createGroups($groups)
    {
        if (!self::$roster) {
            return new Exception('Initialize roster first');
        }

        self::$rosterGroups = collect();

        foreach ($groups as $group) {
            self::$rosterGroups->push(self::$roster->employeeGroups()->updateOrCreate([
                'roster_id' => self::$roster->id,
                'group_id' => $group['group_id'],
            ], [
                'group_id' => $group['group_id'],
                'description' => $group['description']
            ]));
        }

        return new self();
    }

    public function getGroups()
    {
        return self::$rosterGroups;
    }

    /**
     * The function `createPartitions` generates partitions based on the number of weeks, partition
     * size, and type specified.
     *
     * @param number_of_weeks The `number_of_weeks` parameter in the `createPartitions` function
     * represents the total number of weeks that need to be partitioned. This value is used to
     * calculate the duration of each partition based on the specified partition size and type (full or
     * half).
     * @param partition The `partition` parameter in the `createPartitions` function represents the
     * number of partitions you want to create within the specified number of weeks. It is used to
     * divide the total number of weeks into equal parts based on the value provided.
     * @param type The `type` parameter in the `createPartitions` function determines how the
     * partitions will be created. It can have two values: "full" or "half".
     *
     * @return An instance of the class that contains the `createPartitions` method is being returned.
     */
    public static function createPartitions($number_of_weeks, $partition, $type)
    {
        $partition_weeks = $number_of_weeks / $partition;
        if ($type == "full") {
            for ($i = 0; $i < $partition; $i++) {
                self::$roster->partitions()->create([
                    'number_of_weeks' => $partition_weeks,
                    'start_date' => Carbon::parse(self::$roster->start_date)->addWeeks($partition_weeks * $i),
                    'end_date' => Carbon::parse(self::$roster->start_date)->addWeeks($partition_weeks * ($i + 1))->subDays(1),
                ]);
            }
        } elseif ($type == "half") {
            self::$roster->partitions()->create([
                'number_of_weeks' => $number_of_weeks,
                'start_date' => Carbon::parse(self::$roster->start_date),
                'end_date' => Carbon::parse(self::$roster->start_date)->addWeeks($number_of_weeks)->subDays(1),
            ]);
        }

        return new self();
    }

    /**
     * The function `createPartitionDays` creates partitioned days for a roster, assigns required
     * hours, sets remarks and legends, and creates employee shifts for each day.
     *
     * @return An instance of the class that contains the `createPartitionDays` method is being
     * returned.
     */
    public static function createPartitionDays()
    {
        $roster_partitions = RosterPartition::where('roster_id', self::$roster->id)->get();
        foreach ($roster_partitions as $partition) {
            for ($x = 0; $x < $partition->number_of_weeks; $x++) {
                for ($i = 0; $i < 7; $i++) {
                    $partition->days()->create([
                        'roster_partition_id' => $partition->id,
                        'date' => Carbon::parse($partition->start_date)->addDays($i)->addWeeks($x),
                        'day' =>  Carbon::parse($partition->start_date)->addDays($i)->addWeeks($x)->dayOfWeek,
                    ]);
                }
            }
        }

        foreach ($roster_partitions as $partition) {
            RosterHeadRequiredHours::create([
                'roster_partition_id' => $partition->id,
                'roster_id' => $partition->roster_id
            ]);
            foreach ($partition['days'] as $day) {
                list($remarks, $legend) = self::setRemarksAndLegends(self::$roster->head_nurse_id, $day['date']);
                RosterHeadEmployeeShift::create([
                    'roster_id' => $partition->roster_id,
                    'roster_day_id' => $day['id'],
                    'remarks' => $remarks,
                    'roster_legend_id' => $legend
                ]);
            }
        }

        return new self();
    }

    /**
     * The function creates group shifts by updating or creating a record in the database and adding
     * partitions to the roster.
     *
     * @param shiftId The `shiftId` parameter in the `createGroupShifts` function represents the ID of
     * the shift that you want to assign to a specific group. This ID is used to uniquely identify the
     * shift within the system.
     * @param groupId The `groupId` parameter in the `createGroupShifts` function represents the ID of
     * the roster employee group to which the shift will be assigned. This parameter is used to
     * associate the shift with a specific group of employees in the roster system.
     *
     * @return An instance of the current class is being returned.
     */
    public function createGroupShifts($shiftId, $groupId)
    {

        $group_shift = RosterEmployeeGroupShift::updateOrCreate([
            'roster_employee_group_id' => $groupId,
            'shift_id' => $shiftId
        ], [
            'roster_employee_group_id' => $groupId,
            'shift_id' => $shiftId
        ]);

        $this->addPartitions(self::$roster, $group_shift);
        return $this;
    }

    /**
     * The function `addEmployeeToGroup` adds an employee to a specified group and assigns required
     * hours and shifts for each day in the roster partitions.
     *
     * @param userId The `userId` parameter in the `addEmployeeToGroup` function represents the unique
     * identifier of the user that you want to add to a specific group. This user will be associated
     * with the group through the `roster_employee_group_id` in the `RosterEmployeesPerGroup` table.
     * @param groupId The `groupId` parameter in the `addEmployeeToGroup` function represents the ID of
     * the group to which you want to add the employee identified by the `userId`. This function
     * creates a new entry in the `RosterEmployeesPerGroup` table linking the user to the specified
     * group. It also
     *
     * @return An instance of the class that contains the `addEmployeeToGroup` method is being
     * returned.
     */
    public function addEmployeeToGroup($userId, $groupId)
    {

        $employee = RosterEmployeesPerGroup::where([
            'user_id'   => $userId,
            'roster_employee_group_id'  =>  $groupId
        ])->first();

        if (!$employee) {
            $employee = RosterEmployeesPerGroup::create([
                'user_id'   => $userId,
                'roster_employee_group_id'  =>  $groupId
            ]);

            foreach (self::$roster->partitions as $partition) {
                RosterEmployeeRequiredHours::create([
                    'roster_partition_id' => $partition->id,
                    'roster_employee_id' => $employee->id
                ]);
                foreach ($partition['days'] as $day) {
                    list($remarks, $legend) = self::setRemarksAndLegends($userId, $day['date']);
                    $employee->dayEmployeeShifts()->create([
                        'roster_day_id' => $day['id'],
                        'remarks' => $remarks,
                        'roster_legend_id' => $legend
                    ]);
                }
            }
        } else {
            $employee->update([
                'user_id'   => $userId,
                'roster_employee_group_id'  =>  $groupId
            ]);
        }

        $this->employeeGroup = $employee;

        return $this;
    }

    public function getEmployeeGroup()
    {
        return $this->employeeGroup;
    }

    public function addPartitions(Roster $roster, RosterEmployeeGroupShift $group_shift)
    {
        foreach ($roster->partitions as $partition) {
            foreach ($partition['days'] as $day) {
                $group_shift->staffPerShift()->updateOrCreate([
                    'roster_day_id' => $day['id'],
                ], [
                    'number_of_staff' => 0
                ]);
            }
        }
    }

    /**
     * The function `setRemarksAndLegends` retrieves remarks and legends based on user ID and date for
     * display in the frontend.
     *
     * @param user_id The `user_id` parameter is the unique identifier of the user for whom we are
     * setting remarks and legends. It is used to retrieve specific information related to that user,
     * such as their leave, CTO (Compensatory Time Off), and holiday details.
     * @param date The `setRemarksAndLegends` function takes two parameters: `` and ``.
     * The `` is used to retrieve information related to a specific user, while the `` is
     * used to determine the date for which the remarks and legends are being set.
     *
     * @return The function `setRemarksAndLegends` returns an array containing two elements: ``
     * and ``.
     */
    public function setRemarksAndLegends($user_id, $date)
    {

        list($leave, $cto, $holiday) = self::getCtoLeaveHoliday($user_id, $date);
        $legend = RosterLegend::where('key', 'leave')->value('id');
        if (!empty($leave)) {
            $remarks = $leave->leaveType->remark;
        } elseif (!empty($cto)) {
            $remarks = $cto->remarks;
            $legend = null;
        } elseif (!empty($holiday)) {
            $remarks = '‎ '; //insert this remarks for frontend to see blank with color
        } else {
            $remarks = ' ';
            $legend = null;
        }

        return [$remarks, $legend];
    }

    public function getCtoLeaveHoliday($id, $date)
    {
        $leave = Leave::where([['user_id', $id], ['status', 1]])
            ->whereDate('start_date', '<=', $date)
            ->whereDate('end_date', '>=', $date)
            ->with('leaveType')
            ->latest('updated_at')
            ->first();
        $cto = CTOApplication::where([['user_id', $id], ['status', 1]])
            ->whereDate('start_date', '<=', $date)
            ->whereDate('end_date', '>=', $date)
            ->latest('updated_at')
            ->first();
        $holiday = Holiday::whereDate('date', $date)
            ->where('status', 1)
            ->first();

        return [$leave, $cto, $holiday];
    }

    public function removeEmployeeShifts($employeeShifts)
    {
        foreach ($employeeShifts as $employeeShift) {
            $employeeShift = RosterDayEmployeeShift::where('id', $employeeShift)->first();
            $shift_id = $employeeShift->shift_id;

            $roster_employee_per_group = RosterEmployeesPerGroup::where('id', $employeeShift->roster_employees_per_group_id)->first();

            $employeeShift->update([
                'shift_id' => null,
                'roster_legend_id' => null,
                'remarks' => ''
            ]);

            $shift = RosterDayEmployeeShift::where([
                ['shift_id', $shift_id]
            ])->whereHas('rosterEmployeePerGroup', function ($query) use ($employeeShift) {
                $query->where('roster_employee_group_id', $employeeShift->rosterEmployeePerGroup->roster_employee_group_id);
            })->count();

            if ($shift == 0) {
                $roster_employee_group_shift = RosterEmployeeGroupShift::where([
                    ['roster_employee_group_id', $roster_employee_per_group->roster_employee_group_id],
                    ['shift_id', $shift_id]
                ])->first();

                if ($roster_employee_group_shift) {
                    RosterStaffPerDay::where('roster_employee_group_shift_id', $roster_employee_group_shift->id)->delete();
                    $roster_employee_group_shift->delete();
                }
            }

            $user_id = $employeeShift->rosterEmployeePerGroup->user_id;
            $date = $employeeShift->rosterDay()->first()->date;
            $roster_id = $employeeShift->rosterEmployeePerGroup->employeeGroup->roster_id;

            $user = User::whereId($user_id)->first();
            $attendance_service = new AttendanceService($user);
            $attendance_service->reEvaluateAttendance($date);

            $this->updateRequiredHoursPerPartition([$user_id], $roster_id);
            $this->updateMinStaffPerDays($employeeShift->rosterEmployeePerGroup->employeeGroup->id);
        }
    }

    public function updateEmployeeShifts($employeeShifts, $shiftId, $legendId, $remarks)
    {
        foreach ($employeeShifts as $employeeShift) {
            $employeeShift = RosterDayEmployeeShift::where('id', $employeeShift)->first();
            $previous_shift_id = $employeeShift->shift_id;
            $existing_group_shifts = $employeeShift->rosterEmployeePerGroup->employeeGroup->employeeGroupShifts->pluck('shift_id')->toArray();

            if (!in_array($shiftId, $existing_group_shifts)) {
                $group_shift = RosterEmployeeGroupShift::create([
                    'roster_employee_group_id' => $employeeShift->rosterEmployeePerGroup->employeeGroup->id,
                    'shift_id' => $shiftId
                ]);

                $roster = $employeeShift->rosterEmployeePerGroup->employeeGroup->roster;

                $this->addPartitions($roster, $group_shift);
            }

            $employeeShift->update([
                'shift_id' => $shiftId,
                'roster_legend_id' => $legendId,
                'remarks' => $remarks
            ]);

            $shift = RosterDayEmployeeShift::where([
                ['shift_id', $previous_shift_id]
            ])->whereHas('rosterEmployeePerGroup', function ($query) use ($employeeShift) {
                $query->where('roster_employee_group_id', $employeeShift->rosterEmployeePerGroup->roster_employee_group_id);
            })->count();
            if ($shift == 0) {
                $roster_employee_group_shift = RosterEmployeeGroupShift::where([
                    ['roster_employee_group_id', $employeeShift->rosterEmployeePerGroup->employeeGroup->id],
                    ['shift_id', $previous_shift_id]
                ])->first();

                if ($roster_employee_group_shift) {
                    RosterStaffPerDay::where('roster_employee_group_shift_id', $roster_employee_group_shift->id)->delete();
                    $roster_employee_group_shift->delete();
                }
            }

            $user_id = $employeeShift->rosterEmployeePerGroup->user_id;
            $date = $employeeShift->rosterDay()->first()->date;
            $roster_id = $employeeShift->rosterEmployeePerGroup->employeeGroup->roster_id;

            $user = User::whereId($user_id)->first();
            $attendance_service = new AttendanceService($user);
            $attendance_service->reEvaluateAttendance($date);

            $this->updateRequiredHoursPerPartition([$user_id], $roster_id);
            $this->updateMinStaffPerDays($employeeShift->rosterEmployeePerGroup->employeeGroup->id);
        }
    }

    private function updateRequiredHoursPerPartition($user_ids, $roster_id)
    {
        $roster = Roster::whereId($roster_id)->with('partitions')->first();
        foreach ($user_ids as $user_id) {
            foreach ($roster->partitions as $partition) {
                $shifts = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) use ($user_id, $roster_id) {
                    $query->where('user_id', $user_id)
                        ->whereHas('employeeGroup.roster', function ($query) use ($roster_id) {
                            $query->where('id', $roster_id);
                        });
                })->whereHas('rosterDay.rosterPartition', function ($query) use ($partition) {
                    $query->whereId($partition->id);
                })->with('shift')->get();

                $total_hours = 0;
                foreach ($shifts->pluck('shift') as $shift) {
                    if (!$shift) continue;
                    if (!$shift->break_in || !$shift->break_out) {
                        $break_hours = 0;
                    } else {
                        $break_in = strtotime($shift->break_in);
                        $break_out = strtotime($shift->break_out);
                        $break_hours = ($break_out - $break_in) / 3600;
                    };
                    $total_hours += ($shift->work_hours  - $break_hours);
                }

                $req_hours = RosterEmployeeRequiredHours::where([
                    'roster_partition_id' => $partition->id,
                    'roster_employee_id' => $shifts[0]->roster_employees_per_group_id,
                ])->first();
                $req_hours->required_hours = $total_hours;
                $req_hours->save();
            }
        }
    }

    private function updateMinStaffPerDays($roster_employee_group_id)
    {
        $staff_per_days = RosterStaffPerDay::whereHas('shift.rosterEmployeeGroup', function ($query) use ($roster_employee_group_id) {
            $query->whereId($roster_employee_group_id);
        })->with('rosterDay', 'shift.shift')->get();

        foreach ($staff_per_days as $staff_per_day) {
            $count = 0;
            $count = RosterDayEmployeeShift::where('roster_day_id', $staff_per_day->rosterDay->id)
                ->where('shift_id', $staff_per_day->shift->shift->id)
                ->whereHas('rosterEmployeePerGroup.employeeGroup', function ($query) use ($staff_per_day) {
                    $query->whereId($staff_per_day->shift->roster_employee_group_id);
                })
                ->count();

            $staff_per_day->number_of_staff = $count;
            $staff_per_day->save();
        }
    }

    public function getUsersShift($rosterId, $startDate, $endDate, $userIds)
    {

        $userShifts = [];
        $shifts = [];

        $employeeShifts = RosterDayEmployeeShift::whereHas('rosterDay', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('date', [$startDate, $endDate]);
        })->whereHas('rosterEmployeePerGroup', function ($query) use ($userIds) {
            $query->whereIn('user_id', $userIds);
        })->whereHas('rosterEmployeePerGroup.employeeGroup', function ($query) use ($rosterId) {
            $query->whereNotIn('roster_id', [$rosterId]);
        })->whereNotNull('shift_id')->with(['rosterEmployeePerGroup.user', 'rosterEmployeePerGroup.employeeGroup.roster', 'rosterEmployeePerGroup.employeeGroup.rosterGroup', 'rosterDay', 'rosterLegend', 'shift'])->get();

        foreach ($employeeShifts as $employeeShift) {
            $shifts[$employeeShift->rosterEmployeePerGroup->user_id][] = $employeeShift;
        }

        foreach ($shifts as $key => $shift) {
            $userShifts[] = [
                'user' => User::without([
                    'roles',
                    'permissions',
                    'storage',
                    'employeeMetaInfo',
                    'supervisor',
                    'user_supervisor',
                    'exitInterview',
                    'userProfilePicture',
                    'profileBasicInfo'
                ])->find($key),
                'shifts' => $shift
            ];
        }

        return $userShifts;
    }

    public function removeEmployees($employee){
        if ($employee) {
            $employee->dayEmployeeShifts()->delete();
            $employee->requiredHours()->delete();

            $roster_id = Roster ::whereHas('employeeGroups.employees', function ($query) use ($employee) {
                $query->where('id', $employee->id);
            })->pluck('id')->first();

            $roster_days = RosterDay::whereHas('rosterPartition', function ($query) use ($roster_id) {
                $query->where('roster_id', $roster_id);
            })->get();

            $user = User::whereId($employee->user_id)->first();
            $attendance_service = new AttendanceService($user);

            foreach ($roster_days as $day) {
                $attendance_service->reEvaluateAttendance($day->date);
            }

            $this->updateMinStaffPerDays($employee->employeeGroup->id);
            $employee->delete();
        }
    }
}
