<?php

namespace Suiterus\Adg\Services\Reports;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use App\Enums\LeaveStatus;
use App\Enums\ReportStatus;
use Suiterus\Adg\Models\SM\Shift;
use Illuminate\Support\Facades\Log;
use Suiterus\Adg\Models\SM\Holiday;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Reports\AbsenteeismReport;
use Suiterus\Adg\Models\Reports\AbsenceReports;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Models\Reports\AbsenceReportData;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;

class AbsenceRateReportService
{
    private $user;

    public function getAbsenteeismExcelData($from, $to, $user)
    {
        $data = [];
        $currentDate = Carbon::parse($from, 'UTC');
        $endDate = Carbon::parse($to, 'UTC');
    
        $monthlyData = [];
        $employee = 1;
    
        while ($currentDate <= $endDate) {
            $this->user = $user;
            $attendance = Attendance::where('user_id', $user->id)->whereDate('time_in', $currentDate)->first();
            $remarks = $this->determineAbsence($currentDate);
    
            $monthYear = $currentDate->format('F-Y');
    
            if (!isset($monthlyData[$monthYear])) {
                $monthlyData[$monthYear] = [
                    'working_days' => 0,
                    'days_lost' => 0,
                ];
            }
    
            if (!$remarks["isHoliday"] && !$remarks["isOnLeave"] && !$remarks["isRestDay"] && !$attendance) {
                $monthlyData[$monthYear]['days_lost']++;
            }
            if (!$remarks['isRestDay']) {
                $monthlyData[$monthYear]['working_days']++;
            }

            $currentDate->addDay();
        }
    
        foreach ($monthlyData as $monthYear => $values) {
            $averageWorkingDays = $employee > 0 ? number_format($values['working_days'] / $employee, 2) : 0;
            $absenceRate = $employee > 0 && $averageWorkingDays > 0 ? number_format($values['days_lost'] / ($employee * $averageWorkingDays) * 100, 2) : 0;
    
            $data[] = [
                'date' => $monthYear,
                'total_employees' => $employee,
                'working_days' => $values['working_days'],
                'average_working_days' => $averageWorkingDays,
                'days_lost' => $values['days_lost'],
                'absent_rate' => $absenceRate,
            ];
        }
    
        return $data;
    }
    
    public function getAbsentRate($from, $to)
    {
        $startDate = Carbon::parse($from, 'UTC');
        $endDate = Carbon::parse($to, 'UTC');
        $employees = $this->getUsers($startDate, $endDate);
        $totalEmployees = $employees->count();
        $totalWorkingDays = 0;
        $totalDaysLost = 0;
        $averageWorkingDays = 0.00;
        $absenceRate = 0.00;

        $currentDate = $startDate->copy();
        while ($currentDate <= $endDate) {
            foreach ($employees as $user) {
                $this->user = $user;
                $attendance = Attendance::where('user_id', $user->id)->whereDate('time_in', $currentDate)->first();
                $remarks = $this->determineAbsence($currentDate);
                if (!$remarks["isHoliday"] && !$remarks["isOnLeave"] && !$remarks["isRestDay"] && !$attendance) {
                    $totalDaysLost++;
                }
                if (!$remarks['isRestDay']) {
                    $totalWorkingDays++;
                }
            }
            $currentDate->addDay();
        }

        if ($totalEmployees > 0 && $totalWorkingDays > 0) {
            $averageWorkingDays = number_format($totalWorkingDays / $totalEmployees, 2);
        }

        if ($totalDaysLost > 0) {
            $absenceRate = number_format($totalDaysLost / ($totalEmployees * $averageWorkingDays) * 100, 2);
        }

        return [
            'start_date' => $startDate->format('Y-m-d'),
            'end_date' => $endDate->format('Y-m-d'),
            'total_employees' => $totalEmployees,
            'working_days' => $totalWorkingDays,
            'average_working_days' => $averageWorkingDays,
            'days_lost' => $totalDaysLost,
            'absent_rate' => $absenceRate,
        ];
    }
    
    public function getUsers($startDate, $endDate)
    {
        return User::where(function ($query) use ($startDate, $endDate) {
            $query->whereHas('employeeMetaInfo', function ($query) use ($endDate){
                $query->where('date_hired', '<=', $endDate);
            })->orWhereHas('attendance', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('time_in', [$startDate, $endDate]);
            });
        })
        ->with([
            'employeeSchedules' => function ($query) {
                $query->with([
                    'schedule_template' => function ($query) {
                        $query->with('schedule_template');
                    }
                ])->without('user');
            },
            'temporary_schedule' => function ($query) {
                $query->with('schedule_template');
            }
        ])
        ->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo',
            'employeeMetaInfo'
        ])
        ->whereNotIn('id', [1, 2])
        ->get();
    }

    public function employeeAbsenteeism($data, $user) 
    {
        $absenceReport = AbsenceReports::where('id', $data->id)->first();
    
        if ($absenceReport->status == ReportStatus::PENDING) {
            $absenceReport->update(['status' => ReportStatus::RUNNING]);
        }
    
        $absenteeismData = $this->getAbsenteeismExcelData($data->start_date, $data->end_date, $user);
    
        foreach ($absenteeismData as $dataRow) {
            $monthYear = $dataRow['date'];
            $existingData = AbsenceReportData::where('absence_rate_report_id', $data->id)
                ->where('date', $monthYear)
                ->first();
    
            if ($existingData) {
                $existingData->update([
                    'total_employees' => $existingData->total_employees + $dataRow['total_employees'],
                    'working_days' => $existingData->working_days + $dataRow['working_days'],
                    'days_lost' => $existingData->days_lost + $dataRow['days_lost'],
                ]);
            } else {
                AbsenceReportData::create([
                    'absence_rate_report_id' => $data->id,
                    'date' => $monthYear,
                    'total_employees' => $dataRow['total_employees'],
                    'working_days' => $dataRow['working_days'],
                    'days_lost' => $dataRow['days_lost'],
                ]);
            }
        }

        // Check if there are no records where total_employees is not equal to data->total_employees
        $allMatch = AbsenceReportData::where('absence_rate_report_id', $data->id)
        ->where('total_employees', '!=', $data->total_employees)
        ->doesntExist();

        if ($allMatch) {
            $data->update(['status' => ReportStatus::COMPLETED]);
            $this->fetchAbsenteeismReportData($data->id);
        }
    }

    public function fetchAbsenteeismReportData($reportId)
    {
        $absenceReport = AbsenceReports::find($reportId);

        if (!$absenceReport) {
            throw new Exception('Absence report not found.');
        }

        $existingData = [];
        foreach (AbsenceReportData::where('absence_rate_report_id', $reportId)->get() as $row) {
            $monthYear = $row->date;
            $existingData[$monthYear] = [
                'total_employees' => $row->total_employees,
                'working_days' => $row->working_days,
                'days_lost' => $row->days_lost,
            ];
        }

        $totalWorkingDays = 0;
        $totalDaysLost = 0;
        $totalEmployees = $absenceReport->total_employees;

        foreach ($existingData as $monthYear => $values) {
            $averageWorkingDays = $totalEmployees > 0 ? $values['working_days'] / $totalEmployees : 0;
            $absenceRate = ($totalEmployees > 0 && $averageWorkingDays > 0) ? min(number_format($values['days_lost'] / ($totalEmployees * $averageWorkingDays) * 100, 2), 100) : 0;
                
            $formattedData[] = [
                'date' => $monthYear,
                'total_employees' => $totalEmployees,
                'working_days' => $values['working_days'],
                'average_working_days' => $averageWorkingDays,
                'days_lost' => $values['days_lost'],
                'absent_rate' => $absenceRate
            ];

            $totalWorkingDays += $values['working_days'];
            $totalDaysLost += $values['days_lost'];
        }

        $absenceReport->working_days = $totalWorkingDays;
        $absenceReport->days_lost = $totalDaysLost;
        $absenceReport->average_working_days = $totalEmployees > 0 ? $totalWorkingDays / $totalEmployees : 0;
        $absenceReport->absent_rate = $totalEmployees > 0 && $absenceReport->average_working_days > 0 ? $totalDaysLost / ($totalEmployees * $absenceReport->average_working_days) * 100 : 0;
        $absenceReport->save();

        $fileName = Carbon::parse($absenceReport->start_date)->format('F-d-Y') . '-' . Carbon::parse($absenceReport->end_date)->format('F-d-Y') . '-Absenteeism-Report-' . $absenceReport->id . '.xlsx';

        $absenceReport->update([
            'file_path' => $fileName,
            'remarks' => "Data stored",    
        ]);

        if (!Storage::disk('absenteeism_reports')->exists(dirname($fileName))) {
            Storage::disk('absenteeism_reports')->makeDirectory(dirname($fileName));
        }
        
        Excel::store(new AbsenteeismReport($formattedData, $absenceReport), $fileName, 'absenteeism_reports');
    }

    private function determineAbsence($date)
    {
        return [
            'isHoliday' => $this->getHoliday($date) ? true : false,
            'isOnLeave' => $this->getLeave($date) ? true : false,
            'isRestDay' => $this->isRestDay($date) ? true : false
        ];
    }

    private function getHoliday($date)
    {
        $holiday = Holiday::whereDate('date', Carbon::parse($date)->format('Y-m-d'))->where('status', Status::ACTIVE)->orderBy('purpose', 'asc')->first();

        return $holiday ? $holiday : null;
    }

    private function getLeave($date)
    {

        $leave = Leave::where('user_id', $this->user->id)->whereDate('start_date', '<=', Carbon::parse($date)->format('Y-m-d'))
            ->whereDate('end_date', '>=', Carbon::parse($date)->format('Y-m-d'))
            ->where('status', LeaveStatus::APPROVED)->first();
        return $leave ? $leave : null;
    }

    private function isRestDay($day)
    {
        $date = Carbon::parse($day)->format('Y-m-d');
        $day = date("l", strtotime($day));

        $roster_schedule = $this->getRosterShift($date);

        if ($roster_schedule) {
            return false;
        }

        if (!$this->user->employeeSchedules->isEmpty()) {
            $schedules = $this->user->employeeSchedules[0]['schedule_template']['schedule_template'];
            $days = [];
            foreach ($schedules as $scheduled) {
                $days[] = $scheduled['day'];
            }
            return !in_array($day, $days);
        }

        return true;
    }

    private function getRosterShift($currentdate)
    {
        $user = $this->user;
        return Shift::whereHas('employeeShift', function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date', $currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            });
        })->orWhereHas('headEmployeeShift', function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date', $currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id);
            });
        })->first();
    }
}
