<?php

namespace Suiterus\Adg\Services\Promotion;

use DateTime;
use Exception;
use Suiterus\Adg\Models\ActualDesignation;
use ZipArchive;
use Carbon\Carbon;
use NumberFormatter;
use mikehaertl\pdftk\Pdf;
use App\Enums\ItemCodeStatus;
use App\Enums\CorporationList;
use App\Enums\RescheduleStatus;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Remark;
use Suiterus\Adg\Models\SM\Sector;
use Suiterus\Hrjp\Models\ItemCode;
use Illuminate\Support\Facades\Auth;
use App\Enums\Promotion\ProcessStatus;
use App\Enums\Promotion\ProcessSteps;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\SM\Corporation;
use Suiterus\Adg\Models\SM\SalaryGrade;
use Suiterus\Adg\Models\SM\EmployeeType;
use App\Enums\Promotion\RequirementStatus;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Hrjp\Models\Position_history;
use Suiterus\Adg\Models\PDS\PDSPerPosition;
use SoareCostin\FileVault\Facades\FileVault;
use Suiterus\Adg\Models\Salary\PrivateSalary;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use App\Enums\Promotion\PromotionApplicationStatus;
use App\Http\Classes\NotificationMessage;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Hrjp\Models\Position_has_salary as PHS;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;
use Suiterus\Adg\Models\Promotion\PromotionApplication;
use Suiterus\Adg\Models\Salary\PublicSalary;
use Suiterus\Adg\Models\Supplemental\SupplementalDetailUser;
use Suiterus\Adg\Models\Supplemental\SupplementalCompletionUser;
use Suiterus\Adg\Models\Supplemental\SupplementalRequirementUser;
use Suiterus\Hrjp\Models\Position;
use Symfony\Component\HttpFoundation\File\Exception\FileException;

class PromotionService
{
    use HasCustomLogs;

    public function create_application($user_id, $position_id, $phs_id, $position_option = [])
    {
        $application = PromotionApplication::create([
            'user_id' => $user_id,
            'status' => PromotionApplicationStatus::ACTIVE,
            'position_id' => $position_id,
            'phs_id' => $phs_id,
            'created_by' => $user_id,
        ]);

        $application->process()->create([
            'step' => 1,
            'status' => RequirementStatus::PENDING,
            'created_by' => $user_id
        ]);

        $positionTitle = Position::whereId($position_id)->pluck('title')->first();
        NotificationMessage::notifySenderAndReceiver('Create Application', $application->phs->hiring_manager->id, $user_id, null, null, null, null, $positionTitle);
        return $application;
    }

    public function fetch_all_position()
    {
        $today = Carbon::now();
        $posted = 1;
        $available = 3;

        $getPositions = PHS::select('position_id', 'description', 'salary', 'show_salary')->where([
            ['post_status', $posted],
            ['slots', '>', 0]
        ])
            ->whereDate('post_due', '>=', $today)
            ->groupBy('position_id', 'description', 'salary', 'show_salary')
            ->get();

        $data = [];
        $processedPositionIds = [];

        foreach ($getPositions as $position) {
            // Check if position_id is already processed
            if (!in_array($position->position_id, $processedPositionIds)) {
                $processedPositionIds[] = $position->position_id;


                $fetchPositions = PHS::where([
                    ['position_id', $position->position_id],
                    ['description', $position->description],
                    ['salary', $position->salary['salary_grade']],
                    ['post_status', $posted]
                ])->with('position', 'itemCodes')->get();
                $slots = 0;
                $itemCodes = $fetchPositions->map(function ($item) use ($available, &$slots) {
                    if ($item->itemCodes->status == $available) {
                        $slots++;
                        return [
                            'hiring_manager' => $item->hiring_manager->name,
                            'post_due' => $item->post_due,
                            'post_start' => $item->post_start,
                            'agency' => Corporation::where('employer_id', CorporationList::NKTI)->pluck('name')->first(),
                            'phs_id' => $item->id,
                            'position_id' => $item->position->id,
                            'id' => $item->itemCodes->id,
                            'item_code' => $item->itemCodes->item_code,
                            'experience' => $item->experience,
                            'education' => $item->education,
                            'eligibility' => $item->eligibility,
                            'training' => $item->training,
                            'placement_of_assignment' => $item->place_of_assignment,
                            'position_title' => $item->position->title,
                            'salary' => $item->salary,
                            'description' => ItemCode::where('item_code', $item->itemCodes->item_code)->pluck('description')->first(),
                            'show_salary' => $item->show_salary
                        ];
                    }
                })->filter()->toArray();

                $data[] = [
                    'details' => $itemCodes,
                    'position_id' => $position->position_id,
                    'position_title' => $position->position->title,
                    // 'show_salary' => $position->show_salary,
                    // 'salary' => $position->salary,
                    'slots' => $slots, // Include slots here
                ];
            }
        }

        return $data;
    }

    public function create_user_requirements($applicationId, $supplementalReqIds, $supplementaDetailId, $supplementalCompletionIds)
    {

        foreach ($supplementalReqIds as $reqId) {
            SupplementalRequirementUser::create([
                'application_id' => $applicationId,
                'supplemental_req_id' => $reqId,
            ]);
        }

        foreach ($supplementaDetailId as $detalId) {
            SupplementalDetailUser::create([
                'application_id' => $applicationId,
                'supplemental_detail_id' => $detalId,
            ]);
        }

        foreach ($supplementalCompletionIds as $completionId) {
            SupplementalCompletionUser::create([
                'application_id' => $applicationId,
                'supplemental_completion_id' => $completionId,
            ]);
        }
    }

    public function updateFileAttachment($entityName, $id, $application_id, $filePath, $status)
    {

        if ($entityName === 'SupplementaryRequirements') {
            SupplementalRequirementUser::where('id', $id)
                ->where('application_id', $application_id)
                ->update([
                    'attachment' => $filePath,
                    'status' => $status
                ]);
        } elseif ($entityName === 'OtherRequirements') {
            SupplementalDetailUser::where('id', $id)
                ->where('application_id', $application_id)
                ->update([
                    'attachment' => $filePath,
                    'status' => $status
                ]);
        } elseif ($entityName === 'CompletionAccomplishment') {
            SupplementalCompletionUser::where('id', $id)
                ->where('application_id', $application_id)
                ->update([
                    'attachment' => $filePath,
                    'status' => $status
                ]);
        }

    }

    public function downloadFileAttachment($entityName, $id, $application_id)
    {
        $application = PromotionApplication::where('id', $application_id)->first();
        if ($entityName === 'SupplementaryRequirements') {
            return SupplementalRequirementUser::where('id', $id)
                ->where('application_id', $application->id)
                ->first();
        } elseif ($entityName === 'OtherRequirements') {
            return SupplementalDetailUser::where('id', $id)
                ->where('application_id', $application->id)
                ->first();
        } elseif ($entityName === 'CompletionAccomplishment') {
            return SupplementalCompletionUser::where('id', $id)
                ->where('application_id', $application->id)
                ->first();
        }

    }

    public function updateRequirementStatus($id, $application_id, $status, $justification, $entityName)
    {
        $application = PromotionApplication::where('id', $application_id)->with('user')->first();
        if ($entityName === 'SupplementaryRequirements') {
            $requirements = SupplementalRequirementUser::where('id', $id)
                ->where('application_id', $application_id)->first();
            $requirementDetails = $requirements->supplementalRequirement()->first();
        } elseif ($entityName === 'OtherRequirements') {
            $requirements = SupplementalDetailUser::where('id', $id)
                ->where('application_id', $application_id)->first();
            $requirementDetails = $requirements->supplementalDetail()->first();
        } elseif ($entityName === 'CompletionAccomplishment') {
            $requirements = SupplementalCompletionUser::where('id', $id)
                ->where('application_id', $application_id)->first();
            $requirementDetails = $requirements->supplementalCompletion()->first();
        }

        $oldRequirements = clone $requirements;

        $requirements->update([
            'status' => $status,
            'justification' => $justification,
            'updated_at' => now()
        ]);

        $requirements->attributes = collect($requirements);
        $requirements->old = collect($oldRequirements);

        $statusMessage = [
            1 => 'pending',
            2 => 'approved',
            3 => 'declined'
        ];

        $fullName = $application->user->name;

        $this->logCustomMessage("{$statusMessage[$status]}_employee_requirements", $requirements, Auth::user()->name . " {$statusMessage[$status]} {$requirementDetails->title} for employee {$fullName}", $requirements, ucfirst($statusMessage[$status]) . ' employee requirements', new Activity());

        if ($status == 2)  {
            $customMessage = 'Approved';
        } elseif ($status == 3) {
            $customMessage = 'Disapproved';
        } else {
            $customMessage = '';
        }

        NotificationMessage::notifySenderAndReceiver('Review Requirements', $application->user_id, Auth::id(), null, null, null, null, $customMessage);
    }

    public function processApplication($application_id, $step, $status, $schedule = null, $interviewer = null, $link = null, $description = null)
    {
        $application = PromotionApplication::where('id', $application_id)->with('process','phs')->first();

        // if ($status != ProcessStatus::RESCHEDULE || $status != ProcessStatus::DECLINE) {
        //     $step += 1;
        // }

        $statusMessage = [
            1 => 'pending',
            2 => 'approved',
            3 => 'declined',
            4 => 'reschedule',
            5 => 'acknowledge'
        ];

        $this->logCustomMessage("{$statusMessage[$status]}_employee_requirement", $application, Auth::user()->name . " {$statusMessage[$status]} an employee requirement", $application, "{$statusMessage[$status]} employee requirement", new Activity());

        $this->generateNotification($application->user_id, $application->phs->hiring_manager->id, $step, $status, $schedule);

        if ($status == ProcessStatus::APPROVE) {
            $status = ProcessStatus::PENDING;
            $application->process->reschedule()->forceDelete();
        }

        if ($status == ProcessStatus::DECLINE) {
            $application->status = PromotionApplicationStatus::FAILED;
        }

        $application->process->update([
            'schedule' => $schedule,
            'interviewer' => $interviewer,
            'link' => $link,
            'description' => $description,
            'step' => $step,
            'status' => $status,
        ]);
    }

    public function requestReschedule($application_id, $reason, $schedule = null)
    {
        $application = PromotionApplication::where('id', $application_id)->with('process','phs')->first();

        $application->process->reschedule()->create([
            'schedule' => $schedule,
            'reason' => $reason,
            'status' => RescheduleStatus::PENDING
        ]);

        $process = $application->process;
        $process->status = ProcessStatus::RESCHEDULE;
        $process->save();

        if ($application->process->step == ProcessSteps::EXAM) {
            NotificationMessage::notifySenderAndReceiver('Request Exam Reschedule', $application->phs->hiring_manager->id, $application->user_id);
        }
        elseif ($application->process->step == ProcessSteps::INTERVIEW) {
            NotificationMessage::notifySenderAndReceiver('Request Interview Reschedule', $application->phs->hiring_manager->id, $application->user_id);
        }

        $this->logCustomMessage('request_reschedule_promotion', $application, Auth::user()->name . ' requested a reschedule for exam/interview', $application, 'Requested reschedule promotion', new Activity());
    }

    public function processReschedule($application_id, $schedule = null, $status, $reason)
    {
        $application = PromotionApplication::where('id', $application_id)->with('process','phs')->first();
        $reschedule = $application->process->reschedule;
        $process = $application->process;

        if ($status == RescheduleStatus::HR_APPROVED) {
            $process->schedule = $schedule;
            if ($reschedule) {
                $reschedule->schedule = $schedule;
            }
        }

        $process->status = ProcessStatus::PENDING;
        $process->save();

        if ($reschedule) {
            $reschedule->status = $status;
            $reschedule->reason = $reason;
            $reschedule->save();
        }

        $statusMessage = [
            1 => 'pending',
            2 => 'approved',
            3 => 'rejected',
        ];

        if ($application->process->step == ProcessSteps::EXAM && $status == RescheduleStatus::HR_APPROVED) {
            NotificationMessage::notifySenderAndReceiver('Approve Reschedule Exam', $application->user_id, Auth::id());
        }
        elseif ($application->process->step == ProcessSteps::EXAM && $status == RescheduleStatus::HR_DECLINED) {
            NotificationMessage::notifySenderAndReceiver('Disapprove Reschedule Exam', $application->user_id, Auth::id());
        }
        elseif ($application->process->step == ProcessSteps::INTERVIEW && $status == RescheduleStatus::HR_APPROVED) {
            NotificationMessage::notifySenderAndReceiver('Approve Reschedule Interview', $application->user_id, Auth::id());
        }
        elseif ($application->process->step == ProcessSteps::INTERVIEW && $status == RescheduleStatus::HR_DECLINED) {
            NotificationMessage::notifySenderAndReceiver('Disapprove Reschedule Interview', $application->user_id, Auth::id());
        }

        $this->logCustomMessage("{$statusMessage[$status]}_employee_reschedule_exam", $application, Auth::user()->name . " {$statusMessage[$status]} an employee reschedule exam", $application, "{$statusMessage[$status]} employee reschedule exam", new Activity());
    }

    public function generateForm($applicantion_id)
    {
        $application = PromotionApplication::where('id', $applicantion_id)->with('user', 'phs')->first();
        $fullName = $application->user->name;

        $employee_type = EmployeeType::where('id', $application->phs->employee_type)->first();

        $salary = str_replace(',', '', $application->phs->salary->value);
        $num_to_words = new NumberFormatter('en', NumberFormatter::SPELLOUT);

        $post_due = new DateTime($application->phs->post_due);

        $designation = ActualDesignation::where('user_id', $application->user_id)
        ->with('office','department', 'division','section','unit')
        ->first();
        $relationships = ['office', 'department', 'division', 'section', 'unit'];
        $actualDesignation = [];
        foreach ($relationships as $relationship) {
            if ($designation->$relationship) {
                $actualDesignation[] = $designation->$relationship->name;
            }
        }
        $actualDesignation = implode(' / ', $actualDesignation);

        $uuid = uniqid();
        $uuid_1 = uniqid();
        $uuid_2 = uniqid();
        $uuid_3 = uniqid();
        $uuid_4 = uniqid();

        $data = [
            'current_date' => Carbon::now()->format('F j, Y'),
            'name' => strtoupper($application->user->name),
            'from' => $application->phs->created_at->format('m/d/Y'),
            'to' => $post_due->format('m/d/Y'),
            'position' => strtoupper($application->phs->position->title),
            'item_code' => $application->phs->itemCodes->item_code,
            'employee_type' => strtoupper($employee_type->title),
            'salary_words' => str_replace('-', ' ', strtoupper($num_to_words->format($salary))),
            'salary_number' => '( Php ' . number_format($salary, 2, '.', ',') . ' )',
            'SG' => $application->phs->salary->salary_grade . "." .$application->phs->salary->step,
            'nature_of_appointment' => 'Promotion',
            'education' => $application->phs->education,
            'experience' => $application->phs->experience,
            'training' => $application->phs->training,
            'eligibility' => $application->phs->eligibility,
            'goverment_department' => 'DEPARTMENT OF HEALTH',
            'bureau_office' => 'NATIONAL KIDNEY & TRANSPLANT INSTITUTE',
            'company_address' => 'East Avenue, Diliman, Quezon City 1101, Philippines',
            'department' => $actualDesignation,
        ];

        $pdf = new Pdf(public_path('csc-form-template/Appointment_Form_acroform_v2.1.pdf'), config('pdftk.options'));
        $pdf_assumption = new Pdf(public_path('csc-form-template/assumption_of_duty_form.pdf'), config('pdftk.options'));
        $pdf_oath = new Pdf(public_path('csc-form-template/oath_of_office_form.pdf'), config('pdftk.options'));
        $pdf_position = new Pdf(public_path('csc-form-template/position_description.pdf'), config('pdftk.options'));

        if (!Storage::disk('promotion_appointment_form')->exists($application->user->id)) {
            Storage::disk('promotion_appointment_form')->makeDirectory($application->user->id);
        }

        $pdf->fillForm($data)->needAppearances()
            ->saveAs(storage_path('app/promotion-appointment-form/' . $application->user->id . '/appointment_form_template' . $uuid . '.pdf'));
        $pdf_assumption->fillForm($data)->needAppearances()
            ->saveAs(storage_path('app/promotion-appointment-form/' . $application->user->id . '/assumption_of_duty_template' . $uuid_1 . '.pdf'));
        $pdf_oath->fillForm($data)->needAppearances()
            ->saveAs(storage_path('app/promotion-appointment-form/' . $application->user->id . '/oath_of_office_template' . $uuid_2 . '.pdf'));
        $pdf_position->fillForm($data)->needAppearances()
            ->saveAs(storage_path('app/promotion-appointment-form/' . $application->user->id . '/position_description_template' . $uuid_3 . '.pdf'));

        $file_paths = [
            storage_path('app/promotion-appointment-form/' . $application->user->id . '/appointment_form_template' . $uuid . '.pdf'),
            storage_path('app/promotion-appointment-form/' . $application->user->id . '/assumption_of_duty_template' . $uuid_1 . '.pdf'),
            storage_path('app/promotion-appointment-form/' . $application->user->id . '/oath_of_office_template' . $uuid_2 . '.pdf'),
            storage_path('app/promotion-appointment-form/' . $application->user->id . '/position_description_template' . $uuid_3 . '.pdf')
        ];

        if (!Storage::disk('local')->exists('job-form-zip')) {
            Storage::disk('local')->makeDirectory('job-form-zip');
        }

        $zipFilePath = storage_path('app/job-form-zip/' . $uuid_4 . '.zip');
        $zip = new ZipArchive();
        $zip->open($zipFilePath, ZipArchive::CREATE | ZipArchive::OVERWRITE);

        foreach ($file_paths as $file_path) {
            $file = basename($file_path);
            $zip->addFile($file_path, $file);
        }

        $zip->close();

        foreach ($file_paths as $file_path) {
            unlink($file_path);
        }

        $this->logCustomMessage('generate_appointment_form', $application, Auth::user()->name . ' generated the appointment form for employee ' . $fullName, $application, 'Generate appointment form', new Activity());

        $filePath = Storage::disk('local')->path('job-form-zip/' . $uuid_4 . '.zip');

        return $filePath;
    }

    public function uploadForm($application_id, $files)
    {
        $path = 'employee/' . $application_id;

        $application = PromotionApplication::where('id', $application_id)->with('user')->first();

        foreach ($files as $file) {
            $fileName = $file->getClientOriginalname();
            $filePath = Storage::disk('promotion_appointment_form')->putFileAs($path, $file, $fileName);
            FileVault::disk('promotion_appointment_form')->encrypt($filePath);

            $application->files()->updateOrCreate([
                'application_id' => $application->id,
                'file_path' => $filePath,

            ], [
                'created_by' => Auth::id(),
                'updated_by' => Auth::id()
            ]);
        }
        $fullName = $application->user->name;

        NotificationMessage::notifyReceiver('Upload Form', $application->user_id, Auth::id());

        $this->logCustomMessage('upload_appointment_form', $application, Auth::user()->name . ' uploaded an appointment form for employee ' . $fullName, $application, 'Upload appointment form', new Activity());

    }

    public function downloadForm($application_id)
    {
        $application = PromotionApplication::where('id', $application_id)->with('files')->first();

        if (!Storage::disk('local')->exists('job-form-zip')) {
            Storage::disk('local')->makeDirectory('job-form-zip');
        }

        $fullName = $application->user->name;
        $uuid = uniqid();
        $filePathsToRemove = [];
        $zipFilePath = storage_path('app/job-form-zip/' . $uuid . '.zip');
        $zip = new ZipArchive();
        $zip->open($zipFilePath, ZipArchive::CREATE | ZipArchive::OVERWRITE);

        foreach ($application->files as $file) {
            $filePath = $file->file_path . '.enc';
            FileVault::disk('promotion_appointment_form')->decryptCopy($filePath);
            $decryptedFilePath = Storage::disk('promotion_appointment_form')->path($file->file_path);
            $filePathsToRemove[] = $decryptedFilePath;

            $pdf = new Pdf($decryptedFilePath, config('pdftk.options'));
            $result = $pdf->flatten()->saveAs($decryptedFilePath);

            if (!$result) {
                throw new FileException('There was a problem processing your file form');
            }

            $file = basename($decryptedFilePath);
            $zip->addFile($decryptedFilePath, $file);
        }

        $zip->close();

        foreach ($filePathsToRemove as $filePath) {
            unlink($filePath);
        }

        $this->logCustomMessage('download_appointment_form', $application, Auth::user()->name . ' downloaded an appointment form for employee ' . $fullName, $application, 'Download appointment form', new Activity());

        $filePath = Storage::disk('local')->path('job-form-zip/' . $uuid . '.zip');

        return $filePath;
    }

    public function promoteEmployee($application_id, $item_code_id, $phs_id, $corp_id = null, $branch_id = null, $division_id, $department_id)
    {
        DB::connection(env('HRJP_DB_CONNECTION'))->beginTransaction();
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $application = PromotionApplication::where('id', $application_id)->with('user', 'phs')->first();
            $user = $application->user;
            $employeeMetaInfo = $user->employeeMetaInfo;
            $phs = PHS::where('id', $phs_id)->first();

            //Update MetaInfo
            $employeeMetaInfo->position_id = $phs->position_id;
            $employeeMetaInfo->item_code_id = $item_code_id;
            $employeeMetaInfo->division_id = $division_id;
            $employeeMetaInfo->department_id = $department_id;
            $employeeMetaInfo->corp_id = $corp_id;
            $employeeMetaInfo->branch_id = $branch_id;
            $employeeMetaInfo->save();

            //create Service record

            ServiceRecord::where('user_id', $user->id)->whereNull('end_date')->update([
                'end_date' => date('Y-m-d', strtotime(now()))
            ]);

            $totalAllowance = 0;
            $allowance = EmployeeAllowance::where('user_id', $user->id)->get();
            foreach ($allowance as $allowances) {
                $totalAllowance += $allowances['amount'];
            }
            $salary = $phs->getSalaryAttribute($phs->salary)->value;

            $activeSector = Sector::where('status', 'active')->first();
            $sector = $activeSector->name;

            if($sector == 'public'){
                $newSalary = PublicSalary::whereHas('userSalary', function ($query) use($user){
                    $query->where('user_id', $user->id);
                })->first();

                $newSalary->update([
                    'salary_grade_id' => $phs->getSalaryAttribute($phs->salary)->id
                ]);

            }
            else{
                $newSalary = PrivateSalary::whereHas('userSalary', function ($query) use($user){
                    $query->where('user_id', $user->id);
                })->first();

                $newSalary->update([
                    'salary_grade_id' => $phs->getSalaryAttribute($phs->salary)->id
                ]);

            }

            $remark = Remark::whereId(2)->first();
            $user->serviceRecord()->create([
                'start_date' => date('Y-m-d', strtotime(now())),
                'position_id' => $phs->position_id,
                'salary' => floatval($salary) * 12,
                'allowance' => $totalAllowance,
                'employee_type_id' => $phs->employee_type,
                'division_id' => $division_id,
                'remark_id' => $remark->id,
                'created_by' => Auth::id(),
                'updated_by' => Auth::id()
            ]);


            //Update Position History
            $old_position_record = Position_history::where('user_id', $user->id)->latest()->first();
            if ($old_position_record) {
                $previous_position = 2;
                $old_position_record->update([
                    'position_status' => $previous_position,
                    'updated_by' => Auth::id(),
                ]);
            }

            $position_history = Position_history::create([
                'user_id' => $user->id,
                'position_id' => $phs->position_id,
                'appointment_date' => now(),
                'date_of_effectivity' => now(),
                'employment_status' => 0, //* currently employed
                'position_status' => 1, //* current position
                'created_by' => Auth::id()
            ]);

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            DB::connection(env('HRJP_DB_CONNECTION'))->commit();

            DB::connection(env('HRJP_DB_CONNECTION'))->beginTransaction();
            DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();

            $employeeExtraFieldColumn = EmployeeExtraFieldColumn::whereHas('employeeExtraField', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })->get();

            $data = [];
            foreach ($employeeExtraFieldColumn as $pds) {
                $data[] = [
                    'position_history_id' => $position_history->id,
                    'eefc_id' => $pds->id,
                    'field_name' => $pds->field_name,
                    'field_value' => $pds->field_value,
                    'created_by' => Auth::id()
                ];
            }
            PDSPerPosition::insert($data);

            //update statuses
            $application->status = PromotionApplicationStatus::COMPLETED;
            $application->save();

            $process = $application->process;
            $process->status = ProcessStatus::APPROVE;
            $process->step = 5; //Hired
            $process->save();

            $notPosted = 2;
            $phs->post_status = $notPosted;
            $phs->save();

            ItemCode::where('id', $item_code_id)->update([
                'status' => ItemCodeStatus::ACTIVE
            ]);

            $this->logCustomMessage('promote_employee', $application, Auth::user()->name . ' Promoted an employee ' . $user->name, $application, 'Promote employee', new Activity());
            NotificationMessage::notifySenderAndReceiver('Promote Employee', $application->user_id, Auth::id());
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            DB::connection(env('HRJP_DB_CONNECTION'))->commit();
        } catch (Exception $errors) {
            DB::connection(env('HRJP_DB_CONNECTION'))->rollBack();
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            throw $errors;
        }
    }

    public function generateNotification($employee, $hiringManager, $step, $status, $schedule)
    {
        if ($step == ProcessSteps::EXAM && $status == ProcessStatus::PENDING && $schedule == null) {
            NotificationMessage::notifySenderAndReceiver('Approve Requirements', $employee, Auth::id());
        }
        elseif ($step == ProcessSteps::EXAM && $status == ProcessStatus::PENDING && $schedule) {
            NotificationMessage::notifySenderAndReceiver('Set Schedule Exam', $employee, Auth::id());
        }
        elseif ($step == ProcessSteps::EXAM && $status == ProcessStatus::ACKNOWLEDGE && $schedule) {
            NotificationMessage::notifySenderAndReceiver('Acknowledge Exam', $employee, $hiringManager);
        }
        elseif ($step == ProcessSteps::EXAM && $status == ProcessStatus::DECLINE && $schedule) {
            NotificationMessage::notifySenderAndReceiver('Fail Exam', $employee, Auth::id());
        }
        elseif ($step == ProcessSteps::INTERVIEW && $status == ProcessStatus::APPROVE && $schedule == null) {
            NotificationMessage::notifySenderAndReceiver('Pass Exam', $employee, Auth::id());
        }
        elseif ($step == ProcessSteps::INTERVIEW && $status == ProcessStatus::PENDING && $schedule) {
            NotificationMessage::notifySenderAndReceiver('Set Schedule Interview', $employee, Auth::id());
        }
        elseif ($step == ProcessSteps::INTERVIEW && $status == ProcessStatus::ACKNOWLEDGE && $schedule) {
            NotificationMessage::notifySenderAndReceiver('Acknowledge Interview', $employee, $hiringManager);
        }
        elseif ($step == ProcessSteps::INTERVIEW && $status == ProcessStatus::DECLINE && $schedule) {
            NotificationMessage::notifySenderAndReceiver('Fail Interview', $employee, Auth::id());
        }
        elseif ($step == ProcessSteps::APPOINTMENT && $status == ProcessStatus::APPROVE && $schedule == null) {
            NotificationMessage::notifySenderAndReceiver('Pass Interview', $employee, Auth::id());
        }

    }
}
