<?php

namespace Suiterus\Adg\Services\Payroll;

use Carbon\Carbon;
use App\Enums\Log\PayrollLogType;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Enums\Payroll\SpecialPayFrequency;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;

class SpecialPayService 
{

    use HasCustomLogs;

    public function addEmployee($employeeIds, $specialPayId, $amount, $title) 
    {
        foreach ($employeeIds as $employeeId) {
            $hasEmployeeSpecialPay = EmployeeSpecialPay::whereHas('special_pay', function ($query) use($title) {
                $query->where('title', $title);
            })->where('user_id', $employeeId)->first();

            if ($hasEmployeeSpecialPay) {
                DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
                return response()->json([
                    'errors' => ['Employee has an existing special pay named ' . $title],
                ], 400);
            }
            
            $employeeSpecialPay = EmployeeSpecialPay::create([
                'user_id' => $employeeId,
                'special_pay_id' => $specialPayId,
                'amount' => $amount,
                'created_by' => Auth::id(),
            ]);

            $this->logCustomMessage(
                'add_employee_special_pay',
                $employeeSpecialPay,
                Auth::user()->name . ' added ' . $employeeSpecialPay->user->name . ' to the special pay: ' . $employeeSpecialPay->special_pay->title,
                $employeeSpecialPay,
                PayrollLogType::ADD_EMPLOYEE_SPECIAL_PAY,
                new Activity()
            );
        };

        return null;
    }

    public function getSpecialPayByFrequency($specialPay, $employeeSpecialPay, $startDate, $endDate, $isSemiMonthly) {

        $specialPayStartDate = Carbon::parse($specialPay->start_date);
        $specialPayEndDate = Carbon::parse($specialPay->end_date);
        $specialPayFrequency = $specialPay->special_pay_frequency_id;
    
        if ($startDate >= $specialPayStartDate && $endDate <= $specialPayEndDate) {
            $amount = $employeeSpecialPay / $specialPay->period;
            $endDateCarbon = Carbon::parse($endDate);
            $endDateDay = $endDateCarbon->day;
            $endDateMonth = $endDateCarbon->month;

            $quarterlyMonths = $this->getQuarterlyMonths($specialPayStartDate, $specialPayEndDate);
            $semiAnnualMonths = $this->getSemiAnnualMonths($specialPayStartDate, $specialPayEndDate);
            $annualMonths = $this->getAnnualMonths($specialPayStartDate, $specialPayEndDate);

            if ($specialPayFrequency == SpecialPayFrequency::MONTHLY) {
                if ($endDateCarbon->endOfMonth()->equalTo($endDateCarbon)) {
                    return $this->calculateAmountIfSemiMonthlyOrMonthly($amount, $isSemiMonthly, $endDateDay, $endDateCarbon);
                }
            }
            else if ($specialPayFrequency == SpecialPayFrequency::SEMI_MONTHLY) {
                if (!$isSemiMonthly) {
                    if ($endDateCarbon->endOfMonth()->equalTo($endDateCarbon)) {
                        return $amount * 2;
                    }
                }

                if ($endDateDay == 15 || $endDateDay == 16 || $endDateCarbon->endOfMonth()->equalTo($endDateCarbon)) {
                    return $amount;
                }
                
            }
            else if ($specialPayFrequency == SpecialPayFrequency::QUARTERLY) {
                if (in_array($endDateMonth, $quarterlyMonths)) {
                    return $this->calculateAmountIfSemiMonthlyOrMonthly($amount, $isSemiMonthly, $endDateDay, $endDateCarbon);
                }
            }
            else if ($specialPayFrequency == SpecialPayFrequency::SEMI_ANNUALLY) {
                if (in_array($endDateMonth, $semiAnnualMonths)) {
                    return $this->calculateAmountIfSemiMonthlyOrMonthly($amount, $isSemiMonthly, $endDateDay, $endDateCarbon);
                }
            }
            else if ($specialPayFrequency == SpecialPayFrequency::ANNUALLY) {
                if (in_array($endDateMonth, $annualMonths)) {
                    return $this->calculateAmountIfSemiMonthlyOrMonthly($amount, $isSemiMonthly, $endDateDay, $endDateCarbon);
                }
            }
        }
    
        return 0; 
    }

    private function calculateAmountIfSemiMonthlyOrMonthly($amount, $isSemiMonthly, $endDateDay, $endDateCarbon)
    {
        if ($isSemiMonthly) {
            if ($endDateDay == 15 || $endDateDay == 16 || $endDateCarbon->endOfMonth()->equalTo($endDateCarbon)) {
                return $amount / 2;
            }
        }
        return $amount;
    }

    private function getQuarterlyMonths($startDate, $endDate) {
        $months = [];

        $current = $startDate->copy();
        while ($current <= $endDate) {
            $months[] = $current->month;
            $current->addMonths(3);
        }
    
        return $months;
    }
    
    private function getSemiAnnualMonths($startDate, $endDate) {
        $months = [];

        $current = $startDate->copy();
        while ($current <= $endDate) {
            $months[] = $current->month;
            $current->addMonths(6);
        }
    
        return $months;
    }

    private function getAnnualMonths($startDate, $endDate) {
        $months = [];

        $current = $startDate->copy();
        while ($current <= $endDate) {
            $months[] = $current->month;
            $current->addMonths(12);
        }
    
        return $months;
    }
}