<?php

namespace Suiterus\Adg\Services\OrgStructure;

use Exception;
use App\Models\User;
use Suiterus\Adg\Models\SM\Unit;
use Suiterus\Adg\Models\SM\Office;
use Suiterus\Adg\Models\SM\Section;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Database\Eloquent\Model;
use Suiterus\Adg\Models\SM\OfficeHierarchy;
use Suiterus\Adg\Models\SPMS\PCRFormPermission;

class OrgStructureService
{

    public function getPcrConfig($model)
    {

        if (!$model) {
            return [];
        }

        $pcrs = $model->pcrs;

        if (!isset($pcrs[0])) {
            if ($model instanceof Unit) {
                $unitParent = $model->section ?? $model->division;
                $unitParent = $unitParent ?? $model->department;
                $unitParent = $unitParent ?? $model->office;
                $pcrs = $this->getPcrConfig($unitParent);
            }
            if ($model instanceof Section) {
                $pcrs = $this->getPcrConfig($model->division);
            }
            if ($model instanceof Division) {
                $divisionParent = $model->department ?? $model->office;
                $pcrs = $this->getPcrConfig($divisionParent);
            }
            if ($model instanceof Department) {
                $pcrs = $this->getPcrConfig($model->office);
            }

            if ($model instanceof Office) {
                $hierarchy = OfficeHierarchy::where('child_office_id', $model->id)->first();

                if ($hierarchy) {
                    return $pcrs;
                }

                if (isset($hierarchy->parentOffice)) {
                    $pcrs = $this->getPcrConfig($hierarchy->parentOffice);
                }
            }
        }

        return $pcrs;
    }

    public function getMinimumOvertimeConfig($model)
    {
        if (!$model) {
            return [];
        }

        $overtime = $model->overtime;

        if (!isset($overtime[0])) {
            if ($model instanceof Unit) {
                $unitParent = $model->section ?? $model->division;
                $unitParent = $unitParent ?? $model->department;
                $unitParent = $unitParent ?? $model->office;
                $overtime = $this->getMinimumOvertimeConfig($unitParent);
            }
            if ($model instanceof Section) {
                $overtime = $this->getMinimumOvertimeConfig($model->division);
            }
            if ($model instanceof Division) {
                $divisionParent = $model->department ?? $model->office;
                $overtime = $this->getMinimumOvertimeConfig($divisionParent);
            }
            if ($model instanceof Department) {
                $overtime = $this->getMinimumOvertimeConfig($model->office);
            }

            if ($model instanceof Office) {
                $hierarchy = OfficeHierarchy::where('child_office_id', $model->id)->first();

                if ($hierarchy) {
                    return $overtime;
                }

                if (isset($hierarchy->parentOffice)) {
                    $overtime = $this->getMinimumOvertimeConfig($hierarchy->parentOffice);
                }
            }
        }

        return $overtime;
    }

    public function configMinimumOvertime(Model $model, $overtime, $createdBy = null)
    {
        $validClass = [
            Office::class,
            Department::class,
            Division::class,
            Section::class,
            Unit::class,
        ];

        if (!in_array(get_class($model), $validClass)) {
            throw new Exception('Invalid model type');
        }

        $model->overtime()->delete();

        if ($overtime != '00:00') {
            $overtime = $model->overtime()->create([
                'minimum_overtime' => $overtime,
                'created_by' => $createdBy ?? Auth::id(),
            ]);
        }
    }

    public function configPcr(Model $model, $pcrs, $createdBy = null)
    {
        $validClass = [
            Office::class,
            Department::class,
            Division::class,
            Section::class,
            Unit::class,
        ];

        if (!in_array(get_class($model), $validClass)) {
            throw new Exception('Invalid model type');
        }

        $model->pcrs()->delete();

        foreach ($pcrs as $pcr) {
            if (isset($pcr['pcfr_selected'])) {
                $pcrs = $model->pcrs()->create([
                    'pcr_form_type_id' => $pcr['pcfr_selected'],
                    'created_by' => $createdBy ?? Auth::id(),
                ]);

                foreach ($pcr['mfo_classification_type'] as $mfo) {
                    //insert selected mfo classification type
                    if (isset($mfo['mfo_classification_id'])) {
                        $pcrs->mfos()->create([
                            'mfo_id' => $mfo['mfo_classification_id'],
                            'percentage' => $mfo['mfo_percentages'],
                        ]);
                    }
                }
            }
        }
    }

    public function getHeadEmployeePcrs()
    {

        $organizations = [];

        $unit = Unit::where('head_employee_id', Auth::id())->get();

        if ($unit) {
            $organizations = [...$organizations, ...$unit];
        }

        $section = Section::where('head_employee_id', Auth::id())->get();

        if ($section) {
            $organizations = [...$organizations, ...$section];
        }

        $division = Division::where('head_employee_id', Auth::id())->get();

        if ($division) {
            $organizations = [...$organizations, ...$division];
        }

        $department = Department::where('head_employee_id', Auth::id())->get();

        if ($department) {
            $organizations = [...$organizations, ...$department];
        }

        $office = Office::where('head_employee_id', Auth::id())->get();

        if ($office) {
            $organizations = [...$organizations, ...$office];
        }

        $headEmployeePcrs = [];

        foreach ($organizations as $organization) {
            $headEmployeePcrs = [...$headEmployeePcrs, ...$this->getPcrConfig($organization)];
        }

        return $headEmployeePcrs;
    }

    public function getEmployeePcrPermissions()
    {
        $permissions = [];

        $type = $this->verifyHeadEmployee();

        if (!$type) {
            $permissions = PCRFormPermission::where('name', 'Rank and File')->get();
        }

        //head employee
        if ($type) {
            if ($type['unit'] || $type['section']) {
                $permissions = PCRFormPermission::where('name', 'Supervisor')->get();
            }

            if ($type['division']) {
                $permissions = [...$permissions,  ...PCRFormPermission::where('name', 'Division Chief')->get()];
            }

            if ($type['department']) {
                $permissions = [...$permissions,  ...PCRFormPermission::where('name', 'Department Head')->get()];
            }

            if ($type['office']) {
                $permissions = [...$permissions,  ...PCRFormPermission::where('name', 'Executive Director/Deputy Executive Director')->get()];
            }
        }

        return collect($permissions);
    }

    public function verifyHeadEmployee()
    {

        $unit = Unit::where('head_employee_id', Auth::id())->first();
        $section = Section::where('head_employee_id', Auth::id())->first();
        $division = Division::where('head_employee_id', Auth::id())->first();
        $department = Department::where('head_employee_id', Auth::id())->first();
        $office = Office::where('head_employee_id', Auth::id())->first();

        if ($unit || $section || $division || $department || $office) {
            return [
                'unit' => $unit,
                'section' => $section,
                'division' => $division,
                'department' => $department,
                'office' => $office
            ];
        }

        return false;
    }

    public function getEmployeeOrganizationChild(User $user, $recurseOrganization = null){

        if (!$user->actualDesignation) {
            return null;
        }

        $officeIds = [];
        $departmentIds = [];
        $divisionIds = [];
        $sectionIds = [];
        $unitIds = [];

        $organization = $recurseOrganization ?? $user->actualDesignation->unit;
        $organization = $organization ?? $user->actualDesignation->section;
        $organization = $organization ?? $user->actualDesignation->division;
        $organization = $organization ?? $user->actualDesignation->department;
        $organization = $organization ?? $user->actualDesignation->office;

        if ($organization instanceof Office) {
            $officeIds = [$organization->id, ...$this->getNestedOffices($organization->childOffices)];
            $departmentIds = Department::whereIn('office_id', $officeIds)->pluck('id')->toArray();
            $divisionIds = Division::whereIn('department_id', $departmentIds)->orWhereIn('office_id', $officeIds)->pluck('id')->toArray();
            $sectionIds = Section::whereIn('division_id', $divisionIds)->pluck('id')->toArray();
            $unitIds = Unit::whereIn('section_id', $sectionIds)->orWhereIn('division_id', $divisionIds)->orWhereIn('department_id', $departmentIds)->orWhereIn('office_id', $officeIds)->pluck('id')->toArray();
        }else if ($organization instanceof Department) {
            $divisionIds = Division::where('department_id', $organization->id)->pluck('id')->toArray();
            $sectionIds = Section::whereIn('division_id', $divisionIds)->pluck('id')->toArray();
            $unitIds = Unit::whereIn('section_id', $sectionIds)->orWhereIn('division_id', $divisionIds)->pluck('id')->toArray();
        }else if($organization instanceof Division) {
            $sectionIds = Section::where('division_id', $organization->id)->pluck('id')->toArray();
            $unitIds = Unit::whereIn('section_id', $sectionIds)->pluck('id')->toArray();
        }else if($organization instanceof Section) {
            $unitIds = Unit::where('section_id', $organization->id)->pluck('id')->toArray();
        }else if($organization instanceof Unit) {
            $unitIds = [$organization->id];
        }

        $type = false;

        if (!$recurseOrganization) {
            $type = $this->verifyHeadEmployee();
        }

        if ($type) {
            if ($type['unit']) {
                $headOrganization = $this->getEmployeeOrganizationChild($user, $type['unit']);
            }

            if ($type['section']) {
                $headOrganization = $this->getEmployeeOrganizationChild($user, $type['section']);
            }

            if ($type['division']) {
                $headOrganization = $this->getEmployeeOrganizationChild($user, $type['division']);
            }

            if ($type['department']) {
                $headOrganization = $this->getEmployeeOrganizationChild($user, $type['department']);
            }

            if ($type['office']) {
                $headOrganization = $this->getEmployeeOrganizationChild($user, $type['office']);
            }

            if (isset($headOrganization)) {
                $officeIds = [...$officeIds, ...$headOrganization['officeIds']];
                $departmentIds = [...$departmentIds, ...$headOrganization['departmentIds']];
                $divisionIds = [...$divisionIds, ...$headOrganization['divisionIds']];
                $sectionIds = [...$sectionIds, ...$headOrganization['sectionIds']];
                $unitIds = [...$unitIds, ...$headOrganization['unitIds']];
            }
        }

        return [
            'officeIds' => array_unique($officeIds),
            'departmentIds' => array_unique($departmentIds),
            'divisionIds' => array_unique($divisionIds),
            'sectionIds' => array_unique($sectionIds),
            'unitIds' => array_unique($unitIds)
        ];
    }

    public function getEmployeeCurrentOrg(User $user){
        $model = null;

        if (!$user->actualDesignation) {
            return;
        }

        $model = $user->actualDesignation->unit;
        $model = $model ?? $user->actualDesignation->section;
        $model = $model ?? $user->actualDesignation->division;
        $model = $model ?? $user->actualDesignation->department;
        $model = $model ?? $user->actualDesignation->office;

        return $model;
    }

    public function evaluationScoreMapper($models, $startDate = null, $endDate = null){
        return $models->map(function ($model) use($startDate, $endDate){
            $model->evaluation_score = $model->evaluation()->when($startDate && $endDate, function($query) use($startDate, $endDate){
                $query->whereDate('start_date', $startDate)->whereDate('end_date', $endDate);
            })->avg('overall_rate');
            $model->evaluation_score_adjective = $this->getEvaluationScoreAdjective($model->evaluation_score);
            return $model;
        });
    }

    public function getEvaluationScoreAdjective($score){
        $score = round($score, 2);
        if ($score == 5) {
            return "Outstanding";
        } else if ($score >= 4.0 && $score <= 4.99) {
            return "Very Satisfactory";
        } else if ($score >= 3.0 && $score <= 3.99) {
            return "Satisfactory";
        } else if ($score >= 2.0 && $score <= 2.99) {
            return "Unsatisfactory";
        } else if ($score >= 1.0 && $score <= 1.99) {
            return "Poor";
        }
    }

    private function getNestedOffices($offices, $childOfficeIds = []) {
        $childOfficeIds = [];

        foreach ($offices as $office) {
            $childOfficeIds[] = $office->id;
            if (isset($office->childOffices[0])) {
                $childOfficeIds = [...$childOfficeIds, ...$this->getNestedOffices($office->childOffices->pluck('childOffice'), $childOfficeIds)];
            }
        }

        return $childOfficeIds;
    }
}
