<?php

namespace Suiterus\Adg\Services\OrgStructure\Office;

use Carbon\Carbon;
use App\Models\User;
use Suiterus\Adg\Models\SM\Unit;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Office;
use Suiterus\Adg\Models\SM\Section;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Department;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Models\SM\OfficeHierarchy;
use Suiterus\Adg\Models\SM\OfficePerEmployee;

class OfficeService
{
    /**
     * The function assigns an office to a group of users.
     *
     * @param userIds An array of user IDs. It represents the IDs of the employees to whom the office is
     * being assigned.
     * @param Office office The "office" parameter is an instance of the "Office" class. It represents
     * the office that will be assigned to the employees specified by the "userIds" parameter.
     */
    public function assignOffice(?int $parentId, array $childIds = [], ?array $departmentIds, ?array $userIds, Office $office)
    {
        OfficeHierarchy::whereIn('child_office_id', $childIds)->orWhere('parent_office_id', $parentId)->delete();
        $this->assignParentOffice($parentId, $office);
        $this->assignChildOffice($childIds, $office);
        $this->assignOfficePerEmployee($userIds, $office);
        $this->assignDepartment($departmentIds, $office);
    }

    /**
     * The function assigns a parent office to a child office in a hierarchical structure.
     *
     * @param parentId The parentId parameter is an optional integer that represents the ID of the
     * parent office. If a parentId is provided, it means that the office being assigned has a parent
     * office. If the parentId is null, it means that the office being assigned does not have a parent
     * office.
     * @param Office office The "office" parameter is an instance of the "Office" class. It represents
     * the office that needs to be assigned a parent office.
     */
    public function assignParentOffice(?int $parentId, Office $office)
    {
        if ($parentId != null) {
            OfficeHierarchy::create([
                'parent_office_id' => $parentId,
                'child_office_id' => $office->id,
                'created_by' => Auth::id(),
            ]);
        }
    }

    /**
     * The function assigns child offices to a parent office in a hierarchical structure.
     *
     * @param childIds An optional array of child office IDs that need to be assigned to the given
     * office.
     * @param Office office An instance of the Office class, representing the parent office to which
     * the child offices will be assigned.
     */
    public function assignChildOffice(?array $childIds, Office $office)
    {
        if ($childIds != null) {
            foreach ($childIds as $childId) {
                OfficeHierarchy::create([
                    'parent_office_id' => $office->id,
                    'child_office_id' => $childId,
                    'created_by' => Auth::id(),
                ]);
            }
        }
    }

    /**
     * The function assigns an office to each employee specified by their user IDs.
     *
     * @param userIds An array of user IDs representing the employees to whom the office will be
     * assigned.
     * @param Office office The "office" parameter is an instance of the "Office" class. It represents
     * the office to which the employees will be assigned.
     */
    public function assignOfficePerEmployee(?array $userIds, Office $office)
    {
        if (isset($userIds)) {
            //remove existing employees from that office
            ActualDesignation::where('office_id', $office->id)->update([
                'office_id' => null,
            ]);

            //assign new employees to that office
            foreach ($userIds as $userId) {
                ActualDesignation::updateOrCreate(
                    [
                        'user_id' => $userId
                    ],
                    [
                        'office_id' => $office->id,
                        'department_id' => null,
                        'division_id' => null,
                        'section_id' => null,
                        'unit_id' => null,
                        'modified_by' => Auth::id(),
                    ]
                );
            }
        }
    }

    /**
     * The function assigns an office to multiple departments by updating their office_id field.
     *
     * @param departmentIds An array of department IDs that need to be assigned to the office.
     * @param Office office The `` parameter is an instance of the `Office` class. It represents
     * the office to which the departments will be assigned.
     */
    public function assignDepartment(?array $departmentIds, Office $office)
    {
        if (isset($departmentIds)) {
            foreach ($departmentIds as $departmentId) {
                $department = Department::whereId($departmentId)->first();
                $department->update([
                    'office_id' => $office->id,
                ]);
            }
        }
    }

    public function assignUnits(?array $unitIds, Office $office)
    {
        foreach ($unitIds as $unitId) {
            Unit::whereId($unitId)->update([
                'section_id' => null,
                'office_id' => $office->id,
                'department_id' => null,
                'division_id' => null
            ]);
        }

        return $this;
    }

    public function assignDivisions(?array $divisionIds, Office $office){
        foreach ($divisionIds as $divisionId) {
            Division::whereId($divisionId)->update([
                'office_id' => $office->id,
                'department_id' => null,
            ]);
        }

        return $this;
    }

    /**
     * The function deletes all office relations, including department relations, parent office
     * relations, child office relations, and employee office relations.
     *
     * @param office The parameter "office" is an instance of the Office model.
     */
    public function deleteOfficeRelations(Office $office)
    {
        Department::where('office_id', $office->id)->update(['office_id' => null]);

        $parentOffice = OfficeHierarchy::where('parent_office_id', $office->id)->first();
        if ($parentOffice) {
            OfficeHierarchy::where('parent_office_id', $office->id)->delete();
        }

        $childOffice = OfficeHierarchy::where('child_office_id', $office->id)->first();
        if ($childOffice) {
            OfficeHierarchy::where('child_office_id', $office->id)->delete();
        }

        OfficePerEmployee::where('office_id', $office->id)->delete();
    }

    public function getEmployees($officeId)
    {
        $parentOffice = Office::where('id', $officeId)->first();
        $childOffices = $this->getNestedOffices($parentOffice->childOffices->pluck('childOffice'));

        $officeIds = array_merge([intval($officeId)], $childOffices);

        $parentOfficeDepartments = $parentOffice->department->pluck('id')->toArray();

        $departmentIds = Department::select('id')->whereIn('id', $childOffices)->get()->toArray();

        $departmentIds = array_merge($parentOfficeDepartments, $departmentIds);

        $divisionsIds = Division::select('id')->whereIn('department_id', $departmentIds)->get()->toArray();

        $sectionIds = Section::select('id')->whereIn('division_id', $divisionsIds)->get()->toArray();

        $unitIds = Unit::select('id')->whereIn('section_id', $sectionIds)->get()->toArray();

        return User::whereHas('actualDesignation', function ($query) use ($officeIds, $departmentIds, $divisionsIds, $sectionIds, $unitIds) {
            $query->whereIn('office_id', $officeIds)->orWhereIn('department_id', $departmentIds)->orWhereIn('division_id', $divisionsIds)->orWhereIn('section_id', $sectionIds)->orWhereIn('unit_id', $unitIds);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();
    }

    private function getNestedOffices($offices, $childOfficeIds = [])
    {
        $childOfficeIds = [];

        foreach ($offices as $office) {
            $childOfficeIds[] = $office->id;
            if (isset($office->childOffices[0])) {
                $childOfficeIds = [...$childOfficeIds, ...$this->getNestedOffices($office->childOffices->pluck('childOffice'), $childOfficeIds)];
            }
        }

        return $childOfficeIds;
    }
}
