<?php

namespace Suiterus\Adg\Services\LeaveManagement;

use App\Models\User;
use mikehaertl\pdftk\Pdf;
use Illuminate\Support\Carbon;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Traits\EmployeeTrait;
use Illuminate\Support\Facades\Storage;
use SoareCostin\FileVault\Facades\FileVault;
use Illuminate\Contracts\Filesystem\Filesystem;
use Suiterus\Adg\Models\LeaveManagement\LeaveType;
use Suiterus\Adg\Models\LeaveManagement\LeaveApprover;
use Suiterus\Adg\Models\LeaveManagement\LeaveAllowance;
use Suiterus\Adg\Models\LeaveManagement\LeaveAttachment;
use Symfony\Component\HttpFoundation\File\Exception\FileException;

class LeaveService
{

    use EmployeeTrait;

    public function assignApprovers(array $approvers, int $leaveId)
    {
        foreach ($approvers as $approver) {
            LeaveApprover::create([
                'leave_id' => $leaveId,
                'approver_id' => $approver,
                'status' => 1,
            ]);
        }
    }

    /**
     * The function "getLeaveFormData" returns an array of data for a leave form, including information
     * about the employee, leave type, and dates.
     *
     * @param int selected_leave_type The selected_leave_type parameter is an integer that represents
     * the ID of the selected leave type.
     * @param int commutation A flag indicating whether commutation is requested or not. If the value
     * is 1, it means commutation is requested. If the value is 2, it means commutation is not
     * requested.
     * @param int working_days The parameter "working_days" represents the number of working days for
     * which the leave is being requested.
     * @param start_date The start date of the leave period.
     * @param end_date The end_date parameter is the date when the leave period ends.
     *
     * @return array an array containing the leave form data.
     */
    public function getLeaveFormData(User $user, int $selected_leave_type, int $commutation, $working_days, $start_date, $end_date): array
    {
        $this->initializeEmployeeInfo($user);
        $position = $user->employeeMetaInfo()->first()->position()->first();
        $department = $user->employeeMetaInfo()->first()->department()->first();

        $commutation = $commutation == 1 ? 'requested' : 'not_requested';

        $fillFormData = [
            'date_of_filing' => Carbon::now()->format('Y-m-d'),
            'position' => $position->title,
            'last_name' => $this->last_name,
            'first_name' => $this->first_name,
            'middle_name' => $this->middle_name,
            'salary' => $this->basic_salary,
            'department' => $department->name,
            'working_days' => $working_days ?? '',
            'inclusive_dates' => $start_date && $end_date ? $start_date . ' to ' . $end_date : '',
        ];

        $fillFormData[$commutation] = 'On';

        $leaveType = LeaveType::find($selected_leave_type);

        $leaveTypeTitle = strtolower($leaveType->title);

        $leaveTypeTitle = preg_replace('/\s+/', '_', $leaveTypeTitle);

        $fillFormData[$leaveTypeTitle] = 'On';

        return $fillFormData;
    }

    /**
     * The function creates a PDF leave form file by filling a form template with data and saving it to
     * a specified directory.
     *
     * @param Filesystem storage The `` parameter is an instance of the `Filesystem` class,
     * which is responsible for interacting with the file system. It provides methods for creating
     * directories, checking if a directory exists, and getting the path of a file.
     * @param String directory The directory parameter is a string that represents the directory where
     * the leave form file will be stored.
     * @param String fileName The `fileName` parameter is a string that represents the name of the PDF
     * file that will be created.
     * @param array fillFormData An array containing the data to be filled in the leave form. The keys
     * of the array should correspond to the field names in the PDF form, and the values should be the
     * data to be filled in those fields.
     * @param array pdfTkOptions The  parameter is an optional array that allows you to
     * specify additional options for the PDF generation process using the PDFtk library. PDFtk is a
     * popular command-line tool for manipulating PDF files.
     *
     * @return String a string representing the file path of the created leave form file.
     */
    public function createLeaveFormFile(Filesystem $storage, String $directory, String $fileName, array $fillFormData, array $pdfTkOptions = []): String
    {

        if (!$storage->exists($directory)) {
            $storage->makeDirectory($directory);
        }

        // Fill form with data array
        $leaveFormTemplate = public_path('csc-form-template/CS_Form_No._6_Revised_2020_Application_for_Leave_Secured.pdf');

        $pdf = new Pdf($leaveFormTemplate, $pdfTkOptions);

        $filePath = $directory . '/' . $fileName;

        $leaveFormPath = $storage->path($filePath);

        $result = $pdf->fillForm($fillFormData)->needAppearances()
            ->saveAs($leaveFormPath);

        if (!$result) {
            throw new FileException($pdf->getError());
        }

        FileVault::disk('leave_form')->encrypt($filePath);

        return $filePath;
    }

    /**
     * The function updates a leave form file by filling in form data, saving the updated file,
     * encrypting it, and deleting the original file.
     *
     * @param array data An array containing the data to be filled in the leave form PDF. The keys of
     * the array should correspond to the field names in the PDF form, and the values should be the
     * data to be filled in those fields.
     * @param String filePath The path to the leave form file that needs to be updated.
     * @param String fileName The `fileName` parameter is a string that represents the name of the file
     * to be updated. It is used to specify the name of the file when saving the updated PDF form.
     * @param String disk The "disk" parameter is used to specify the disk or storage location where
     * the file is stored. It is optional and defaults to 'leave_form' if not provided.
     *
     * @return If the `result` is `false`, the method will return the error message from the `pdf`
     * object. Otherwise, it will not return anything.
     */
    public function updateLeaveFormFile(array $data, String $filePath, String $fileName, String $disk = 'leave_form')
    {
        $storage = Storage::disk($disk);
        $fileVault = FileVault::disk($disk);
        $fileVault->decryptCopy($filePath);

        $decryptedPath = dirname($filePath) . '/' . $fileName;

        $pdf = new Pdf($storage->path($decryptedPath), config('pdftk.options'));

        $result = $pdf->fillForm($data)->saveAs($storage->path($decryptedPath));

        if ($result === false) {
            return $pdf->getError();
        }

        $fileVault->encrypt($decryptedPath);

        $storage->delete($decryptedPath);
    }


    /**
     * The function attaches files to a leave form PDF template and returns the paths of the attached
     * files.
     *
     * @param Filesystem storage The `` parameter is an instance of the `Filesystem` class,
     * which is responsible for managing files and directories. It provides methods for storing,
     * retrieving, and deleting files.
     * @param String templatePath The path to the template file that will be used for the leave form.
     * This file will be used as the base for generating the final PDF document.
     * @param attachments An array of file attachments that need to be attached to the leave form. Each
     * attachment should be an instance of a file object.
     * @param String path The `path` parameter is the directory path where the attachments will be
     * stored. It is used to create a subdirectory called "attachments" within the specified path and
     * store the attachments in that subdirectory.
     * @param array pdfTkOptions The  parameter is an optional array that allows you to
     * specify additional options for the PDF generation process using the PDFtk library. These options
     * can include things like password protection, page ranges, and other PDF manipulation options.
     *
     * @return either an array of attachment paths or an error message.
     */
    public function attachFilesToLeaveForm(Filesystem $storage, int $leaveId, String $templatePath, ?array $attachments, String $path, array $pdfTkOptions = [])
    {
        $attachmentPaths = [];

        if (!$attachments) {
            return;
        }

        FileVault::disk('leave_form')->decrypt($templatePath . '.enc');

        foreach ($attachments as $attachment) {
            $attachmentPath = $attachment->storeAs($path . '/attachments', $attachment->getClientOriginalName(), 'leave_form');
            $attachmentPaths[] = $storage->path($attachmentPath);
            $rawPaths[] = $attachmentPath;

            LeaveAttachment::create([
                'leave_id' => $leaveId,
                'file_name' => $attachment->getClientOriginalName(),
                'file_path' => $attachmentPath . '.enc'
            ]);
        }

        $pdf = new Pdf($storage->path($templatePath), $pdfTkOptions);

        $result = $pdf->attachFiles($attachmentPaths)
            ->saveAs($storage->path($templatePath));

        if (!$result) {
            return $pdf->getError();
        }

        FileVault::disk('leave_form')->encrypt($templatePath);

        foreach ($rawPaths as $path) {
            FileVault::disk('leave_form')->encrypt($path);
        }

        return $attachmentPaths;
    }

    /**
     * The function retrieves leave allowances for a specific user within a given period.
     *
     * @param int allowanceId The allowanceId parameter is an integer that represents the ID of the
     * leave allowance.
     * @param int userId The userId parameter is the ID of the user for whom you want to retrieve the
     * leave allowance.
     * @param paid The "paid" parameter is used to filter the leave allowances based on whether they
     * are paid or not. If the value of "paid" is 1, it will only return leave allowances that are
     * paid. If the value is 2, it will return leave allowances that are not paid.
     * @param startDate The startDate parameter is the starting date of the period for which you want
     * to retrieve the leave allowances.
     * @param endDate The endDate parameter is the date until which you want to retrieve the leave
     * allowances.
     *
     * @return a collection of LeaveAllowance models that meet the specified conditions.
     */
    public function getLeaveAllowanceByPeriod(int $allowanceId, int $userId, $paid = 2, $startDate, $endDate){
        return LeaveAllowance::where([
            ['allowance_id', $allowanceId],
            ['type', $paid]
        ])->whereHas('leave', function ($query) use($userId, $startDate, $endDate){
            $approved = 1;
            $query->where([
                ['user_id', $userId],
                ['status', $approved]
            ])->whereDate('start_date', '>=', $startDate)->whereDate('end_date', '<=', $endDate);
        })->get();
    }

    public function getLeaveBalance($user_id, $leave_type_id)
    {
        $balance = LeaveBalance::where('user_id', $user_id)
            ->where('leave_type_id', $leave_type_id)
            ->pluck('balance')
            ->first();

        if (is_null($balance)) {
            $balance = 0.0;
        }

        return $balance;
    }
}
