<?php

namespace Suiterus\Adg\Services\LeaveManagement;

use App\Enums\LeaveCreditAdjustmentType;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveBalance;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveCreditEarning;
use Suiterus\Adg\Models\LeaveManagement\LeaveCreditHistory;
use Suiterus\Adg\Models\LeaveManagement\LeaveEligibleGroup;
use Suiterus\Adg\Models\LeaveManagement\LeaveEligibleUser;
use Suiterus\Adg\Models\LeaveManagement\LeaveType;

class LeaveEligibleService
{
    /**
     * The function `getUserEligibleLeaveTypes` retrieves the eligible leave types for a given user
     * based on their roles and individual eligibility.
     *
     * @param User user The parameter "user" is an instance of the User class. It represents a user for
     * whom we want to retrieve the eligible leave types.
     *
     * @return array an array of eligible leave types for a given user.
     */
    public function getUserEligibleLeaveTypes(User $user): array
    {
        $eligibleLeaves = [];
        $eligibleLeavesId = []; //will use for checking if group or user has access to leave

        foreach ($user->roles as $role) {

            $groupLeaves = LeaveEligibleGroup::where('group_id', $role['id'])->with(['leaveType'])->get();

            foreach ($groupLeaves as $groupLeave) {
                if (!in_array($groupLeave->leaveType->id, $eligibleLeavesId)) {
                    $eligibleLeaves[] = $groupLeave->leaveType;
                    $eligibleLeavesId[] = $groupLeave->leaveType->id;
                }
            }
        }

        $userLeaves = LeaveEligibleUser::where('user_id', $user->id)->with(['leaveType'])->get();

        foreach ($userLeaves as $userLeave) {
            if (!in_array($userLeave->leaveType->id, $eligibleLeavesId)) {
                $eligibleLeaves[] = $userLeave->leaveType;
                $eligibleLeavesId[] = $userLeave->leaveType->id;
            }
        }

        return $eligibleLeaves;
    }

    /**
     * The function assigns a leave type to a group of users and individual users.
     *
     * @param groupIds An array of group IDs to which the leave type should be assigned.
     * @param userIds An array of user IDs to whom the leave type should be assigned. Each user ID
     * represents an individual user.
     * @param int leaveTypeId The leaveTypeId parameter is an integer that represents the ID of the
     * leave type that is being assigned to the groups and users.
     */
    public function assignLeaveType(?array $groupIds, ?array $userIds, LeaveType $leaveType)
    {
        $this->assignGroupLeaveType($groupIds, $leaveType);
        $this->assignUserLeaveType($userIds, $leaveType);
    }

    /**
     * The function assigns a leave type to multiple users and creates a leave balance for each user if
     * it doesn't already exist.
     *
     * @param array userIds An array of user IDs to whom the leave type needs to be assigned.
     * @param LeaveType leaveType The  parameter is an instance of the LeaveType class. It
     * represents the type of leave that is being assigned to the users.
     */
    public function assignUserLeaveType(?array $userIds, LeaveType $leaveType)
    {
        foreach ($userIds as $userId) {
            LeaveEligibleUser::create([
                'leave_type_id' => $leaveType->id,
                'user_id' => $userId,
                'created_by' => Auth::id(),
            ]);
            $isLeaveTypeExists = LeaveBalance::where([
                'user_id' => $userId,
                'leave_Type_id' => $leaveType->id
            ])->first();

            if (!$isLeaveTypeExists) {
                LeaveBalance::create([
                    'user_id' => $userId,
                    'leave_type_id' => $leaveType->id,
                    'balance' => $leaveType->initial_points,
                    'created_by' => Auth::id(),
                    'updated_by' => Auth::id()
                ]);
                if ($leaveType->initial_points != null && $leaveType->initial_points > 0) {
                    LeaveCreditHistory::create([
                        'user_id' => $userId,
                        'leave_type_id' => $leaveType->id,
                        'credits' => $leaveType->initial_points,
                        'resulting_balance' => $leaveType->initial_points,
                        'leave_adjustment_type' => LeaveCreditAdjustmentType::EARNED,
                        'remarks' => 'Initial Points',
                        'period' => Carbon::now(),
                        'created_by' => Auth::id(),
                    ]);
                }
            }
        }
    }

    /**
     * The function assigns a leave type to a group of users and their corresponding user IDs.
     *
     * @param array groupIds An array of group IDs to which the leave type should be assigned.
     * @param LeaveType leaveType The  parameter is an instance of the LeaveType class.
     */
    public function assignGroupLeaveType(?array $groupIds, LeaveType $leaveType)
    {
        foreach ($groupIds as $groupId) {
            LeaveEligibleGroup::create([
                'leave_type_id' => $leaveType->id,
                'group_id' => $groupId,
                'created_by' => Auth::id(),
            ]);

            $users = User::whereHas('roles', function ($query) use ($groupId) {
                $query->where('id', $groupId);
            })->get();

            if ($users) {
                $userIds = $users->pluck('id')->toArray();

                $this->assignUserLeaveType($userIds, $leaveType);
            }
        }
    }
}
