<?php

namespace Suiterus\Adg\Services\Attendance;

use Exception;
//Enums
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use App\Enums\LeaveStatus;
use App\Enums\OvertimeStatus;
use App\Enums\OvertimePurpose;
use App\Enums\AttendanceLegend;
use App\Enums\ScheduleBasis;
use Suiterus\Adg\Models\SM\Holiday;
use Suiterus\Adg\Models\SM\Suspension;
use Suiterus\Adg\Services\Time\TimeService;
use Suiterus\Adg\Models\Approvals\UserOvertime;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Models\Timekeeping\EmployeeSchedule;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Shift;
use Suiterus\Adg\Models\Timekeeping\AttendanceLegend as TimekeepingAttendanceLegend;

class AttendanceService
{

    protected $employee;
    public function __construct($employee)
    {
        $this->employee = $employee;
    }


    /**
     * The function `getTimekeepingAttendanceDisplay` retrieves and formats attendance data for a given
     * date range.
     *
     * @param Carbon start_date The start date of the timekeeping attendance display. It is of type
     * Carbon, which is a date and time library in PHP.
     * @param Carbon end_date The end_date parameter is a Carbon object that represents the end date of
     * the timekeeping attendance display.
     * @param array dtr  is an array that stores the attendance data for each day
     * within the specified date range. It is initially empty and gets populated with attendance data
     * as the function recursively calls itself.
     * @param previousTimeOut is used to store the previous time out
     * value for each day in the timekeeping attendance display. It is initially set to `null` and is
     * updated with the time out value from the previous day's attendance record.
     *
     * @return array an array of timekeeping attendance display data.
     */
    public function getTimekeepingAttendanceDisplay(Carbon $start_date, Carbon $end_date, array $dtr = [], $previousTimeOut = null): array
    {

        if (!($start_date <= $end_date)) {
            return $dtr;
        }

        $attendance = Attendance::where('user_id', $this->employee->id)->whereDate('time_in', $start_date)->first();
        $previousTimeOut = Attendance::where('user_id', $this->employee->id)->whereDate('time_out', $start_date)->first();

        $previousTimeOut = $previousTimeOut ? $previousTimeOut->previous_time_out : null;


        /**
         *  * Lines 68–90 show the shift code of the employee, even if the employee is considered absent.
         *  TODO: Creating a shift code using the roster schedule.
         */

        $employee_schedule = $this->getEmployeeSchedulePerDay($this->employee, Carbon::parse($start_date));

        $code = null;
        $days = null;

        if ($employee_schedule instanceof EmployeeSchedule) {
            $code = $employee_schedule->schedule_template->shift_code;
            $days = $employee_schedule->schedule_template->schedule_template;
        }

        $filteredDays = null;

        $remarks = $start_date > Carbon::now() ? str_replace('Absent', '', $this->getRemarks($start_date)) : $this->getRemarks($start_date , $attendance ? $attendance->remarks : '');

        // Shift is for roster
        if (str_contains($this->getRemarks($start_date), 'Absent') && $employee_schedule instanceof Shift) {
            $attendance['shift_code'] = $employee_schedule->code . ' ' . $employee_schedule->formatted_shift_in . ' to ' . $employee_schedule->formatted_shift_out;
            if ($previousTimeOut) {
                $scheduled_in = Carbon::parse($employee_schedule->shift_in);
                $scheduled_out = Carbon::parse($employee_schedule->shift_out);
                $time_out = Carbon::parse($previousTimeOut);
                if ($scheduled_out->lt($scheduled_in))
                {
                    $scheduled_out->addDay();
                }
                
                if ($time_out->gte($scheduled_out) && $time_out->gte($scheduled_in)){
                    $attendance['next_day_out'] = $time_out->format('g:i A');
                    $previousTimeOut = null;
                }
            }
        }

        // EmployeeSchedule class is for timetable
        if (str_contains($this->getRemarks($start_date), 'Absent') && $employee_schedule instanceof EmployeeSchedule) {
            $filteredDays = $days[array_search(date('l', strtotime($start_date)), array_column(json_decode($days), 'day'))];
            $shiftCode = null;
            $filterDay = $filteredDays['timetables'][0];
            $shiftCode .= $code . ' ' . $filterDay['formatted_time_in'] . ' to ' . $filterDay['formatted_time_out'];
            $attendance['shift_code'] = $shiftCode;

            if ($previousTimeOut) {
                $scheduled_in = Carbon::parse($filterDay['formatted_time_in']);
                $scheduled_out = Carbon::parse($filterDay['formatted_time_out']);
                $time_out = Carbon::parse($previousTimeOut);
                if ($scheduled_out->lt($scheduled_in))
                {
                    $scheduled_out->addDay();
                }
                
                if ($time_out->gte($scheduled_out) && $time_out->gte($scheduled_in)){
                    $attendance['next_day_out'] = $time_out->format('g:i A');
                    $previousTimeOut = null;
                }
            }
        }

        $attendance['remarks'] = $remarks;

        $dtr[] = [
            'data' => $attendance,
            'previous_time_out' => $previousTimeOut ? date("g:i A", strtotime($previousTimeOut)) : null,
            'date' => Carbon::parse($start_date),
            'time_date' => $start_date->format('d'),
            'time_day' => $start_date->format('l'),
        ];

        $start_date->addDay();

        return $this->getTimekeepingAttendanceDisplay($start_date, $end_date, $dtr, $previousTimeOut, $employee_schedule);
    }

    public function getEmployeeSchedulePerDay($user, $currentdate)
    {
        $shift = Shift::whereHas('employeeShift',function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user){
                $query->where('user_id',$user->id);
            });
        })
        ->orWhereHas('headEmployeeShift', function($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id);
            });
        })->first();

        if ($shift) {
            return $shift;
        }

        $timetableEmployeeSchedule = EmployeeSchedule::where('user_id', $this->employee->id)->with('schedule_template.schedule_template')->first();

        if ($timetableEmployeeSchedule) {
            return $timetableEmployeeSchedule;
        }

        return null;
    }

    //UTILITIES
    public function determineAbsence($date)
    {
        return [
            'isHoliday' => $this->getHoliday($date)  ? true : false,
            'isOnLeave' => $this->getLeave($date) ? true : false,
            'isRestDay' => $this->isRestDay($date)  ? true : false
        ];
    }

    /**
     * The function "getHoliday" retrieves the purpose of a holiday based on a given date.
     * @return The purpose of the holiday is being returned.
     */
    public function getHoliday($date)
    {
        $holiday = Holiday::whereDate('date', Carbon::parse($date)->format('Y-m-d'))->where('status', Status::ACTIVE)->orderBy('purpose', 'asc')->first();

        return $holiday ? $holiday : null;
    }

    /**
     * The function retrieves a leave record for a specific date and employee if it exists and has been
     * approved.
     * @return the leave object if it exists and is approved for the given date, otherwise it returns null.
     */
    public function getLeave($date)
    {

        $leave = Leave::where('user_id', $this->employee->id)->whereDate('start_date', '<=', Carbon::parse($date)->format('Y-m-d'))
            ->whereDate('end_date', '>=', Carbon::parse($date)->format('Y-m-d'))
            ->where('status', LeaveStatus::APPROVED)->first();
        return $leave ? $leave : null;
    }

    /**
     * The function "getRemarks" determines the remarks for a given date based on whether it is a
     * holiday, leave, rest day, or absence.
     *
     * @param date The parameter "date" is the date for which you want to determine the remarks.
     *
     * @return the remarks for a given date. The possible remarks are "Holiday", "Leave", "Day-Off", or
     * "Absent".
     */
    public function getRemarks($date, $remarks = "")
    {
        $attendance = Attendance::where('user_id', $this->employee->id)->whereDate('time_in', $date)->first();
        $nextAttendance = Attendance::where('user_id', $this->employee->id)->whereDate('time_in', Carbon::parse($date)->addDay())->first();
        $absent_remark = $this->determineAbsence($date);

        $addSlash = $remarks == '' ? '' : '/';

        if ($absent_remark["isHoliday"]) {
            $holiday = $this->getHoliday($date);
            $remarks .= $addSlash . $holiday->title;
        }

        if ($absent_remark["isOnLeave"]) {
            $leave = $this->getLeave($date);
            $addSlash = $remarks == '' ? '' : '/';
            $remarks .= $addSlash . $leave->leaveType->remark . ' ' . $leave->credit_deductions->credits;
        }

        if ($absent_remark["isRestDay"]) {
            $addSlash = $remarks == '' ? '' : '/';
            $remarks .= $addSlash . "Day-Off";
        }

        if ($attendance) {
            if ($this->isSuspended($attendance)) {
                $addSlash = $remarks == '' ? '' : '/';
                $remarks .= $addSlash . "Suspended";
            }

            if (strtotime(date('H:i', strtotime($attendance->time_in))) > strtotime(date('H:i', strtotime($attendance->scheduled_max_time_in))) && !$absent_remark['isRestDay'] && $attendance->time_out){
                $addSlash = $remarks == '' ? '' : '/';
                $remarks .= $addSlash . "Late";
            } elseif ($attendance->break_out && strtotime(date('H:i', strtotime($attendance->break_out))) > strtotime(date('H:i', strtotime($attendance->scheduled_break_out))) && !$absent_remark['isRestDay'] && $attendance->time_out) {
                $addSlash = $remarks == '' ? '' : '/';
                $remarks .= $addSlash . "Late";
            }

            if ($attendance->time_out && strtotime(date('H:i', strtotime($attendance->time_out))) < strtotime($attendance->scheduled_time_out) && !$absent_remark['isRestDay']) {
                $addSlash = $remarks == '' ? '' : '/';
                $remarks .= $addSlash . "UT";
            } elseif ($attendance->break_in && strtotime(date('H:i', strtotime($attendance->break_in))) < strtotime(date('H:i', strtotime($attendance->scheduled_break_in))) && !$absent_remark['isRestDay'] && $attendance->time_out) {
                $addSlash = $remarks == '' ? '' : '/';
                $remarks .= $addSlash . "UT";
            }

            if ($date == $absent_remark["isRestDay"] || (!$attendance->timetable_basis && !$attendance->shift)) {
                $remarks = "OT";
            }

            if (!$attendance->time_out && $nextAttendance) {
                $addSlash = $remarks == '' ? '' : '/';
                $remarks .= $addSlash . "Absent";
            }
        }

        if (!$absent_remark["isHoliday"] && !$absent_remark["isOnLeave"] && !$absent_remark["isRestDay"] && !$attendance) {
            $addSlash = $remarks == '' ? '' : '/';
            $remarks .= $addSlash . "Absent";
        }

        return $remarks;
    }

    //Returns if attendance is suspended or not | true or false.
    public function isSuspended($attendance)
    {
        $suspension = Suspension::whereDate('date', Carbon::parse($attendance->time_in)->format('Y-m-d'))->where('status', Status::ACTIVE)->first();

        if (!$suspension) {
            return false;
        }
        return ($suspension->time_effectivity >= Carbon::parse($attendance->time_in)->format('H:i:s')) || ($suspension->time_effectivity <= Carbon::parse($attendance->time_in)->format('H:i:s'));
    }

    /**
     * The function checks if a given day is a rest day for a specific employee.
     *
     * @param string The "day" parameter is a string representing a specific date.
     *
     * @return boolean value. It returns true if the given day is not found in the employee's
     * schedule, indicating that it is a rest day. It returns false if the given day is found in the
     * employee's schedule, indicating that it is not a rest day.
     */
    public function isRestDay($day)
    {
        $user = User::where('id', $this->employee->id)->with(['employeeSchedules' => function ($query) {
            $query->with(['schedule_template' => function ($query) {
                $query->with('schedule_template');
            }]);
        }, 'temporary_schedule' => function ($query) {
            $query->with('schedule_template');
        }])
            ->whereHas('employeeMetaInfo')
            ->first();

        $date = Carbon::parse($day)->format('Y-m-d');
        $day = date("l", strtotime($day));

        $roster_schedule = $this->getRosterShift($date);

        if ($roster_schedule) {
            return false;
        }

        if (!$user->employeeSchedules->isEmpty()) {
            $schedules = $user->employeeSchedules[0]['schedule_template']['schedule_template'];
            $days = [];
            foreach ($schedules as $scheduled) {
                $days[] = $scheduled['day'];
            }
            return !in_array($day, $days);
        }

        return true;
    }

    public function getActiveRosterShift($user, $currentdate) {
        return Shift::whereHas('employeeShift', function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date', $currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user) {
                $query->where('user_id', $user->id)->whereHas('employeeGroup.roster', function ($query) {
                    $query->where('status', Status::ACTIVE);
                });
            });
        })->orWhereHas('headEmployeeShift', function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date', $currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id)
                    ->where('status', Status::ACTIVE);
            });
        })->first();
    }

    /**
     * The function calculates the total Leave Without Pay (LWOP) by summing the total undertime and
     * total late minutes from a given set of attendances.
     *
     * @param attendances The parameter "attendances" is likely an array that contains information
     * about the attendance records of an employee. It could include details such as the date, time in,
     * time out, and any other relevant information for each attendance record.
     *
     * @return An array is being returned with the following keys and values:
     * - 'total_undertime': the total undertime calculated from the total number of undertime minutes calculated
     *  from the given attendance records.
     * - 'total_late_minutes': the total late minutes calculated from thetotal number of late minutes
     * calculated from the given attendance records.
     * - 'total_lwop': the sum of the total undertime and total late minutes, representing the total
     * Leave Without Pay (LWOP)
     */
    public function getTotalLWOP($attendances)
    {
        return [
            'total_undertime' => $this->totalUndertime($attendances),
            'total_late_minutes' => $this->totalLates($attendances),
        ];
    }



    /**
     * The function `getPaidOvertime` calculates the total amount of paid overtime for a given employee
     * within a specified date range.
     * @return the total amount of paid overtime in minutes for a given employee within a specified date range.
     */
    public function getPaidOvertime($start_date, $end_date)
    {
        $overtimes = UserOvertime::where('user_id', $this->employee->id)
            ->whereDate('start_date', '>=', $start_date)
            ->whereDate('end_date', '<=', $end_date)
            ->where('purpose', OvertimePurpose::PAY)
            ->where('status', OvertimeStatus::APPROVED)
            ->get();
        $time = 0;
        foreach ($overtimes as $overtime) {
            $start_time = strtotime($overtime->start_time);
            $end_time = strtotime($overtime->end_time);
            $time += ($end_time - $start_time) / 60;
        }

        return $time;
    }

    /**
     * The function computes the total overtime from a given array of attendance records.
     *
     * @param attendances An array of attendance records. Each attendance record should have a
     * 'over_time' field that represents the overtime hours for that attendance.
     *
     * @return the result of the TimeService::calculateTotalTime() method, which is the total overtime
     * calculated from the given attendances.
     */
    public function totalOvertime($attendances)
    {
        $overtime = $this->pluckArrayableTime($attendances, 'over_time');
        return TimeService::calculateTotalTime($overtime);
    }

    /**
     * The function computes the total undertime based on the given attendance records.
     *
     * @param attendances An array of attendance records. Each attendance record should have a
     * 'under_time' field, which represents the amount of time the employee was under the required
     * working hours for that day.
     *
     * @return the total undertime calculated from the given attendances.
     */
    public function totalUndertime($attendances)
    {
        $undertime = $this->pluckArrayableTime($attendances, 'under_time');
        return TimeService::calculateTotalTime($undertime);
    }

    /**
     * The function computes the total number of late hours from a given array of attendance records.
     *
     * @param attendances parameter is expected to be an array of attendance
     * records. Each attendance record should have a `late_hours` property, which represents the number
     * of hours a person was late for a particular event or shift.
     *
     * @return the result of the TimeService::calculateTotalTime() method, which is the total time of
     * late hours calculated from the given attendances.
     */
    public function totalLates($attendances)
    {
        $lates = $this->pluckArrayableTime($attendances, 'late_hours');
        return TimeService::calculateTotalTime($lates);
    }

    /**
     * The function computes the total night differential time for a given set of attendances.
     *
     * @param attendances parameter is an array that contains attendance records.
     * Each attendance record should have a `night_differential` property, which represents the night
     * differential hours for that attendance.
     *
     * @return the total time of night differential calculated from the given attendances.
     */
    public function totalNightDifferential($attendances)
    {
        $nightdiff = $this->pluckArrayableTime($attendances, 'night_differential');
        return TimeService::calculateTotalTime($nightdiff);
    }

    /**
     * The function computes the total night differential overtime for a given set of attendances.
     *
     * @param attendances The parameter "attendances" is an array that contains the attendance records
     * of employees. Each attendance record should have a "night_differential_over_time" field, which
     * represents the amount of overtime worked during night hours.
     *
     * @return the total time of night differential overtime calculated from the given array of
     * attendances.
     */
    public function totalNightDifferentialOvertime($attendances)
    {
        $nightdiffovertime = $this->pluckArrayableTime($attendances, 'night_differential_over_time');
        return TimeService::calculateTotalTime($nightdiffovertime);
    }

    public function pluckArrayableTime($attendances, String $pluckable)
    {
        return collect(is_array($attendances) || $attendances instanceof Collection ? $attendances : [$attendances])->pluck($pluckable)->toArray();
    }

    /**
     * The function `getTimeInCount` returns the count of non-null `time_in` values in the given
     * `attendances` collection.
     *
     * @param attendances The parameter "attendances" is expected to be a collection or array of
     * attendance records. Each attendance record should have a "time_in" field that represents the
     * time when the attendance was recorded.
     *
     * @return the count of non-null values in the 'time_in' column of the  collection.
     */
    public function getTimeInCount($attendances)
    {
        $timeIn = $attendances->whereNotNull('time_in');
        return $timeIn->count();
    }

    /**
     * The function "computeTotalMinutes" calculates the total number of minutes from a given array of
     * attendance records.
     *
     * @param attendances The parameter "attendances" is expected to be an array or collection of
     * attendance records. Each attendance record should have a "time_in" field that represents the
     * time when the attendance was recorded.
     *
     * @return the total number of minutes calculated from the given attendances.
     */
    public function computeTotalMinutes($attendances)
    {
        $totalMinutes = 0;
        $totals = [];
        $timeIn = $attendances->whereNotNull('time_out');
        foreach ($timeIn as $time) {
            $timetotal = $time['total'];
            array_push($totals, $timetotal);
        }

        foreach ($totals as $times) {
            list($hours, $minutes) = explode(':', $times);
            $totalMinutes += ($hours * 60) + $minutes;
        }
        return $totalMinutes;
    }

    /**
     * The computeSumHours function takes a total number of minutes and returns the equivalent time in
     * hours and minutes format.
     *
     * @param totalMinutes The total number of minutes that you want to convert into hours and minutes.
     *
     * @return a formatted string representing the total hours and minutes.
     */
    public function computeSumHours($totalMinutes)
    {
        return sprintf('%02d:%02d', floor($totalMinutes / 60), $totalMinutes % 60);
    }

    /**
     * The function `computeFrequencyUndertime` calculates the frequency of under time occurrences in a
     * given array of attendances.
     *
     * @param attendances An array of attendance records. Each attendance record is an associative
     * array with the following keys: undertime
     *
     * @return the frequency of undertime occurrences in the given array of attendances.
     */
    public function totalUnderTimeFrequency($attendances)
    {
        $undertimeFrequency = 0;
        $undertimes = [];
        foreach ($attendances as $attendance) {

            $frqUnder = $attendance['under_time'] ?? null;
            array_push($undertimes, $frqUnder);
        }

        $undertimeFrequency = count(array_filter($undertimes, function ($value) {
            return $value !== null && $value !== '';
        }));
        return $undertimeFrequency;
    }

    /**
     * The function computes the frequency of late hours in a given array of attendances.
     *
     * @param attendances An array of attendance records. Each record should have a 'late_hours' key
     * that represents the number of hours a person was late.
     *
     * @return the frequency of late attendances.
     */
    public function totalLateFrequency($attendances)
    {
        $lateFrequency = 0;
        $lates = [];
        foreach ($attendances as $late) {
            $frqLate = $late['late_hours'] ?? null;
            array_push($lates, $frqLate);
        }
        $lateFrequency = count(array_filter($lates, function ($value) {
            return $value !== null && $value !== '';
        }));
        return $lateFrequency;
    }

    /**
     * The function retrieves the latest attendance records for employees who were late between a
     * specified start and end date.
     *
     * @param start_date The start date is the date from which you want to start retrieving the
     * attendance records.
     * @param end_date The end_date parameter is the date until which you want to fetch the latest
     * attendance records.
     *
     * @return a collection of employee attendance records that fall within the specified start and end
     * dates and have a legend of "LATE".
     */
    public function getLates($start_date, $end_date)
    {
        return $this->employee->attendance()
        ->where(function ($query) use ($start_date, $end_date) {
            $query->whereDate('time_in', '>=', $start_date)
                ->whereDate('time_in', '<=', $end_date);
        })
        ->whereHas('legends', function ($query) {
            $query->where('legend', AttendanceLegend::LATE);
        })->get();
    }

    /**
     * The function "getAbsentValues" takes an array of attendance records and returns an array of
     * dates and minutes for which the attendance record has the remark "Absent".
     *
     * @param dtr The parameter `dtr` is an array that contains attendance records. Each attendance
     * record is an associative array with the following keys: data and remarks
     *
     * @return an array of absences. Each absence is represented by an associative array with two keys:
     * 'date' and 'minutes'.
     */
    public function getAbsentValues($dtr)
    {
        $absences = [];
        foreach ($dtr as $attendance) {
            if ($attendance['data']['remarks'] == 'Absent') {
                $schedule = $this->getEmployeeSchedulePerDay($this->employee, $attendance['date']);

                $absentMinutes = 0;

                if ($schedule instanceof Shift) {
                    $breakMinutes = abs(Carbon::parse($schedule->break_in)->diffInMinutes($schedule->break_out));
                    $absentMinutes = abs(Carbon::parse($schedule->shift_in)->diffInMinutes($schedule->shift_out)) - $breakMinutes;
                }elseif ($schedule instanceof EmployeeSchedule) {
                    $day = $schedule->schedule_template->schedule_template[array_search(date('l', strtotime($attendance['date'])), array_column(json_decode($schedule->schedule_template->schedule_template), 'day'))];
                    $breakMinutes = abs(Carbon::parse($day->timetables[0]->break_in)->diffInMinutes($day->timetables[0]->break_out));
                    $absentMinutes = abs(Carbon::parse($day->timetables[0]->time_in)->diffInMinutes($day->timetables[0]->time_out) - $breakMinutes);
                }

                array_push($absences, [
                    'date'      => date('Y-m-d', strtotime($attendance['date'])),
                    'minutes'   => $absentMinutes,
                ]);
            }
        }
        return $absences;
    }

    /**
     * The function `getUndertimes` retrieves a collection of undertime attendances for a specific
     * date range, filtering out any attendances that have suspensions on the same date.
     *
     * @param start_date The start date is the date from which you want to start checking for
     * undertimes. It is the lower limit of the date range for which you want to retrieve undertimes.
     * @param end_date The end_date parameter is the date until which you want to fetch the
     * undertimes. It is the upper limit of the dte range for which you want to retrieve undertimes.
     *
     * @return a collection of undertimes.
     */
    public function getUndertimes($start_date, $end_date)
    {
        $suspensions = Suspension::where('status', Status::ACTIVE)
        ->where(function ($query) use ($start_date, $end_date) {
            $query->whereDate('date', '>=', $start_date)
                ->whereDate('date', '<=', $end_date);
        })->get();

        $undertimes = collect();

        $attendances = $this->employee->attendance()
            ->where(function ($query) use ($start_date, $end_date) {
                $query->whereDate('time_in', '>=', $start_date)
                    ->whereDate('time_in', '<=', $end_date);
            })
            ->whereHas('legends', function ($query) {
                $query->where('legend', AttendanceLegend::UNDERTIME);
            })->get();

        //unfiltered undertimes contain attendances that are unaffected by post-made suspensions, loop to filter and remove attendances that has suspension on that date
        foreach ($attendances as $attendance) {
            if (!$suspensions->contains('date', $attendance->date_in)) {
                $undertimes->push($attendance);
            }
        }

        return $undertimes;

    }

    public function reEvaluateAttendance($date, $attendance = null)
    {
        if (!$attendance) {
            $attendance = Attendance::where('user_id', $this->employee->id)
            ->whereDate('time_in', $date)->first();
        }

        if (!$attendance) {
            return 0;
        }

        $legends = [];
        $sched_timetable = null;
        $schedule = null;
        $break_in = null;
        $break_out = null;
        $time_in = date('Y-m-d H:i:s', strtotime($attendance->time_in));
        $time_out = $attendance->time_out ? date('Y-m-d H:i:s', strtotime($attendance->time_out)) : null;
        $total = $time_out ? TimeService::calculateTotalDateTime($time_in, $time_out) : '0:00';

        $roster_shift = $this->getRosterShift(Carbon::parse($time_in)->format('Y-m-d'));

        if ($roster_shift) {
            //roster
            $break_in = $roster_shift->break_in != null ? $roster_shift->break_in : null;
            $break_out = $roster_shift->break_out != null ? $roster_shift->break_out : null;

            if (strtotime(date('H:i:s', strtotime($attendance->time_in))) > strtotime($roster_shift->shift_in)) {
                array_push($legends, AttendanceLegend::LATE);
            }

            // Check undertime
            if (strtotime(date('H:i:s', strtotime($attendance->time_out))) < strtotime($roster_shift->shift_out)) {
                array_push($legends, AttendanceLegend::UNDERTIME);
            }
        } else {
            //timetable
            $employee_schedule = $this->getActiveSchedule();
            if ($employee_schedule) {
                if ($employee_schedule->basis === ScheduleBasis::SCHEDULE) {
                    // Get the schedule template day and the timetables associated with the day
                    $day_of_week = date('l', strtotime($time_in));

                    /**
                     * Getting the timetable for the current time in recorded
                     * To get the timetable, check if the time-in time is greater than the time in of the timetables
                     * if there are two timetables, there will be 2 results. Only the first result is fetched and
                     * compared which is the latest (if morning and afternoon and the time in is greater than the time in of both, get the afternoon),
                     * since if the time is greater than the morning only, then it automatically means that the schedule used will not be the afternoon session
                     */
                    $schedule = $employee_schedule->schedule_template()->where('day', $day_of_week)->first();
                    $sched_timetable = $schedule !== null ? ($schedule->timetables()->whereTime('time_out', '!=', date('H:i:s', strtotime($time_in)))->orderBy('time_in', 'asc')->first()) : null;

                    // If timetable is found
                    if($sched_timetable !== null) {
                        //set time for break_in & out
                        $break_in = $sched_timetable->break_in !== null ? $sched_timetable->break_in : null;
                        $break_out = $sched_timetable->break_out !== null ? $sched_timetable->break_out : null;
                        // Check Late
                        if(strtotime(date('H:i:s', strtotime($attendance->time_in))) > strtotime($sched_timetable->max_time)) {
                            array_push($legends, AttendanceLegend::LATE);
                        }

                        // Check undertime
                        if(strtotime(date('H:i:s', strtotime($attendance->time_out))) < strtotime($sched_timetable->time_out)) {
                            array_push($legends, AttendanceLegend::UNDERTIME);
                        }
                    }

                    $schedule = $schedule !== null ? $schedule->id : null;
                    $sched_timetable = $sched_timetable !== null ? $sched_timetable->id : null;
                }
            }
        }

        if ($attendance->break_in && $attendance->break_out) {
            $break_in = $attendance->break_in;
            $break_out = $attendance->break_out;
        }

        if($break_in && $break_out && $time_out) {
            $hours_diff = Carbon::parse($break_in)->diffInSeconds(Carbon::parse($break_out));
            $hours_diff = Carbon::parse(gmdate('H:i', $hours_diff));
            $total = Carbon::parse($total);
            $total->subHours($hours_diff->format('H'));
            $total->subMinutes($hours_diff->format('i'));
            $total = $total->format('G:i');
        }

        $attendance->break_in = $attendance->break_in ? $break_in : null;
        $attendance->break_out = $attendance->break_out ? $break_out : null;
        $attendance->total = $total;
        $attendance->timetable_id = $sched_timetable;
        $attendance->schedule_day_id = $schedule;
        $attendance->shift_id = $roster_shift ? $roster_shift->id : null;
        $attendance->updated_by = Auth::id();
        $attendance->save();

        TimekeepingAttendanceLegend::where('attendance_id', $attendance->id)->delete();

        if (($attendance->break_in && $attendance->break_out) && ($attendance->scheduled_break_in && $attendance->scheduled_break_out) ) {
            if (strtotime(date('H:i:s', strtotime($attendance->break_in))) < strtotime($attendance->scheduled_break_in)) {
                array_push($legends, AttendanceLegend::UNDERTIME);
            }
            if (strtotime(date('H:i:s', strtotime($attendance->break_out))) > strtotime($attendance->scheduled_break_out)) {
                array_push($legends, AttendanceLegend::LATE);
            }
        }

        foreach($legends as $legend) {
            $attendance->legends()->create([
                'legend' => $legend,
            ]);
        }

    }
    public function getRosterShift($currentdate) {
        $user = $this->employee;
        return Shift::whereHas('employeeShift',function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user){
                $query->where('user_id',$user->id);
            });
        })
        ->orWhereHas('headEmployeeShift', function($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id);
            });
        })->first();
    }

    public function getActiveSchedule() {
        $user = $this->employee;
        $employee_schedule = $user->employeeSchedules()->without('user')->first();
        return $employee_schedule === null ? null : $employee_schedule->schedule_template;
    }
}
