<?php

namespace Suiterus\Adg\Reports;

use App\Enums\Payroll\EmployeePayrollVoidStatus;
use App\Enums\Payroll\PayrollStatus;
use App\Models\User;
use Carbon\Carbon;
use Suiterus\Adg\Abstracts\FileReport;
use Suiterus\Adg\Exceptions\Reports\ReportException;
use Suiterus\Adg\Models\Payroll\Payroll;
use Suiterus\Adg\Models\Payroll\PayrollEarningDeduction;
use Suiterus\Adg\Models\Reports\Reports;
use Suiterus\Adg\Services\ReportService;
use Suiterus\Adg\Traits\EmployeeTrait;
use Suiterus\Adg\Traits\EmployerTrait;

class WithholdingTaxReport extends FileReport
{
    use EmployeeTrait;
    use EmployerTrait;

    /**
     * Generate the withholding tax report in Excel format.
     * 
     * This function generates a withholding tax report for employees who are not in user IDs [1, 2].
     * It fetches the necessary data from the database and exports it to an Excel file.
     *
     * @return \Suiterus\Adg\Abstracts\FileReport\exportToCSV The Excel file response.
     */
    public function generate($reportId)
    {

        $grand_total = 0;
        $all_users = User::whereNotIn('id', [1, 2])->with('employeeExtraField', 'salary.publicSalary.salaryGrade', 'salary.privateSalary')->without('salary.user')->get();
        $report = Reports::whereId($reportId)->with('creator')->first();
        $report_service = new ReportService;

        $path = $report_service->getFileTemplate($report->report_type);
        $this->initializeEmployerInfo($report->creator);
        $employer_name = $this->employers_name;


        $payrollPeriod = Payroll::where('status', PayrollStatus::DISBURSED)
            ->whereDate('period_start', '>=', $report->start_date)
            ->whereDate('period_end', '<=', $report->end_date)->first();

        if (!$payrollPeriod) {
            throw new ReportException('This payroll period has not yet generated');
        }

        foreach ($all_users as $key => $user) {
            $tin = $user->employeeExtraField->pluck('employeeExtraFieldColumn')->flatten()->where('field_name', 'tin_id')->pluck('field_value')->first() ?? '';
            if($tin){ 
                if($tin == 'N/A'){
                    $tin = '';
                }else{
                    $tin = $tin . "000";
                }
            }

            $tax = $this->getWithholdingTax($user->id, $report->start_date, $report->end_date);
            $this->initializeEmployeeInfo($user);
            $indexes[] = $key + 1;
            $employees_number[] = $this->employee_id;
            $employees_fullname[] = strtoupper($this->getFullName());
            $tin_ids[] = $tin;
            $voucher_numbers[] = "";
            $total_amounts[] = number_format($tax, 2);
            $grand_total += $tax;
        }


        // Calling the 'exportToCSV' method and passing an associative array of data using 1D array
        // that will be used to populate placeholders in the CSV file per Column.
        return $this->exportToCSV([
            '{employer_name}' => $employer_name,
            '{month}' => strtoupper(Carbon::parse($report->end_date)->format('F')),
            '{year}' => strtoupper(Carbon::parse($report->end_date)->format('Y')),
            '[number]' => $indexes,
            '[employee_number]' => $employees_number,
            '[employee_fullname]' => $employees_fullname,
            '[tin_id]' => $tin_ids,
            '[voucher_number]' => $voucher_numbers,
            '[total_amount]' => $total_amounts,
            '{grand_total}' => $grand_total,
        ],  Carbon::now()->format('F-Y') . '-Withholding-Tax-Report-' . $report->id . '.xlsx', $path);
    }

    private function getWithholdingTax($user_id, $start_date, $end_date)
    {
        $amount = 0;

        $withholding_tax = PayrollEarningDeduction::whereHas('payroll_employee', function ($query) use ($user_id) {
            $query->where('user_id', $user_id)->where('void_status', EmployeePayrollVoidStatus::NOT_VOIDED);
        })->whereHas('payroll_employee.payroll', function ($query) use ($start_date, $end_date) {
            $query->where('status', PayrollStatus::DISBURSED)
                ->whereDate('period_start', '>=', $start_date)
                ->whereDate('period_end', '<=', $end_date);
        })->where('name', 'Withholding Tax')->first();

        if ($withholding_tax) {
            $amount = (float) $withholding_tax->amount;
        }

        return $amount;
    }
}
