<?php

namespace Suiterus\Adg\Reports;

use Carbon\Carbon;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Style\Border;
use Maatwebsite\Excel\Concerns\WithHeadings;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class TimeToHireRateReport implements FromArray, WithHeadings, ShouldAutoSize, WithStyles, WithColumnWidths, WithEvents
{
    use Exportable;

    protected $data;

    public function __construct($data)
    {
       $this->data = $data;
    }

    public function array(): array
    {
        if (empty($this->data)) {
            return [];
        }

        return $this->data;
    }

    public function headings(): array
    {
        return [
            'No.',
            'Job Post ID',
            'No. of Job Post',
            'Vacant Position',
            'Item Code',
            'Date of Job Opened',
            'Date of Contract Signed',
            'Time to Hire the Individual'
        ];
    }

    public function styles(Worksheet $sheet)
    {   
        $styleArray = array(
            'alignment' => [
                'vertical' => Alignment::VERTICAL_CENTER,
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'wrapText' => true
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => array('argb' => '00000000'),
                ],
            ],
        );

        $headerStyle = [
            'alignment' => [
                'vertical' => Alignment::VERTICAL_CENTER,
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'wrapText' => true
            ],
            'font' => [
                'bold' => true,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => array('argb' => '00000000'),
                ],
            ],
        ];

        $sheet->getStyle('A1:H1')->applyFromArray($headerStyle);
        $lastRow = count($this->data) + 1;
        $sheet->getStyle('A2:H' . $lastRow)->applyFromArray($styleArray);
    }

    public function columnWidths(): array
    {
        return [
            'A' => 10,
            'B' => 15,
            'C' => 30,
            'D' => 40,
            'E' => 20,
            'F' => 30,
            'G' => 30,
            'H' => 20,
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $highestRow = $event->sheet->getHighestRow();
                
                $totalTimeToHire = 0;
                $totalSlots = 0;
    
                for ($row = 2; $row <= $highestRow; $row++) {
                    $timeToHireCell = $event->sheet->getCell('H' . $row);
                    $timeToHireValue = $timeToHireCell->getValue();
    
                    preg_match('/(\d+)/', $timeToHireValue, $matches);
                    $timeToHire = isset($matches[1]) ? (int)$matches[1] : 0;
    
                    $slotsCell = $event->sheet->getCell('C' . $row);
                    $slots = $slotsCell->getValue();
    
                    $totalTimeToHire += $timeToHire;
                    $totalSlots += $slots;
                }

                $averageTimeToHire = $totalSlots > 0 ? $totalTimeToHire / $totalSlots : 0;
    
                $event->sheet->setCellValue('H' . ($highestRow + 1), $averageTimeToHire);
    
                $styleArray = [
                    'font' => ['bold' => true],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
                    'numberFormat' => ['formatCode' => NumberFormat::FORMAT_NUMBER_0],
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_THIN,
                            'color' => ['argb' => '00000000'],
                        ],
                    ],
                ];
    
                $event->sheet->getStyle('H' . ($highestRow + 1))->applyFromArray($styleArray);
            },
        ];
    }

}