<?php

namespace Suiterus\Adg\Reports;

use Carbon\Carbon;
use App\Models\User;
use App\Enums\ReportType;
use App\Enums\Payroll\PayrollType;
use Suiterus\Adg\Abstracts\FileReport;
use Suiterus\Adg\Traits\EmployeeTrait;
use Suiterus\Adg\Traits\EmployerTrait;
use Suiterus\Adg\Models\Reports\Reports;
use Suiterus\Adg\Services\ReportService;
use Suiterus\Adg\Models\Payroll\PayrollEmployee;
use Suiterus\Hrjp\Models\Applicant_account as AC;
use Suiterus\Adg\Models\Payroll\PayrollEarningDeduction;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class ThirteenthMonthPayReport extends FileReport
{
    use EmployeeTrait;
    use EmployerTrait;

    public function generate($reportId)
    {
        $report = Reports::whereId($reportId)->with('creator')->first();
        $reportService = new ReportService;

        $payrollEmployees = PayrollEmployee::where('user_id', $report->entity_id)
            ->whereHas('payroll', function ($query) use ($report) {
                $year = Carbon::parse($report->start_date)->format('Y');
                $query->whereYear('period_start', $year)
                ->where('status', 2); // 2 = Disbursed
            })->with('user')->get();

        $payrollEmployees = $payrollEmployees->sortBy(function ($payrollEmployee) {
            return optional($payrollEmployee->payroll)->period_start;
        })->values();

        $path = $reportService->getFileTemplate($report->report_type);
        $this->initializeEmployerInfo($report->creator);
        $employerName = $this->employers_name;

        $indexes = [];
        $salaries = [];
        $months = [];
        $total = 0;
        $processedMonths = [];

        foreach ($payrollEmployees as $payrollEmployee) {
            $this->initializeEmployeeInfo($payrollEmployee->user);
        
            $basicSalary = $this->calculateSalaryWithDeduction($payrollEmployee->payroll, $payrollEmployee->stored_basic_salary, $payrollEmployee->id);
            $periodStart = Carbon::parse($payrollEmployee->payroll->period_start);
            $periodEnd = Carbon::parse($payrollEmployee->payroll->period_end);
        
            $this->getMonthsBasedOnPayrollType(
                $payrollEmployee->payroll->payroll_type,
                $periodStart,
                $periodEnd,
                $payrollEmployee,
                $indexes,
                $salaries,
                $months,
                $processedMonths,
                $basicSalary
            );

            $total += $basicSalary;
        }

        $thirteenMonth = $total / 12;

        $user = User::whereId($report->entity_id)->with('employeeMetaInfo')->first();
        
        return $this->exportToCSV([
            '{name_of_agency}' => $employerName,
            '{year}' => strtoupper(Carbon::parse($report->end_date)->format('Y')),
            '[number]' => $indexes,
            '{employee_fullname}' => $user->name,
            '{employee_number}' => $user->employeeMetaInfo->employee_id,
            '[salary]' => $salaries,
            '[month]' => $months,
            '{total}' => 'PHP ' . number_format($thirteenMonth, 2),
        ], Carbon::now()->format('F-Y') . '-13th-Month-Pay-Report-' . $report->id . '.xlsx', $path);
    }

    private function calculateSalaryWithDeduction($payroll, $salary, $payrollEmployeeId)
    {
        $totalDeductions = PayrollEarningDeduction::where([
            ['payroll_employee_id', $payrollEmployeeId], ['type', 2], ['name', 'Basic Pay']
        ])->sum('amount');
    
        $periodStart = Carbon::parse($payroll->period_start);
        $periodEnd = Carbon::parse($payroll->period_end);
    
        if ($payroll->payroll_type == PayrollType::MONTHLY && $periodStart->day == 01 && ($periodEnd->day >= 28)) {
            return $salary - $totalDeductions;
        }
    
        if ($payroll->payroll_type == PayrollType::SEMIMONTHLY) {
            $isFirstHalf = $periodStart->day == 01 && $periodEnd->day == 15;
            $isSecondHalf = $periodStart->day == 16 && ($periodEnd->day >= 28);
    
            if ($isFirstHalf || $isSecondHalf) {
                return $salary * 2 - $totalDeductions;
            }
        }
    
        return $salary - $totalDeductions;
    }

    private function getMonthsBasedOnPayrollType($payrollType, $periodStart, $periodEnd, $payrollEmployee, &$indexes, &$salaries, &$months, &$processedMonths, $basicSalary)
    {
        $currentMonthKey = $periodStart->format('Y-m-d'); 
    
        if (in_array($currentMonthKey, $processedMonths)) { 
            return;
        }
    
        if ($payrollType == PayrollType::MONTHLY && $periodStart->day == 01 && ($periodEnd->day >= 28)) {
            $processedMonths[] = $currentMonthKey;
            return;
        }
    
        if ($payrollType == PayrollType::SEMIMONTHLY) {
            $isFirstHalf = $periodStart->day == 01 && $periodEnd->day == 15;
            $isSecondHalf = $periodStart->day == 16 && ($periodEnd->day >= 28);
    
            if ($isFirstHalf || $isSecondHalf) {
                $monthName = Carbon::parse($payrollEmployee->payroll->period_start)->format('F');
                $index = array_search($monthName, $months);
    
                if ($index !== false) {
                    $salaries[$index] += $basicSalary;
                } else {
                    $indexes[] = end($indexes) + 1;
                    $salaries[] = $basicSalary;
                    $months[] = $monthName;
                }
                $processedMonths[] = $currentMonthKey;
                return;
            }
        }
    
        $index = array_search(Carbon::parse($payrollEmployee->payroll->period_start)->format('F'), $months);
        if ($index !== false) {
            $salaries[$index] += $basicSalary;
        } else {
            $indexes[] = end($indexes) + 1;
            $salaries[] = $basicSalary;
            $months[] = Carbon::parse($payrollEmployee->payroll->period_start)->format('F');
        }
        $processedMonths[] = $currentMonthKey;
    }
}
