<?php

namespace Suiterus\Adg\Reports;

use App\Models\User;
use App\Enums\ReportType;
use Illuminate\Support\Carbon;
use Suiterus\Adg\Models\SM\Sector;
use Suiterus\Adg\Abstracts\FileReport;
use Suiterus\Adg\Traits\EmployeeTrait;
use Suiterus\Adg\Traits\EmployerTrait;
use Suiterus\Adg\Models\Reports\Reports;
use Suiterus\Adg\Services\ReportService;
use Suiterus\Adg\Exceptions\Reports\ReportException;
use App\Enums\Payroll\EmployeePayrollVoidStatus;
use App\Enums\Payroll\PayrollStatus;
use Suiterus\Adg\Models\Payroll\Payroll;
use Suiterus\Adg\Models\Payroll\PayrollEarningDeduction;
use alhimik1986\PhpExcelTemplator\params\ExcelParam;
use alhimik1986\PhpExcelTemplator\setters\CellSetterArrayValueSpecial;
use alhimik1986\PhpExcelTemplator\setters\CellSetterStringValue;
use alhimik1986\PhpExcelTemplator\PhpExcelTemplator;

class RATAReport extends FileReport
{
    use EmployeeTrait;
    use EmployerTrait;

    public function generate($reportId)
    {
        $report = Reports::find($reportId);

        $users = User::whereNotIn('id', [1, 2])->whereHas('employee_payroll_record.earning_deduction', function($query) {
            $query->where('name', 'Representation and Transportation Allowance(RATA)')->where('amount', '!=', 0.00);
        })->with('employeeExtraField', 'salary.publicSalary.salaryGrade', 'salary.privateSalary')
        ->without('salary.user')->get();
        $report = Reports::whereId($reportId)->with('creator')->first();

        $userCount = count($users);

        if ($userCount <= 0) {
            throw new ReportException('No employees found');
        }

        $grandTotalRATA = 0;

        $reportService = new ReportService;
        $this->initializeEmployerInfo($report->creator);
        $employerId = $this->employers_id . ' ';
        $employerName = $this->employers_name;

        $payrollPeriod = Payroll::where('status', PayrollStatus::DISBURSED)
            ->whereDate('period_start', '>=', $report->start_date)
            ->whereDate('period_end', '<=', $report->end_date)->first();

        if (!$payrollPeriod) {
            throw new ReportException('This payroll period has not yet generated');
        }

        

        foreach ($users as $key => $user) {
            $allowance = $this->getRATA($user->id, $report->start_date, $report->end_date);
            $this->initializeEmployeeInfo($user);
            $employeeNumber[] = ' ' . $this->employee_id . ' ';
            $index[] = $key + 1;
            $division[] = strtoupper($this->getEmployeeMetaInfo()->division->name);
            $fullName[] = strtoupper($this->last_name.', '.$this->first_name.' '.$this->suffix.' '.$this->middle_name);
            $rataAmount[] = number_format($allowance, 2);
            $grandTotalRATA += $allowance;
        }

        define('SPECIAL_ARRAY_TYPE', CellSetterArrayValueSpecial::class);
        //use special setter for array values to insert the whole row, and not insert cell with shifting down.
        return $this->exportToCSV([
                '{name_of_agency}' => $employerName,
                '{month}' => Carbon::parse($report->end_date)->format('F'),
                '{year}' => Carbon::parse($report->end_date)->format('Y'),
                '[index]' => new ExcelParam(SPECIAL_ARRAY_TYPE, $index),
                '[employee_no]' => new ExcelParam(SPECIAL_ARRAY_TYPE, $employeeNumber),
                '[division]' => new ExcelParam(SPECIAL_ARRAY_TYPE, $division),
                '[employee_name]' => new ExcelParam(SPECIAL_ARRAY_TYPE, $fullName),
                '[rata]' => new ExcelParam(SPECIAL_ARRAY_TYPE, $rataAmount),
                '{gt_rata}' =>  new ExcelParam(CellSetterStringValue::class, $grandTotalRATA)
            ],  Carbon::now()->format('M-Y') . '-' . 'Monthly-RATA-Report(' . Carbon::parse($report->end_date)->format('F-Y'). ')-' . $reportId . '.xlsx', $reportService->getFileTemplate(ReportType::RATA));
    }

    private function getRATA($user_id, $start_date, $end_date)
    {
        $amount = 0;

        $rata = PayrollEarningDeduction::whereHas('payroll_employee', function ($query) use ($user_id) {
            $query->where('user_id', $user_id)->where('void_status', EmployeePayrollVoidStatus::NOT_VOIDED);
        })->whereHas('payroll_employee.payroll', function ($query) use ($start_date, $end_date) {
            $query->where('status', PayrollStatus::DISBURSED)
                ->whereDate('period_start', '>=', $start_date)
                ->whereDate('period_end', '<=', $end_date);
        })->where('name', 'Representation and Transportation Allowance(RATA)')->first();

        if ($rata) {
            $amount = (float) $rata->amount;
        }

        return $amount;
    }
}
