<?php

namespace Suiterus\Adg\Reports;

use alhimik1986\PhpExcelTemplator\PhpExcelTemplator;
use alhimik1986\PhpExcelTemplator\params\CallbackParam;
use alhimik1986\PhpExcelTemplator\params\ExcelParam;
use alhimik1986\PhpExcelTemplator\setters\CellSetterArray2DValue;
use alhimik1986\PhpExcelTemplator\setters\CellSetterArrayValueSpecial;
use alhimik1986\PhpExcelTemplator\setters\CellSetterStringValue;
use Carbon\Carbon;
use App\Enums\Status;
use App\Enums\Payroll\PayrollStatus;
use App\Models\User;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use Suiterus\Adg\Abstracts\FileReport;
use Suiterus\Adg\Traits\EmployeeTrait;
use Suiterus\Adg\Traits\EmployerTrait;
use Suiterus\Adg\Models\Reports\Reports;
use Suiterus\Adg\Services\ReportService;
use Suiterus\Adg\Controllers\Payroll\Generator\PayrollGenerator;
use Suiterus\Adg\Exceptions\Reports\ReportException;
use Suiterus\Adg\Models\Payroll\Payroll;
use Suiterus\Adg\Models\Payroll\PayrollEmployee;
use Suiterus\Adg\Models\SM\Loan;

class OtherIncomeDeductionReport extends FileReport
{
    use EmployeeTrait;
    use EmployerTrait;

    public function generate($reportId)
    {
        $report = Reports::whereId($reportId)->with('creator')->first();
        $all_users = User::whereNotIn('id', [1, 2])->get();

        $payrollPeriod = Payroll::where('status', PayrollStatus::DISBURSED)
            ->whereDate('period_start', '>=', $report->start_date)
            ->whereDate('period_end', '<=', $report->end_date)->with('payroll_employee')->first();

        if (!$payrollPeriod) {
            throw new ReportException('This payroll period has not yet generated and disbursed');
        }

        $path = (new ReportService)->getFileTemplate($report->report_type);
        $this->initializeEmployerInfo($report->creator);
        $employer_name = $this->employers_name;
        $generator = new PayrollGenerator();
        $grand_total_income = 0;
        $grand_total_deduction = 0;
        foreach ($all_users as $key => $user) {
            $employee_payroll = $this->getPayrollEmployee($user->id, $report->start_date, $report->end_date);

            if (!$employee_payroll) {
                continue;
            }

            $this->initializeEmployeeInfo($user);

            $deductions = $generator->fetch_deductions($employee_payroll);
            $earnings = $generator->fetch_allowances($employee_payroll);
            $other_income = array_values($this->filterEarnings($earnings));
            $other_deduction = array_values(array_merge($this->filterDeductions($deductions), $this->getLoans($deductions['loans'])));
            $indexes[] = $key + 1;

            $employees_fullname[] = $this->getFullName();
            $employee_id[] = $this->employee_id;
            $positions[] = $user->employeeMetaInfo->position->title;

            $income_arr[] = $other_income;
            $total_incomes[] = number_format(array_sum($this->transformArrayToNumeric($other_income)), 2);
            $grand_total_income += array_sum($this->transformArrayToNumeric($other_income));

            $deduction_arr[] = $other_deduction;
            $total_deduction[] = number_format(array_sum($this->transformArrayToNumeric($other_deduction)), 2);
            $grand_total_deduction += array_sum($this->transformArrayToNumeric($other_deduction));
        }


        $deductions = $generator->fetch_deductions($payrollPeriod->payroll_employee[0]);
        $earnings = $generator->fetch_allowances($payrollPeriod->payroll_employee[0]);
        $column1 = array_keys($this->filterEarnings($earnings));
        $column2 = array_keys(array_merge($this->filterDeductions($deductions), $this->getLoans($deductions['loans'])));
        $income_columns[] = $column1;
        $deduction_columns[] = $column2;


        $income_arr[] = $this->setDummyArray($column1);
        $deduction_arr[] = $this->setDummyArray($column2);
        $income_arr[] = $this->setGrandTotalToArray($column1);
        $deduction_arr[] = $this->setGrandTotalToArray($column2);
        
       
        $total_incomes[] = '';
        $total_deduction[] = '';

        $total_incomes[] = number_format($grand_total_income, 2);
        $total_deduction[] = number_format($grand_total_deduction, 2);

        $params = [
            '{employer_name}' => new ExcelParam(CellSetterStringValue::class, $employer_name, function (CallbackParam $param) {
                $sheet = $param->sheet;
                $coordinates = $param->coordinate;
                $sheet->getStyle($coordinates)->getFont()->setSize(14.0);
            }),
            '{month}' => strtoupper(Carbon::parse($report->end_date)->format('F')),
            '{year}' => strtoupper(Carbon::parse($report->end_date)->format('Y')),
            '[index]' => new ExcelParam(CellSetterArrayValueSpecial::class, $indexes),
            '[name]' => new ExcelParam(CellSetterArrayValueSpecial::class, $employees_fullname),
            '[employee_id]' => new ExcelParam(CellSetterArrayValueSpecial::class, $employee_id),
            '[position]' => new ExcelParam(CellSetterArrayValueSpecial::class, $positions),

            '[[incomes_column]]' => new ExcelParam(CellSetterArray2DValue::class, $income_columns),
            '[[incomes]]' => new ExcelParam(CellSetterArray2DValue::class, $income_arr, function (CallbackParam $param) {
                $sheet = $param->sheet;
                $cell_coordinates = $param->coordinate;
                $sheet->getStyle($cell_coordinates)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                $sheet->getStyle($cell_coordinates)->getNumberFormat()->setFormatCode('#,##0.00');
            }),
            '[total_incomes]' => new ExcelParam(CellSetterArrayValueSpecial::class, $total_incomes, function (CallbackParam $param) {
                $sheet = $param->sheet;
                $cell_coordinates = $param->coordinate;
                $sheet->getStyle($cell_coordinates)->getNumberFormat()->setFormatCode('#,##0.00');
                $sheet->getStyle($cell_coordinates)->getFont()->setBold(true);
                $sheet->getStyle($cell_coordinates)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
            }),

            '[[deductions_column]]' => new ExcelParam(CellSetterArray2DValue::class, $deduction_columns),
            '[[deductions]]' => new ExcelParam(CellSetterArray2DValue::class, $deduction_arr, function (CallbackParam $param) {
                $sheet = $param->sheet;
                $cell_coordinates = $param->coordinate;
                $sheet->getStyle($cell_coordinates)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
                $sheet->getStyle($cell_coordinates)->getNumberFormat()->setFormatCode('#,##0.00');
            }),
            '[total_deductions]' => new ExcelParam(CellSetterArrayValueSpecial::class, $total_deduction, function (CallbackParam $param) {
                $sheet = $param->sheet;
                $cell_coordinates = $param->coordinate;
                $sheet->getStyle($cell_coordinates)->getNumberFormat()->setFormatCode('#,##0.00');
                $sheet->getStyle($cell_coordinates)->getFont()->setBold(true);
                $sheet->getStyle($cell_coordinates)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
            }),
        ];


        // Calling the 'exportToCSV' method and passing an associative array of data using 1D array
        // that will be used to populate placeholders in the CSV file per Column.
        return $this->exportToCSV($params, Carbon::now()->format('F-Y') . '-Other-Income-Deduction-Report-' . $report->id . '.xlsx', $path);
    }

    private function setGrandTotalToArray($array)
    {
        $dummy = $this->setDummyArray($array);
        $last_index = count($dummy) - 1;
        $dummy[$last_index] = 'GRAND TOTAL';
        
        return $dummy;
    }
    private function setDummyArray($array)
    {
        $dummy_array = [];
        foreach($array as $array)
        {
            $dummy_array[] = '';
        }
        return $dummy_array;
    }
    private function getPayrollEmployee($user_id, $start_date, $end_date) 
    {
        return PayrollEmployee::where('user_id', $user_id)
        ->whereHas('payroll', function ($query) use ($start_date, $end_date) {
            $query->where('status', PayrollStatus::DISBURSED)
            ->whereDate('period_start', '>=', $start_date)
            ->whereDate('period_end', '<=', $end_date);
        })->first();
    }

    private function transformArrayToNumeric($array)
    {
        $numeric = array_map(function ($value) {
            return (float) str_replace(',', '', $value);
        }, $array);

        return $numeric;
    }
    private function filterEarnings($earnings)
    {

        $data = $this->setEarnings($earnings);

        if (isset($data['Basic Pay'])) {
            unset($data['Basic Pay']);
        }
        return $data;
    }

    private function setEarnings($earnings)
    {
        foreach ($earnings as $earn) {
            if (isset($earn['title'])) {
                $allowances[] = [
                    $earn['title'] => $earn['amount']
                ];
            }
        }
        return call_user_func_array('array_merge', $allowances);
    }

    private function getLoans($user_loans)
    {
        $active_loans = Loan::where('status', Status::ACTIVE)->get();
        $data = [];
        $user_loans = collect($user_loans);
        foreach ($active_loans as $loan) {
            $result = $user_loans->where('name', $loan->name)->first();
            if ($result) {
                $data[$loan['name']] = $result->amount;
            } else {
                $data[$loan['name']] = number_format(0, 2);
            }
        }
        return $data;
    }

    private function filterDeductions($deductions)
    {
        if (isset($deductions['loans'])) {
            unset($deductions['loans']);
        }

        if (isset($deductions['basic_pay'])) {
            unset($deductions['basic_pay']);
        }

        $transformedArray = array_combine(
            array_map(function ($key) {
                return ucfirst(str_replace('_', ' ', $key));
            }, array_keys($deductions)),
            $deductions
        );

        return $transformedArray;
    }
}
